#!/usr/bin/perl

=head1 NAME

    get-release-events.pl -- собрать релизные события с определённого момента

=head1 SYNOPSIS

    get-release-events.pl   # с понедельника
    get-release-events.pl --from 2015-06-07

=head1 DESCRIPTION

    Выводит список из строк вида

        <ключ тикета> <событие> <дата>

    Например:

        DIRECT-42593 created 2015-06-08
        DIRECT-42593 tested 2015-06-10
        DIRECT-42593 deployed 2015-06-10
        DIRECT-42683 created 2015-06-10
        DIRECT-42683 tested 2015-06-10

    Какие бывают события:

        created -- создание релизного тикета. Правильнее считать выкладку на тестовую среду, но в штатном режиме временной интервал между этими событиями небольшой, а создание тикета учитывать проще.
        tested -- переход в статус 'Ждём подтверждения'. Если вдруг релиз после перехода в этот статус переоткрывался, и переход в этот статус выполнялся ещё раз, будет учитываться только последний переход перед выкладкой.
        deployed -- выкладка релиза в продакшен. Учитывается только первая выкладка, но не последующие хотфиксы.

=head1 TODO

    * доделать для остальных проектов
    * (если будет необходимость) задание временных интервалов (надо придумать ограничивающий запрос)
    
=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use DateTime;
use File::Slurp;
use Getopt::Long;
use JSON;
use LWP::UserAgent;

use ProjectSpecific;
use Startrek::Client::Easy;

run() unless caller();

sub run {
    my $opt = get_options();
    my $weekday_today = DateTime->now()->wday();
    my $from = $opt->{from} || DateTime->now()->subtract(days => ($weekday_today - 1))->strftime("%Y-%m-%d");   # понедельник
    my $today = DateTime->now()->strftime("%Y-%m-%d");

    my $st = Startrek::Client::Easy->new();
    my $issues = $st->get(query => qq/Queue: DIRECT and Components: "Releases: Direct" and ((Status: !Closed and Updated: >=$from) or Status changed: to Closed after $from)/);
    my $ua = LWP::UserAgent->new();

    my %data;

    my $update_log = from_json $ua->get('http://direct-dev.yandex-team.ru/versionica/propertylog?&host=ppcback01f&property=yandex-direct%24&as=json')->decoded_content();

    my %update_time;
    for my $entry (@$update_log) {
        ($update_time{ $entry->{value} } = $entry->{logtime}) =~ s/^([0-9]{4}-[0-9]{2}-[0-9]{2}).*/$1/;
    }
    for my $issue (@$issues) {
        my $key = $issue->{key};
        my $current_version = ProjectSpecific::parse_release_summary($issue->{summary})->{version};

        # TODO для больших changelog'ов сделать подгрузку следующей страницы (использовать заголовок Link: с rel="next" из HTTP-ответа Стартрека)
        my $changelog = $st->request(GET => "/issues/$key/changelog?perPage=200");

        for my $entry (@$changelog) {
            next unless $entry->{type} =~ /^Issue(Created|Updated|Workflow)$/;
            my $type;
            my $time;
            if ($entry->{type} eq 'IssueCreated') {
                $type = 'created';
                $time = $entry->{updatedAt};
            } elsif (grep { $_->{field}->{id} eq 'status' && $_->{to}->{key} eq 'needAcceptance' } @{ $entry->{fields} }) {
                # если есть несколько переходов в needAcceptance (чего обычно не бывает, но может случиться по ошибке), то нам нужен последний, поэтому можем перезаписывать $data->{tested}
                # если вдруг кто-то переводил релиз в needAcceptance второй раз уже после выкладки, то это не засчитается, так как до этого момента соберём все нужные ключи в $data и выйдем из цикла
                $type = 'tested';
                $time = $entry->{updatedAt};
            } elsif (my ($old_summary) = map { $_->{from} } grep { $_->{field}->{id} eq 'summary' } @{ $entry->{fields} }) {
                # здесь наоборот нужна только первая выкладка (собственно, релиз)
                # так как в штатном режиме ключ deployed появится в $data после created и tested, то до последующих выкладок цикл просто не дойдет.
                my $old_version = ProjectSpecific::parse_release_summary($old_summary)->{version};
                if ($update_time{ $old_version }) {
                    $type = 'deployed';
                    $time = $update_time{ $old_version };
                }
            }
            if ($type) {
                $time =~ s/^([0-9]{4}-[0-9]{2}-[0-9]{2}).*/$1/;
                $data{ $key }->{$type} = $time;
            }
            last if $data{ $key } && (scalar values %{ $data{ $key } } == 3);
        }
        if ($update_time{ $current_version } && !$data{ $key }->{deployed}) {    # релиз выложился без добавления нового кода, и с тех пор версия не менялась
            $data{ $key }->{deployed} = $update_time{ $current_version };
        }
    }
    for my $key (sort(map { $_->{key} } @$issues)) {
        for my $type (qw/created tested deployed/) {
            my $time = $data{ $key }->{ $type };
            next unless $time && $time ge $from;
            print "$key $type $data{ $key }->{ $type }\n";
        }
    }
    exit 0;
}

sub get_options {
    my %O;
    GetOptions(
        'from=s' => \$O{from},
        'h|help' => sub { system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8"); exit(0); },
    ) || die "can't parse options";
    return \%O;
}
