#!/usr/bin/perl

# $Id$


=head1 NAME
    
    release-action.pl

=head1 SYNOPSIS


=head1 DESCRIPTION

    Опции:
        -h, --help
            вывести справку

        -t, --test
            использовать тестовую Джиру
            флаг, т.е. можно писать отрицание: --no-test

        справка по отдельным действиям: 
            help <action>
            release-action.pl --action help <action>


=cut


# $Id$

use strict;

#use lib '/home/lena-san/c/direct-utils/project_specific_pm/trunk/lib';
#use lib '/home/lena-san/c/direct-utils/release_shell_c/trunk';


use Cwd;
use POSIX qw/strftime/;
use Data::Dumper;
use JSON;
use YAML::Syck;
#use Getopt::Long qw(GetOptionsFromArray);
use List::Util qw/max/;
use Getopt::Long;
use File::Slurp;
use LWP::UserAgent;
use IO::Prompt;
use XML::LibXML;
use Yandex::Yamb;
use Yandex::Shell;

use FindBin qw/$Bin/;

use lib $Bin;


use lib $ENV{CONGERIA_PATH}; 
use ProjectSpecific qw/
    version2rev
    svn_url
    tabula_location
    tabula_testupdate_url
/;

use Release::Check;
use Release::Issue;

use utf8;
use open ':std' => ':utf8';

#................................................

# префикс для всех имен переменных окружения, относящихся к релизу. 
# бессмысленная, но симпатичная строчка, чтобы было легко отгрепать все переменные релиза.
# Congeria -- род двустворчатых моллюсков (т.е. имеет shell ;))
our $prefix_for_shell_vars = 'CONGERIA_'; 

our %action_sub = (
    vars_and_aliases =>{
        code => \&vars_and_aliases,
        params_obligatory => [qw/
        /],
        params_optional => [qw/
            use_test_tracker
            project
            use_svnroot
        /],
    },
    packages => {
        code => \&packages,
        notify => 1,
        params_obligatory => [],
        params_optional => [],
    },
    tupdate => {
        code => \&tupdate,
        notify => 1,
        params_obligatory => [],
        params_optional => [],
    },
    tupdate2 => {
        code => \&tupdate2,
        notify => 1,
        params_obligatory => [],
        params_optional => [],
    },
    pre_tupdate => {
        code => \&pre_tupdate,
        notify => 1,
        params_obligatory => [],
        params_optional => [],
    },
    set_release_mode => {
        code => \&set_release_mode,
        params_obligatory => [ ],
        # Опция --advanced ничего не делает (режим, задаваемый ей, сейчас включен по умолчанию). Оставлена для обратной совместимости с существующими пользовательскими алиасами.
        params_optional => [qw/
            issue
            advanced
            testserver
        /],
    },
    quit_release_mode => {
        aliases => [qw/ quit_release /],
        code => \&quit_release_mode,
        params_obligatory => [],
        params_optional => [qw/
            force
        /],
    },
    title => {
        code => \&suggest_release_title,
        notify => 0,
        params_obligatory => [],
        params_optional => [],
    },
    url_testupdate => {
        code => \&url_testupdate,
        notify => 0,
        params_obligatory => [],
        params_optional => [],
    },
    hotfix => {
        code => \&hotfix,
        notify => 1,
        params_obligatory => [],
        params_optional => [qw/
            ts
            jira
        /],
    },
    tracker => {
        code => \&tracker,
        #notify => 1,
        params_obligatory => [],
        params_optional => [qw/
            deploy_notes
            release_issue_key
            inspect
            use_test_tracker
            new_version
            prev_version
            name
            deploy_notes
            rename
            do
            comment
            downgrade
            repeat_tests
            migration_done
            migration_undo
            force
            cc
        /],
    },
    'accept' => {
        code => \&accept,
        params_obligatory => [],
        params_optional => [],
    },
    'conductor' => {
        code => \&conductor,
        params_obligatory => [],
        params_optional => [],
    },
    help => {
        code => \&help,
        params_obligatory => [],
        params_optional => [],
    },
    up => {
        code => \&up,
        params_obligatory => [],
        params_optional => [],
    }, 
    slide => {
        code => \&slide,
        notify => 1,
        params_obligatory => [],
        params_optional => [qw/
            no-tupdate
        /],
    }, # бывш. shift
    state => {
        code => \&state,
        params_obligatory => [],
        params_optional => [qw/
            issue
            release_type
            prev_issue
            packages
            testserver
            verbose
            use_svnroot
            review_issue
        /],
    }, 
    check => {
        code => \&check,
        params_obligatory => [],
        params_optional => [],
    },
    ready_version => {
        code => \&ready_version,
        params_obligatory => [qw/ 
            new_version 
        /],
        params_optional => [],
    },
    check_packages => {
        code => \&check_packages,
        params_obligatory => [],
        params_optional => [],
    },
    flag => {
        code => \&flag,
        params_obligatory => [],
        params_optional => [qw/
            drop
            set
        /],
    },
    dmove_testing => {
        code => \&dmove_testing,
        params_obligatory => [],
        params_optional => [],
    },
    dmove_stable => {
        code => \&dmove_stable,
        params_obligatory => [],
        params_optional => [],
    },
    dmove => {
        code => \&dmove,
        params_obligatory => [],
        params_optional => [],
    }, 
    mass_dmove => {
        code => \&mass_dmove,
        params_obligatory => [],
        params_optional => [],
    }, 
    to_dmove => {
        code => \&to_dmove,
        params_obligatory => [],
        params_optional => [],
    }, 
    review => {
        code => \&cmd_review,
        params_obligatory => [],
        params_optional => [],
    }, 
);


#................................................
# TODO с новым GetOpt::Long run(\@ARGV)
run() unless caller();


sub run
{
    my %OPT = parse_options();

    $ProjectSpecific::PROJECT = $OPT{project} if $OPT{project};
    eval "use Release::${ProjectSpecific::PROJECT}::TestUpdate";
    eval "use Release::${ProjectSpecific::PROJECT}";
    die "can't load TestUpdate module for project ${ProjectSpecific::PROJECT}:\n$@" if $@;

    # раскрываем алиасы; новые ключи в action_sub добавляем, чтобы vars_and_aliases их обработала
    for my $act (keys %action_sub){
        for my $syn (@{$action_sub{$act}->{aliases}||[]}){
            $action_sub{$syn} = $action_sub{$act};
        }
    }
    die "Operation '$OPT{action}' not permitted or not available yet.\n" if !exists $action_sub{$OPT{action}};

    my $opt = options_for_action($OPT{action}, \%OPT);

    my $action = $action_sub{$OPT{action}};

    my $res;
    eval{ 
        $res = $action->{code}->($opt, \@ARGV) || {};
        $res->{status} ||= 0;
    };
    if ($@){
        print STDERR "died: $@\n";
        $res->{status} = 1;
        $res->{die_message} = $@;
    }
    print "\nresult of $OPT{action}:\n".Dumper($res) if var_value('verbose');
    if( $action->{notify} || $OPT{notify} ){
        my $message = join "", 
            "Релиз ".var_value('RELEASE_ISSUE_KEY'), 
            " (".var_value('RELEASE_TYPE').")",
            "\n",
            "$OPT{action}: " . ($res->{status}?"произошла ошибка":"успешно");
        notify( $message );
    }

    if (var_value('ACTION_OUTPUT_FILE')){
        my $content = join "", 
            (map { "export ".var_name($_)."=" . yash_quote($res->{export}->{$_} || '') . "\n" } sort keys %{$res->{export}}), 
            (map { "unset  ".var_name($_)."\n" } sort keys %{$res->{unset}}), 
            (map {"$_\n"} @{$res->{postactions}});

        write_file(var_value('ACTION_OUTPUT_FILE'), {atomic => 1, binmode => ':utf8'}, $content);
    }

    exit $res->{status};
}

#................................................
sub var_name
{
    die unless $_[0];
    return "${prefix_for_shell_vars}$_[0]";
}


sub var_value
{
    die unless $_[0];
    return $ENV{var_name($_[0])} || '';   
}

#................................................
sub init_tracker {
    my $instance = var_value('TRACKER_INSTANCE');
    if ($instance eq 'test') {
        $ProjectSpecific::STARTREK_INSTANCE = 'test';
    } elsif ($instance eq 'prod') {
        $ProjectSpecific::STARTREK_INSTANCE = 'prod';
    } else {
        die "unknown tracker instance '$instance'";
    }
}


#................................................
sub options_for_action
{
    my ($action, $all_opt) = @_;

    my $opt = { map {defined $all_opt->{$_} ? ( $_ => $all_opt->{$_} ) : () } @{$action_sub{$action}->{params_obligatory}||[]}, @{$action_sub{$action}->{params_optional}||[]} };

    for my $f ( @{$action_sub{$action}->{params_obligatory}||[]} ){
        die "param '$f' missed (obligatory for action '$action')\n" unless exists $opt->{$f}; 
    }

    return $opt;
}


sub merge_results
{
    my ($target, $source, %O) = @_;
    $source ||= {};

    if ( $source->{export} ){
        for (keys %{$source->{export}}){
            $target->{export}->{$_} = $source->{export}->{$_};
            $ENV{var_name($_)} = $source->{export}->{$_};
        }
    }

    if ( $source->{unset} ){
        for (keys %{$source->{unset}}){
            $target->{unset}->{$_} = $source->{unset}->{$_};
            $ENV{var_name($_)} = '';
        }
    }

    die if $source->{status} && !$O{dont_die};

    if ( $source->{postactions} ){
        push @{$target->{postactions}}, @{$source->{postactions}};
    }

    return;
}


sub notify
{
    my ($message) = @_;

    print "$message\n";
    
    if( var_value('dont_notify') ){
        print "skippping notification...\n";
        return;
    }

    my $login = [getpwuid($<)]->[0];
    send_yamb_message(to => $login, message => $message) or die "notify failed\n";

    return;
}


#................................................
sub vars_and_aliases
{
    my ($opts, $argv) = @_;
    my $res;

    # decorate_prompt
    if ( ! var_value('OLD_PS1') ){
        $res->{export}->{OLD_PS1} = $ENV{PS1};
    }
    my $old_ps1 = var_value('OLD_PS1') || $ENV{PS1};
    push @{$res->{postactions}}, q/export PS1='
===Release $/.var_name('RELEASE_ISSUE_KEY').' ($'.var_name('PKG_VERSION').') $'.var_name('RELEASE_TYPE').'==
' . $old_ps1 . "'";

    # alias_actions
    for my $action (grep {!/(vars_and_aliases)/} keys %action_sub ){ 
        push @{$res->{postactions}}, "alias $action='run ".var_value('PATH')."/release-action.pl --action $action'";
    }

    $ProjectSpecific::PROJECT = $opts->{project} if $opts->{project};
    $ProjectSpecific::STARTREK_INSTANCE = $opts->{use_test_tracker} ? 'test' : 'prod';

    # Используем $ProjectSpecific::STARTREK_INSTANCE также и для определения инстанса Джиры.
    push @{$res->{postactions}}, "export " . var_name('TRACKER_INSTANCE') . "=$ProjectSpecific::STARTREK_INSTANCE";
    push @{$res->{postactions}}, q/alias jr='tracker'/; # алиас оставляем jr, т. к. команда tr уже есть

    # alias_cd 
    my $cd_mes = 'Внимание!\nВо время релиза совершенно не рекомендуется покидать каталог packages.\nЕсли действительно очень надо, и ты знаешь, что делаешь -- воспользуйся chdir';
    push @{$res->{postactions}}, 
        qq/alias   b='echo "$cd_mes"'/, 
        qq/alias cdb='echo "$cd_mes"'/, 
        qq/alias  cd='echo "$cd_mes"'/;
    if ( var_value('SHELL_TO_RETURN_TO') eq 'bash' ){
        # потому что у bash проблемы с chdir
        push @{$res->{postactions}}, q/alias chdir='my_cd'/;
    }

    # remember_shell
    if ( var_value('SHELL_TO_RETURN_TO') eq 'zsh' ){
        push @{$res->{postactions}}, 'setopt PROMPT_SUBST';
    } elsif ( var_value('SHELL_TO_RETURN_TO') eq 'bash' ){
        push @{$res->{postactions}}, 'shopt promptvars';
    } else {
        print "не знаю, что за шелл ты используешь. Корректная работа ни разу не гарантируется. Может, попробуешь zsh? На крайний случай -- bash...";
    }

    # no-svnroot ready
    if ( ! defined $opts->{use_svnroot} ){
        $res->{export}->{USE_SVNROOT} = 0;
    } else {
        $res->{export}->{USE_SVNROOT} = $opts->{use_svnroot} ? 1 : 0;
    }

    $res->{export}->{RELEASE_TYPE} = $res->{export}->{RELEASE_TYPE} || var_value('CONGERIA_RELEASE_TYPE') || 'RC';

    return $res; 
}


=head1 title

    Список "ожидающих релиза" задач из Планирования релизов 
    (запланированные задачи, по которым уже были коммиты, но еще не выехавшие в продакшен). 

=cut
sub suggest_release_title
{
    my $res = {};

    my $url = tabula_location('release_title');
    my $title = qx!wget -qO - '$url'!;
    print "\n### Сейчас ожидают релиза следующие заметные задачи:\n\n$title\n";

    return $res;
}


=head1 packages

    сборка пакетов

=cut
# TODO --no-repos?
sub packages
{
    my ($opt, $argv) = @_;
    my $res = {};

    check_path_for_release();

    my $make_cmd = "make all"; 

    `echo |gpg --sign >/dev/null`;
    if ( !var_value('USE_SVNROOT') ){
        `svn-reset ..`;
    }

    # perl -le '$text = `make all |tee /dev/stderr`; $text =~ /source version([^\n]*)/sm; print "version: $1"'

    my @lines;
    open my $make, qq/(export LC_ALL="en_US.UTF-8";export LANGUAGE="en"; $make_cmd 2>&1) |/ or die;
    while (<$make>) { 
        push @lines, $_;
        print; 
    }
    close $make; # to get $?
    $res->{status} = $? >> 8;

    my $text = join "", @lines;
    $text =~ /source version *([^\n]*)/sm; 
    my $new_version = $1;
    $res->{export}->{PKG_VERSION} = $new_version;

    merge_results $res, ready_version({new_version => $new_version}), dont_die => 1;
    
    return $res;
}


=head1 set_release_mode

    включение релизного режима
    в норме вручную не делается

=cut
sub set_release_mode
{
    my ($opt, $argv) = @_;
    my $res = {};

    check_path_for_release();
    
    init_tracker();
    my $issue = $opt->{issue} || $argv->[0];
    if ( $issue ){
        my $r = Release::Issue::issue_info(key => $issue);
        print Release::Issue::issue_info_to_text($r);
        merge_results $res, _update_release_state_according_to_tracker( $issue );
    } else {
        my $lr = Release::Issue::issue_info(last_release => 1);
        print Release::Issue::issue_info_to_text($lr);
        $res->{export}->{PREV_RELEASED_VERSION} = $lr->{version} or die "Can't get tracker info for the last release\n";
        $res->{export}->{PREV_RELEASE_ISSUE_KEY} = $lr->{key};
        $res->{export}->{RELEASE_ISSUE_KEY} = 'create';
        $res->{export}->{RELEASE_TYPE} = 'RC';
        $res->{export}->{TESTSERVER} = $opt->{testserver} || default_testserver();
        $res->{export}->{REVIEW_ISSUE_KEY} = '';

        switch_release_directories(svn_url => svn_url('trunk'));
    }

    print join "", map "$_\n", 
        "   === Релизное окружение успешно инициализировано ===", 
        "   === По команде help доступна краткая справка ===",
        "   === Пользуйтесь релиз-шеллом Congeria! ===";


    return $res;
}


=head1 quit_release_mode

    выход из релизного режима

=cut
sub quit_release_mode
{
    my ($opt, $argv) = @_;
    my $res;

    #eval{switch_release_directories(svn_url => svn_url('trunk'))};
    switch_release_directories(svn_url => svn_url('trunk')) unless $opt->{force};
    $res->{export}->{RELEASE_TYPE} = '';

    push @{$res->{postactions}}, "export PS1='".var_value('OLD_PS1')."'";
    push @{$res->{postactions}}, "unalias cd";
    push @{$res->{postactions}}, "unalias jr";
    # TODO unalias actions
    $res->{unset}->{$_} = 1 for qw/PKG_VERSION RELEASE_ISSUE_KEY PREV_RELEASE_ISSUE_KEY REVIEW_ISSUE_KEY OLD_PS1 
        directory_for_release verbose SHELL_TO_RETURN_TO/;

    print "\n   === Работа с релизом закончена ===\n";
    push @{$res->{postactions}}, "exec ".var_value('SHELL_TO_RETURN_TO'); 

    return $res;
}


# проверяем, что версия в трекере и в релизном шелле одинаковая
sub check_current_version
{
    my $tracker_version = Release::Issue::version_in_tracker(var_value('RELEASE_ISSUE_KEY'));
    my $local_version = var_value('PKG_VERSION');
    die "Локальная версия '$local_version' не совпадает с версией в трекере '$tracker_version'\nЧто делать:\n  1. разобраться, что случилось\n  2. если все нормально -- сделать up\n" unless $tracker_version eq $local_version;
}


# проверяем, что каталоги -- подходяшие для сборки релиза
sub check_path_for_release
{
    my $xml = `svn info --xml`;
    my $doc = XML::LibXML->new()->parse_string( $xml )->documentElement();
    my $svn_url = ( $doc->findnodes('/info/entry/url') )[0]->textContent();
    my $svn_root = ( $doc->findnodes('/info/entry/repository/root') )[0]->textContent();
    my $correct_root = svn_url('root');

    if ( $svn_root !~ m!^\Q$correct_root\E$!m ){
        die "check_path_for_release: Repository Root should be $correct_root, but it's $svn_root\n";
    }

    if ( $svn_url !~ m/.*packages$/m ){
        die "check_path_for_release: directory should be 'packages'\n";
    }
    # no-svnroot ready
    if (var_value('USE_SVNROOT')){
        -d 'svnroot' or die "no 'svnroot' directory";
    } else {
        my $xml = `svn info --xml ..`;
        my $doc = XML::LibXML->new()->parse_string( $xml )->documentElement();
        my $svn_root = ( $doc->findnodes('/info/entry/repository/root') )[0]->textContent();
        die "'..' should be $correct_root, but $svn_root found" unless $svn_root eq $correct_root;
    }
}


sub _update_release_state_according_to_tracker
{
    my ($issue_key) = @_;
    my $res;
    
    init_tracker();
    my $issue = Release::Issue::issue_info(key => $issue_key);
    if( !$issue || $issue->{type} ne 'release' ){
        print  qq/   Что-то не так: неправильный номер релиз-тикета, неверный пароль или еще что-то.
   Проверь номер тикета и попробуй так: set_release_mode $issue_key Не поможет -- попробуй все с самого начала. Не поможет -- зови на помощь.
        /;
        die "\n";
    }

    $res->{export}->{PKG_VERSION} = $issue->{version};
    $res->{export}->{RELEASE_ISSUE_KEY} = $issue->{key};
    $res->{export}->{TESTSERVER} = $issue->{testserver};

    switch_release_directories(svn_url => $issue->{'svn_path'}. "@" . $issue->{'last_rev'});

    return $res;
}

=head1 switch_release_directories

    вручную делаться не должен

=cut
# TODO должен быть нормальный action?
sub switch_release_directories
{
    my %O = @_;
    my $res;

    check_path_for_release();
    die "switch_release_directories: missed target (svn url)\n" unless $O{svn_url};
    $O{svn_url} =~ s!/(\@|$)!!;
    (my $packages_url = $O{svn_url}) =~ s!^([^\@]*)!$1/packages!;

    # TODO проверять успешность switch'ей
    # no-svnroot ready
    if (var_value('USE_SVNROOT')){
        system("svn switch $O{svn_url} svnroot");
        system("svn switch $packages_url .");
    } else {
        system("svn switch $O{svn_url} ..");
    }

    return $res;
}


=head1 hotfix

    hotfix <rev>[-<rev>][,<rev>]...

    добавить указанные ревизии в релиз (с созданием релизного бранча)

=cut
# TODO параметры: 
#   * jira
#   * ts
#   
#   * releaser/testupdate/ready
#   * yamb-уведомление
sub hotfix
{
    my ($opt, $argv) = @_;
    my $res = {};
    check_path_for_release();
    check_current_version();

    my $base_rev = version2rev(var_value('PKG_VERSION'), rev => 'base');
    my $svn_root = svn_url('root'); 

    print "Base revision: $base_rev\nsvn: $svn_root\nCreating/updating release branch...\n";

    # no-svnroot ready
    my $svn_hotfix_wc_path = var_value('USE_SVNROOT') ? "svnroot" : "..";
    system("svn-hotfix", "--pre-interactive", "--ignore-externals", "-w", $svn_hotfix_wc_path, "--reset-working-copy", $svn_root, $base_rev, @$argv);
    if ($? >> 8){
        print STDERR qq!



        Если произошел конфликт при мерже хотфикса: 

        1. Хорошенько подумай еще раз, может быть, не надо вообще это хотфиксить. 
        2. Посмотри: не было ли "потерянных" коммитов, которые необходимы для этого хотфикса, но не были еще захотфикшены.
        3. Если все-таки будешь разрешать конфликты руками: 
           - позови разработчика, который знает этот код;
           - после мержа обязательно выложи код на какую-нибудь бету и попроси тестировщиков потестировать.



!;
        die "svn-hotfix did not succeed\n";
    }

    my $release_branch_url = "$svn_root/releases/release-$base_rev";

    print "new url: $release_branch_url.\nSwitching directories...\n"; 

    merge_results $res, switch_release_directories(svn_url => $release_branch_url);

    system("svn info $svn_hotfix_wc_path");

    print "Creating packages...";

    merge_results $res, packages();

    #merge_results $res, ready_version({new_version => var_value('PKG_VERSION')}), dont_die => 1;

    #if ( (var_value('RELEASE_TYPE')||'') eq 'RC' ){
    if ( $opt->{ts} ){
        test_update(version => var_value('PKG_VERSION'), ts_id => ['test1', 'test-sand']);
    }

    print "\nТеперь надо передвинуть пакеты и записать что полагается в трекер. Порядок:\n\n\t\tdmove\n\t\tjr\n\t\tjr -do\n\n";

    #jira-release -rt || return
    #jira-release -rt -do || return

    return $res;
}


=head1 slide

    slide [<revision>] 

    сдвинуть релиз вверх по ревизиям

=cut

sub slide
{
    my ($opt, $argv) = @_;
    my $res = {};
    check_path_for_release();
    check_current_version();

    my $rev_param = @$argv ? "-r$argv->[0]" : '';

    # no-svnroot ready
    if ( var_value('USE_SVNROOT') ){
        system("svn up $rev_param svnroot") == 0 or die;
        system("svn up $rev_param .") == 0 or die;
    } else {
        system("svn up $rev_param ..") == 0 or die;
    }
    system("svn info .");

    merge_results $res, packages();

    print "\nТеперь надо передвинуть пакеты и записать что полагается в трекер. Порядок:\n\n\t\tdmove\n\t\tjr\n\t\tjr -do\n\n";

    return $res;
}


=head1 ready_version

    ready_version <version>

    принудительно пометить версию как доступную к выкладке на ТС
    в норме не требуется, выполняется автоматически при сборке пакетов

=cut
sub ready_version
{
    my ($opt, $argv) = @_;
    my $res;

    die "ready version missed" unless $opt->{new_version};

    my $now = strftime("%Y%m%d%H%M", localtime());
    my $url = eval { tabula_location("testupdate") . "/ready?testserver=test1&version=$opt->{new_version}&time=$now" };
    if ( $@ ){
        print "can't register ready version\n";
        $res->{status} = 1;
        return $res;
    }

    my $ua = LWP::UserAgent->new;
    my $response = $ua->get($url);

    $res->{status} = $response->code == 200 ? 0 : 1;
    print "\ntestupdate response code: ".$response->code."\n";

    return $res;
}


=head1 tracker

    алиас jr
    "Записать все необходимое в трекер"

    Без ключа --do только печатает сводку запланированных действий. 
    С ключом --do -- выполняет действия в Джире

    --name, -n <str>
        дать релизу имя при создании

    --rename <str>
        переименовать уже созданный релиз

    --migration-done, --md <filename>
        отметить миграцию как примененную

    --migration-undo <filename>
        отметить миграцию как НЕ примененную

    --comment <comment>
        добавить комментарий

    --сс <login1>,<login2>,...
        добавить людей в наблюдатели

=cut
sub tracker
{
    my ($opt, $argv) = @_;
    my $res;
    init_tracker();

    my %tr_opt = (
        release_issue_key => $opt->{issue} || var_value('RELEASE_ISSUE_KEY'),
        version => $opt->{new_version} || var_value('PKG_VERSION'),
        prev_version => $opt->{prev_version} || var_value('PREV_RELEASED_VERSION'),
        print_only => !$opt->{do},
        name => $opt->{name} || '',
        tupdate_message => test_update_message_for_jira(
            testupdate_url => tabula_testupdate_url(var_value('PKG_VERSION')),
        ),
        inspect_issue => $opt->{inspect},
        deploy_notes => $opt->{deploy_notes},
        comment => $opt->{comment},
        repeat_tests => $opt->{repeat_tests},
        migration_done => $opt->{migration_done},
        migration_undo => $opt->{migration_undo},
        force => $opt->{force},
        cc => $opt->{cc},
        downgrade => $opt->{downgrade},
        testserver => var_value('TESTSERVER'),
        # no-svnroot ready
        use_svnroot => var_value('USE_SVNROOT'),
    );
    $tr_opt{rename} = $opt->{rename} if $opt->{rename};
    my $tr_res = Release::Issue::tracker_release( %tr_opt );
    $res->{export}->{RELEASE_ISSUE_KEY} = $tr_res->{created} if $tr_res->{created};

    return $res;
}


=head1 accept

    Accept релиза с предварительной проверкой консистентности. 
    Не работает. 

=cut
sub accept
{
    my ($opt, $argv) = @_;
    my $res;

    return $res;
}

=head1 conductor

    Создание тикета в Кондукторе. 
    Смотрит на release-type, чтобы создать тикет в правильной ветке (stable/testing)

=cut
sub conductor
{
    my ($opt, $argv) = @_;
    my $res;

    create_conductor_ticket();

    return $res;
}



=head1 tupdate

    Обновление тестового окружения (ТС + необходимые беты)

=cut
sub tupdate
{
    my ($opt, $argv) = @_;
    my $res; 

    my $version = $argv->[0] || var_value('PKG_VERSION') || die;
    test_update(version => $version, ts_id => ['test1', 'test-sand', 'ts1-scripts-rtc']); # TODO ts_id -- параметр

    return $res;
}


=head1 tupdate2

    Обновление ТС2 (и никаких бет)

=cut
sub tupdate2
{
    my ($opt, $argv) = @_;
    my $res; 

    my $version = $argv->[0] || var_value('PKG_VERSION') || die;
    test_update(version => $version, ts_id => ['test2', 'test-sand2']); # TODO ts_id -- параметр

    return $res;
}


=for none
pre_tupdate

    Безопасная часть обновления тестового окружения (беты, но не ТС)

=cut
sub pre_tupdate
{
    my ($opt, $argv) = @_;
    my $res; 

    my $version = $argv->[0] || var_value('PKG_VERSION') || die;
    test_update_prepare(version => $version, ts_id => ['test1', 'test-sand']);

    return $res;
}


=head1 state

    Просмотр и корректировка "состояния" релиза 

=cut
sub state
{
    my ($opt, $argv) = @_;
    my $res;

    my @state = (
        {
            display => 'release-type',
            var => 'RELEASE_TYPE',
            opt => 'release_type',
            re  => '^(RC|STABLE)$',
        },
        {
            display => 'issue',
            var => 'RELEASE_ISSUE_KEY',
            opt => 'issue',
        },
        {
            display => 'prev-issue',
            var => 'PREV_RELEASE_ISSUE_KEY',
            opt => 'prev_issue',
        },
        {
            display => 'pkg',
            var => 'PKG_VERSION',
            opt => 'packages',
        },
        {
            display => 'verbose',
            var => 'verbose',
            opt => 'verbose',
        },
        {
            display => 'testserver',
            var => 'TESTSERVER',
            opt => 'testserver',
        },
        # no-svnroot ready
        {
            display => 'use-svnroot',
            var => 'USE_SVNROOT',
            opt => 'use_svnroot',
        },
        {
            display => 'review-issue',
            var => 'REVIEW_ISSUE_KEY',
            opt => 'review_issue',
        },
    );

    if ( !keys %$opt  ){
        my $n = max map { length $_->{display} } @state;
        print join "", map { "$_->{display}: ".(" " x ($n - length $_->{display})).var_value($_->{var})."\n"} @state;
    } else {
        for my $s ( grep { exists $opt->{$_->{opt}} } @state ){ 
            my $value = $opt->{$s->{opt}} || '';
            print("incorrect value '$value' for $s->{display}, expected $s->{re}\n"), next if $s->{re} && $value !~ /$s->{re}/;
            $res->{export}->{$s->{var}} = $value;
        }
    }

    return $res;
}


=head1 check

    Проверка консистентности релизов

=cut
sub check
{
    my ($opt, $argv) = @_;
    my $res;
    check_path_for_release();

    init_tracker();
    my %releases;
    if ( var_value('RELEASE_TYPE') eq 'RC' ) {
        $releases{rc} = var_value('RELEASE_ISSUE_KEY') or die "release tracker key missed\n";
        $releases{st} = var_value('PREV_RELEASE_ISSUE_KEY') or die "previous release tracker key missed\n";
    } elsif ( var_value('RELEASE_TYPE') eq 'STABLE' ){
        $releases{st} = var_value('RELEASE_ISSUE_KEY') or die "release tracker key missed\n";
    } else {
        die "release type missed;\nstate --release-type (RC|STABLE)\n";
    }

    my $errors = release_errors(%releases);

    $res->{status} = release_errors_count($errors);

    print "\n".(release_errors_to_text($errors) || "OK\n");

    return $res;
}


=head1 up

    обновить переменные/переключить каталоги в соответствии с релиз-тикетом 
    (на случай, если с одним релизом работают двое)

=cut
sub up
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $key = var_value('RELEASE_ISSUE_KEY');
    return $res unless $key;
    merge_results $res, _update_release_state_according_to_tracker( $key );

    return $res;
}


=head1 check_packages

    проверить, что пакет с текущей версией релиза готов для установки

=cut
sub check_packages
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $version = var_value('PKG_VERSION') || die "package version is not set";
    my $package_name = ProjectSpecific::get_project_data('main_deb_package');

    local $Yandex::Shell::PRINT_COMMANDS = 1;
    yash_qx("beta-update apt-get-update 2>/dev/null");  # используем yash_qx, чтобы проглотить вывод в STDOUT
    yash_system("apt-cache policy $package_name | grep $version || exit 0");

    return $res;
}


=head1 flag

    Управление релизным мораторием (работает пока только для Директа)

    Внутри делается
    beta-update --ppcdev-all direct-release-flag

    Посмотреть состояние флага:
    flag

    Установить флаг:
    flag --set 950

    Сбросить флаг
    flag --drop

=cut
sub flag
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $options = join " ", ( $opt->{drop} ? "--drop" : () ), ( $opt->{set} ? "--set $opt->{set} " : () ); 
    
    yash_system("beta-update --ppcdev-all direct-release-flag $options");

    return $res;
}


sub _dmove
{
    my ($where) = @_;
    my $res = {};

    my @source_packages = map {s/^Source: ([\S]+).*$/$1/r} split /\n+/, yash_qx("ls */debian/control |xargs grep -h --color=never '^Source:'");
    my $version = var_value('PKG_VERSION');
    
    my $repo = ProjectSpecific::get_project_data('main_dist_repo');

    my $cmd = { 
        testing => "dmove_testing_from_file", 
        stable  => "dmove_stable_from_file", 
    }->{$where} or die "unsupported dmoving to '$where'";

    my $dmove_cmd = dmove_cmd();
    my $command = "cat <<'EOF' | $dmove_cmd $cmd:$repo,-\n"
    .join("\n", map { "$_ $version" } @source_packages)
    . "\nEOF\n";

    yash_system($command);

    return $res;
}


=head1 dmove_testing

    Переместить проектные пакеты текущей версии из unstable в testing

=cut
sub dmove_testing
{
    my ($opt, $argv) = @_;

    return _dmove("testing");
}


=head1 dmove_stable

    Переместить проектные пакеты текущей версии из testing в stable

=cut
sub dmove_stable
{
    my ($opt, $argv) = @_;

    return _dmove("stable");
}


=head1 dmove

    Переместить проектные пакеты в правильную ветку репозитория.
    Полагается на release_type: RC -- testing; STABLE -- stable

=cut
sub dmove 
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $type = var_value('RELEASE_TYPE');

    if ( $type eq "RC" ){
        _dmove("testing");
    } elsif ( $type eq "STABLE" ){
        _dmove("testing");
        _dmove("stable");
    } else {
        die "unsupported release type '$type', stop\n";
    }

    return $res;
}


=head1 mass_dmove

to_dmove |grep -v ... |mass_dmove

=cut
sub mass_dmove
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $repo = ProjectSpecific::get_project_data('main_dist_repo');
    my $dmove_cmd = dmove_cmd();

    my %to_dmove;
    my $direction = "";
    while (my $str = <>){
        chomp $str;
        next if $str =~ /^\s*$/;
        if ( $str =~ /^# *(\w+) *-> *(\w+)/ ){
            $direction = "$1_$2";
            next;
        }
        die "unsupported direction '$direction' for dmove, stop\n" unless $direction =~ /^(unstable_testing|unstable_stable|testing_stable|prestable_stable)$/;
        push @{$to_dmove{$direction}}, $str;
    }

    my @to_dmove_testing = (@{$to_dmove{unstable_testing}||[]}, @{$to_dmove{unstable_stable}||[]});
    if ( @to_dmove_testing > 0 ){
        open(my $dmove_testing, "|-", "$dmove_cmd dmove_testing_from_file:$repo,-");
        print $dmove_testing $_."\n" or die for @to_dmove_testing;
        close $dmove_testing or die;
    }

    my @to_dmove_stable = (@{$to_dmove{unstable_stable}||[]}, @{$to_dmove{testing_stable}||[]});
    if ( @to_dmove_stable > 0 ){
        open(my $dmove_stable, "|-", "$dmove_cmd dmove_stable_from_file:$repo,-");
        print $dmove_stable $_."\n" or die for @to_dmove_stable;
        close $dmove_stable or die;
    }

    my @to_dmove_prestable_stable = (@{$to_dmove{prestable_stable}||[]});
    if ( @to_dmove_prestable_stable > 0 ){
        open(my $dmove_prestable_stable, "|-", "$dmove_cmd dmove_prestable_stable_from_file:$repo,-");
        print $dmove_prestable_stable $_."\n" or die for @to_dmove_prestable_stable;
        close $dmove_prestable_stable or die;
    }

    return $res;
}


=head1 to_dmove


=cut
sub to_dmove
{
    my ($opt, $argv) = @_;
    my $res = {};

    my $etc_path = var_value('USE_SVNROOT') ? "svnroot/etc" : "../etc"; 
    my $type = var_value('RELEASE_TYPE');
    my $from;
    if ( $type eq "RC" ){
        print "# unstable -> testing\n";
        find_packages_to_dmove(from => "unstable", etc_path => $etc_path);
    } elsif ( $type eq "STABLE" ){
        print "# unstable -> stable\n";
        find_packages_to_dmove(from => "unstable", etc_path => $etc_path);
        print "# prestable -> stable\n";
        find_packages_to_dmove(from => "prestable", etc_path => $etc_path);
        print "# testing -> stable\n";
        find_packages_to_dmove(from => "testing", etc_path => $etc_path);
    } else {
        die "unsupported release type '$type', stop\n";
    }

    return $res;
}

=head1 review

    Управление ревью
    Только для Директа

    review create
    review ok DIRECT-NNN
    review almost-ok ...
    review not-ok ...

=cut
sub cmd_review
{
    my ($opt, $argv) = @_;
    my $res = {};

    die "Direct-only feature, sorry\n" unless $ProjectSpecific::PROJECT eq 'Direct';

    if ( $argv->[0] eq 'create' ){
        die "Ревью уже создано: ". var_value('REVIEW_ISSUE_KEY') if var_value('REVIEW_ISSUE_KEY');
        my $tr_res = Release::Issue::create_review(
            release => var_value('RELEASE_ISSUE_KEY'),
        );
        $res->{export}->{REVIEW_ISSUE_KEY} = $tr_res->{created} if $tr_res->{created};
        print YAML::Dump($tr_res);
    } elsif ( $argv->[0] =~ /^(ok|not-ok|almost-ok)$/ ){
        my $comment = { 
            ok => 'порядок',
            'not-ok' => 'есть вопросы',
            'almost-ok' => 'в целом порядок',
        }->{$1};
        my $review_issue = var_value('REVIEW_ISSUE_KEY');
        my $to_review = $argv->[1] or die "no issue to revie";
        die 'no review issue' unless $review_issue;
        Release::Issue::post_comment( key => $to_review, comment => "Ревью st:$review_issue -- $comment", );
    } else {
        die "unknown subcommand for review, stop\n";
    }

    return $res;
}


=head1 help

    Справка по доступным действиям

=cut
sub help 
{
    my ($opt, $argv) = @_;
    my $res;

    print "\n";
    if (@$argv){
        for my $action (@$argv){
            $action = 'tracker' if $action eq 'jr';
            system("podselect -section '$action/!.+' $0 | pod2text-utf8 >&2");
        }
    } else {
        print <<EOF
 Первоочередное (packages, dmove, tupdate, jr, jr -do):
   packages          -- собрать пакеты (побочный эффект -- запомнится собранная версия)
   dmove             -- передвинуть пакеты в правильную ветку репозитория (testing/stable, в зависимости от типа релиза: RC/STABLE) 
   tupdate           -- обновить ТС
   jr                -- посмотреть сводку изменений для трекера
   jr -do            -- выполнить действия в трекере
   quit_release_mode -- аккуратно выйти из режима релиза

 Хотфикс (hotfix, dmove, jr, jr -do):
   slide                          -- пересобрать релиз от свежайшей ревизии. Внутри обновит рабочую копию и соберет пакеты
   slide revision                 -- пересобрать релиз от ревизии revision                        
   hotfix rev1,rev2-rev3          -- добавить в релиз указанные коммиты. Внутри создаст релизный бранч (если еще нет), замержит указанные коммиты и соберет пакеты
   jr -c 'текст текст текст'      -- ключ -c добавляет указанную строчку в комментарий

 Дополнительно:
   jr --force     -- принудительно обновить текст тикета в трекере
   up             -- обновить переменные/переключить каталоги в соответствии с релиз-тикетом (на случай, если с одним релизом работают двое)"
   jr -dep 'smth' -- ключ -dep добавляет указанную строчку в секцию 'Вообще' инструкции по выкладке
   jr -i          -- посмотреть статус релиз-тикета
   state --release-type RC -- установить тип релиза в RC; нужно для правильной работы dmove
EOF
;

        print "\n\n Все действия:\n".join "", map {"    $_\n"} grep {!/(vars_and_aliases|pre_tupdate)/} sort keys %action_sub; 
    }

    return $res;
}



=head1 url_testupdate

    Вывести ссылку на обновление ТС в Табуле    

=cut
sub url_testupdate
{
    my $res = {};

    print "\n\n".tabula_testupdate_url(var_value('PKG_VERSION'))."\n\n";

    return $res;
}

#................................................

# возвращает хеш %O с опциями из командной строки
sub parse_options
{
    #my ($argv) = @_;

    my %O = (
        "use_test_tracker"       => 0,  
    );

    #GetOptionsFromArray($argv, #TODO
    GetOptions( 
        "show-options"        => \$O{show_options},
        "help"                => \&usage,
        "action=s"            => \$O{action},
        "t|test-tracker!"     => \$O{use_test_tracker},
        "last-release"        => \$O{last_release}, 
        "j|issue=s"           => \$O{issue}, 
        "prev-issue=s"        => \$O{prev_issue}, 
        "project=s"           => \$O{project},
        "co|checkout"         => \$O{checkout},
        "release-type=s"      => \$O{release_type}, 
        "pkg|packages=s"      => \$O{packages}, 
        "md|migration-done=s@"=> \$O{migration_done}, 
        "migration-undo=s@"   => \$O{migration_undo}, 

        "i|inspect"           => \$O{inspect},
        "new-version=s"       => \$O{new_version},
        "prev|prev-version=s" => \$O{prev_version},
        "n|name=s"            => \$O{name},
        "dep|deploy-notes=s"  => \$O{deploy_notes},
        "rename=s"            => \$O{rename},
        "do"                  => \$O{do},
        "c|comment=s"         => \$O{comment},
        "rt|repeat-tests!"    => \$O{repeat_tests},
        "down|downgrade"      => \$O{downgrade},
        "ts"                  => \$O{ts},
        "verbose"             => \$O{verbose},
        "advanced"            => \$O{advanced},
        "use-svnroot!"        => \$O{use_svnroot},
        "review-issue=s"      => \$O{review_issue},
        "force"               => \$O{force},
        "no-tupdate"          => \$O{'no-tupdate'},
        "drop"                => \$O{'drop'},
        "set=i"               => \$O{'set'},
        "cc=s"                => \$O{cc},
        "testserver=s"        => \$O{testserver},

    ) or die "can't parse options";

    for my $k (keys %O){
        utf8::decode($O{$k}) if defined $O{$k} && !ref $O{$k};
    }

    if( $O{show_options} ){
        print Dumper(\%O);
        exit;
    }

    return %O;
}


# Печатает usage-сообщение
sub usage {
    system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}

