#!/usr/bin/perl

# $Id$

=encoding utf8
=head1 NAME

    tracker-limtest.pl - упрощение выкладки на среду ограниченного тестирования

=head1 SYNOPSIS

    tracker-limtest.pl -l 1 -t DIRECT-12345 -b my-super-branch
    tracker-limtest.pl -l 2 -t DIRECT-12345 -b my-super-branch --no-merge

=head1 DESCRIPTION

    Мерджит в текущий релиз изменения, сделанные в бранче, собирает пакеты и создаёт/обновляет 
    тикет в трекере на выкладывание собранного пакета на среду ограниченного тестирования.

    Код, из которого собирается пакет, остаётся в репозитории по путям вида:
    https://arcadia.yandex.ru/arc/branches/direct/limtest/<branch_name>-<release_base_revision>-<svn_head_revision>

    Собираемые пакеты получают версию:
    1.<release_base_revision>.<svn_head_revision>~<branch_name>-1

=head1 ARGUMENTS

    -t|--task -- ключ тестируемой задачи в трекере (например, DIRECT-12345) [обязательный]
    -b|--branch -- название бранча, содержащего тестируемый код (например, my-super-branch) [обязательный]
    -l|--limtest -- номер limtest'а, на который надо выложить код (limtest1/limtest2)
    --use-test-tracker -- использовать тестовый инстанс трекера
    --prod-version -- версия директа, работающая в продакшене (1.12345.12345-1) [по умолчанию получается из intapi]
    --package-version -- на выкладывание какой версии создать тикет (только с --no-merge --no-build --no-unit-test)
    --head-rev -- последняя ревизия репозитория [по умолчанию получается из репозитория]
    --no-build -- не собирать пакеты (предполагается, что пакет нужной версии уже лежит в репозитории)
    --no-unit-tests -- не запускать юнит-тесты
    --no-merge -- не мерджить изменения из релиза и бранча (предполагается, что в репозитории уже есть 
                   limtest/<branch_name>-<release_base_revision>-<svn_head_revision> и код в нём соответствует текущему
                  продакшен-релизу + изменения из бранча)
    --no-tracker -- не общаться с трекером (только мерджить/собирать пакеты/запускать юнит-тесты в зависимости от других аргументов)
    --no-upload-notify -- не посылать письмо о необходимости выложить собранную версию

=cut

use strict;
use warnings;

use Getopt::Long;
use Cwd;
use File::Temp;
use File::chdir;
use XML::Simple;
use List::Util qw/max min/;
use List::MoreUtils qw/any/;
use LWP::Simple qw/get/;
use Net::INET6Glue::INET_is_INET6;
use JSON;
use IO::Prompt;
use Yandex::Yamb;
use Yandex::Validate qw/is_valid_int/;

use ProjectSpecific qw/svn_url version2rev version2svnpath jira_limtest_ticket_project get_upload_notify_email_to get_upload_notify_email_from/;
use Startrek::Client::Easy;
use Yandex::Shell;
use Yandex::SendMail qw/sendmail/;

use utf8;
use open ':locale';

my $DIRECT_CONFIGURATION_URL = 'http://intapi.direct.yandex.ru/DirectConfiguration?keys=direct_version';

$Yandex::Shell::PRINT_COMMANDS = 1;

=head2 $LIMTEST_SERVER_COUNT

    Количество серверов limtestN-direct.yandex.ru

=cut

my $LIMTEST_SERVER_COUNT = 2;

$ENV{TMPDIR} = '/tmp/temp-ttl/ttl_2d'; # настройка File::Spec

run() unless caller();

sub run
{
    my $merge = 1;
    my $unit_tests = 1;
    my $build = 1;
    my $tracker_update = 1;
    my $upload_notify = 1;
    my ($branch_name, $tracker_func_ticket, $direct_production_version, $head_rev, $package_version, $limtest_num);

    my $original_cmd = join " ", $0, @ARGV;

    GetOptions (

        'h|elp' => sub { usage(0) },
        'b|branch=s' => \$branch_name,
        't|task=s' => \$tracker_func_ticket,
        'l|limtest=s' => \$limtest_num,
        'prod-version=s' => \$direct_production_version,
        'head-rev=s' => \$head_rev,
        'package-version=s' => \$package_version,
        'use-test-tracker' => sub { $ProjectSpecific::STARTREK_INSTANCE = $_[1] ? 'test' : 'prod' },
        'merge!' => \$merge,
        'unit-tests!' => \$unit_tests,
        'build!' => \$build,
        'tracker!' => \$tracker_update,
        'upload-notify!' => \$upload_notify,
     ) || usage(1);

    die q/Не указан номер limtest'а (-l 1 или -l 2)/ unless $limtest_num;
    die "Указан некорректный номер limtest'а: $limtest_num" unless is_valid_int($limtest_num, 0) && $limtest_num >= 1 && $limtest_num <= $LIMTEST_SERVER_COUNT;
    die 'Не указано название бранча' unless $branch_name;
    die 'Не указано название тикета для задачи' unless $tracker_func_ticket;
    die "Не верно указана версия директа: $direct_production_version" if $direct_production_version && $direct_production_version !~ /^(?:1\.)?(\d{2,7})(?:\.\d{2,7})?(?:-\d)?$/;
    die 'Нельзя указывать версию пакетов и одновременно проводить сборку' if $package_version && ($build || $merge || $unit_tests);

    my $current_user_login = getlogin || getpwuid($<) || die;
    my $send_message_to = $current_user_login;

    # делаем всё в eval'е, чтобы потом написать в консоль и ямб на каком действии сломались
    my($current_stage, $stage_additional_description);
    my $limtest_svn_path; # прокидываем, чтобы показать куда надо руками мерджить в случае ошибки
    my $tmp_dir;
    eval {
        my $tracker_deploy_ticket;
        # ищем уже существующий тикет про выкладку этого бранча
        if ($tracker_update) {
            $current_stage = 'ppclim_tracker_ticket_search';
            $tracker_deploy_ticket = find_tracker_ppclim_ticket(branch_name => $branch_name, limtest_num => $limtest_num);
        }

        # немного итогов + подтверждение от пользователя
        my $prefix_ok = "*";
        my $prefix_skip = "- [ПРОПУСКАЕМ]";
        print "\nПорядок действий:\n";
        if ($direct_production_version) {
            print "$prefix_ok считать версией продакшена директа $direct_production_version\n";
        } else {
            print "$prefix_ok получить версию продакшена директа из intapi\n";
        }
        my $prefix = '';
        $prefix = $merge ? $prefix_ok : $prefix_skip;
        print "$prefix смержить разницу между транком и бранчем в копию стабильного релиза\n";
        $prefix = $unit_tests ? $prefix_ok : $prefix_skip;
        print "$prefix запустить юнит-тесты\n";
        $prefix = $build ? $prefix_ok : $prefix_skip;
        print "$prefix собрать пакеты\n";
        if ($tracker_update) {
            if ($tracker_deploy_ticket) {
                my $tracker_deploy_ticket_url = get_tracker_ticket_url(ticket => $tracker_deploy_ticket);
                print "$prefix_ok обновить в трекере тикет $tracker_deploy_ticket_url ($tracker_deploy_ticket->{summary})\n";
            } else {
                print "$prefix_ok создать новый тикет на выкладку в трекере\n";
            }
        } else {
            print "$prefix_skip создать/обновить тикет на выкладку в трекере\n";
        }
        my $notify_email_to = get_upload_notify_email_to();
        $prefix = $upload_notify ? $prefix_ok : $prefix_skip;
        print "$prefix написать в $notify_email_to\n";
        print "\n";
        prompt('Подтверждаешь?', -yn, -default => 'y') || exit(0);

        my $branch_svnurl = svn_url('branch' => $branch_name);
        if (!$head_rev) {
            $head_rev = svn_info(svnurl => svn_url('root'))->{revision};
        }

        my ($root_dir, $production_release_base_rev);
        if ($merge || $unit_tests || $build) {
            # получаем версию директа, работающую в продакшене через intapi
            if (!$direct_production_version) {
                $current_stage = 'getting_direct_configuration';
                my $direct_configuration = get_direct_configuration();
                die 'не удалось получить данные о продакшене Директа' unless $direct_configuration && ref($direct_configuration) eq 'HASH' && $direct_configuration->{direct_version};
                $direct_production_version = $direct_configuration->{direct_version};
            }

            $production_release_base_rev = version2rev($direct_production_version, rev => 'base');
            my $limtest_svn_url = svn_url('limtest' => $branch_name, $production_release_base_rev, $head_rev);
            my $svn_root_url = svn_url('root');
            ($limtest_svn_path = $limtest_svn_url) =~ s/^\Q$svn_root_url\E//;
            my $limtest_svn_url_exists = svn_url_exists(svnurl => $limtest_svn_url);
            my $release_svn_url = version2svnpath($direct_production_version);
            my $is_release_from_trunk = $release_svn_url =~ /arc\/trunk\/arcadia\/direct/;

            # если бранча ещё нет, копируем в него текущий релиз
            if ($merge && !$limtest_svn_url_exists) {
                $current_stage = 'limtest_branch_creation';
                # релиз может быть собран как из транка, так и из релизного бранча (при наличии хотфиксов)
                # в случае если релиз собран из транка, то нужно копировать базовую ревизию
                # если релиз собран из бранча, то нужно копировать HEAD релизного бранча
                my $revision_for_copy = $is_release_from_trunk ? $production_release_base_rev : 'HEAD';
                svn_copy(from => $release_svn_url, to => $limtest_svn_url, rev => $revision_for_copy, message => "Создан бранч для тестирования $tracker_func_ticket на среде ограниченного тестирования limtest$limtest_num. Используется код из бранча $branch_name.");
            }

            $current_stage = 'source_checkout';
            $tmp_dir = checkout_source(svnurl => $limtest_svn_url, rev => 'HEAD'); # HEAD -- чтобы захватить ревизию, в которой создали бранч
            $root_dir = "$tmp_dir";

            # мерджим разницу между транком и бранчем в текущий релиз
            if ($merge) {
                $current_stage = 'branch_vs_trunk_merge';
                my $branch_merged_trunk_rev = get_last_merged_trunk_rev(svnurl => $branch_svnurl, rev => $head_rev);
                my $trunk_source_svnurl = svn_url('trunk') . '@' . $branch_merged_trunk_rev;
                my $branch_source2_svnurl = $branch_svnurl . '@' . $head_rev;
                print "Мерджу разницу между $trunk_source_svnurl и $branch_source2_svnurl\n";

                eval { svn_merge(source => $trunk_source_svnurl, source2 => $branch_source2_svnurl, working_copy => $root_dir, reintegrate => 1); };
                if ($@) {
                    $stage_additional_description = "При мердже бранча обнаружились конфликты. Надо их разрешить руками в рабочей копии $root_dir, закоммитить, удалить $tmp_dir и повторить попытку сборки с дополнительными параметрами --no-merge --head $head_rev.\nПодходящее коммит-сообщение:\nразница между $trunk_source_svnurl и $branch_source2_svnurl\n(с ручным разрешением конфликтов)" if $@ =~ /CONFLICT/;
                    $tmp_dir->unlink_on_destroy(0);
                    die $@;
                }
                svn_commit(working_copy => $root_dir, message => "разница между транком и бранчем $branch_name");
            }

            # TODO: проверять mergeinfo в бранче

            # при мердже могли измениться файлы в packages, поэтому обновляем свою рабочую копию
            svn_up(working_copy => $root_dir, accept => 'theirs-full');
        }

        # юнит-тесты
        if ($unit_tests) {
            $current_stage = 'unit_tests';
            run_unit_tests(root_dir => $root_dir);
        }

        # сборка пакетов  
        if ($build) {
            $current_stage = 'packages';
            $package_version = build_packages(root_dir => $root_dir);
        } elsif (!$package_version) {
            # преобразуем имя бранча согласованно с dh-svn-release: меняем "_" на "-" и аккуратно укорачиваем
            (my $branch_name_for_version = $branch_name) =~ s/_/-/g;
            $branch_name_for_version = substr($branch_name_for_version, 0, 45);
            $branch_name_for_version =~ s/-+$//;
            $package_version = "1.${production_release_base_rev}.$head_rev~$branch_name_for_version-1";
        }

        # запись в трекер версии пакета (при необходимости -- создание тикета)
        if ($tracker_update) {
            $current_stage = 'tracker_update';
            my $tracker_deploy_ticket_already_exists = $tracker_deploy_ticket ? 1 : 0;
            my $need_update = !$tracker_deploy_ticket_already_exists || need_update_tracker_ticket(tracker_deploy_ticket => $tracker_deploy_ticket, branch_svnurl => $branch_svnurl);
            if ($need_update) {
                my $ticket_key = update_tracker_ppclim_ticket(
                    ticket => $tracker_deploy_ticket,
                    branch_name => $branch_name,
                    package_version => $package_version,
                    tracker_func_ticket => $tracker_func_ticket,
                    limtest_num => $limtest_num,
                );
                if ($tracker_deploy_ticket_already_exists) {
                    print "Обновлён тикет в трекере: $ticket_key\n";
                } else {
                    print "Создан тикет в трекере: $ticket_key\n";
                }
                if ($upload_notify) {
                    send_upload_notify(
                        upload_ticket => $ticket_key,
                        package_version => $package_version,
                        branch_name => $branch_name,
                        limtest_num => $limtest_num,
                    );
                }
            }
        }

        send_yamb_message(to => $send_message_to, message => "tracker-limtest.pl: успешно");
        print "\n\n\nDONE\n";
    };
    if ($@) {
        $tmp_dir->unlink_on_destroy( 0 ) if $tmp_dir;
        if ($current_stage) {
            # собираем текст о произошедшей ошибке
            my %stage_names = (
                ppclim_tracker_ticket_search => 'поиска в трекере тикета на выкладку',
                getting_direct_configuration => 'получения версии продакшена директа',
                limtest_branch_creation => 'создания бранча в limtest',
                source_checkout => 'чекаута',
                branch_vs_trunk_merge => 'мерджа разницы между транком и бранчем в текущий релиз',
                unit_tests => 'юнит-тестов',
                packages => 'сборки пакетов',
                tracker_update => 'обновления трекера',
            );
            my $stage_name = $stage_names{$current_stage} || "неизвестной стадии $current_stage";
            my $err_text = "Произошла ошибка во время $stage_name: $@";
            if ($stage_additional_description) {
                $err_text .= "\n$stage_additional_description";
            }
            my $tracker_error_text = 'Если с трекером проблемы, можно пропустить обращения к нему, добавив ключ --no-tracker';
            $limtest_svn_path ||= 'limtest/<название бранча>-<базовая-ревизия-продакшена-директа>-<последняя-ревизия-в-репозитории>';
            $head_rev ||= '<последняя-ревизия-в-репозитории>';
            my %stage_additional_info = (
                ppclim_tracker_ticket_search => $tracker_error_text,
                getting_direct_configuration => 'Если с intapi проблемы, можно принудительно указать версию продакшена директа, добавив ключ --prod-version=<версия>',
                limtest_branch_creation => 'Можно вручную сделать бранч limtest/<название бранча>-<базовая-ревизия-продакшена-директа>-<последняя-ревизия-в-репозитории> и пропустить действия по мерджу, добавив ключ --no-merge --head-rev <последняя-ревизия-в-репозитории>',
                source_checkout => '', # не придумал полезного текста :)
                branch_vs_trunk_merge => "Можно вручную сделать мердж в бранч $limtest_svn_path и пропустить действия по мерджу, добавив ключ --no-merge --head $head_rev",
                unit_tests => 'Можно пропустить запуск юнит-тестов, добавив ключ --no-unit-tests',
                packages => 'Можно собрать пакеты вручную и пропустить сборку, добавив ключ --no-build',
                tracker_update => $tracker_error_text,
            );
            my $stage_info = $stage_additional_info{$current_stage};
            if ($stage_info) {
                $err_text .= "\n$stage_info";
            }
            $err_text .= "\nИсходная команда: $original_cmd  --head-rev $head_rev";
            send_yamb_message(to => $send_message_to, message => $err_text);
            die $err_text;
        }
    }
}


=head2 get_tracker_ticket_url

    ticket

=cut

sub get_tracker_ticket_url
{
    my (%O) = @_;

    die unless $O{ticket};
    my $ticket_key = (ref($O{ticket}) eq '') ? $O{ticket} : $O{ticket}->{key};
    my $ticket_url = ProjectSpecific::get_data(startrek_url => $ProjectSpecific::STARTREK_INSTANCE) . '/'. $ticket_key;
    return $ticket_url;
}

=head2 svn_info

    svnurl

    {
          'commit' => {
                      'revision' => '1074',
                      'date' => '2012-02-06T12:51:30.550175Z',
                      'author' => 'lena-san'
                    },
          'repository' => {
                          'root' => 'svn+ssh://svn.yandex.ru/direct-utils',
                          'uuid' => 'e689b325-ea81-41fd-a1b7-d51f2a3aec1c'
                        },
          'kind' => 'dir',
          'revision' => '1113',
          'url' => 'svn+ssh://svn.yandex.ru/direct-utils/release_shell_c/trunk',
          'path' => '.',
          'wc-info' => {
                       'depth' => 'infinity',
                       'schedule' => 'normal'
                     }
    };

=cut

sub svn_info
{
    my(%O) = @_;

    die 'Не указан svnurl' unless $O{svnurl};

    my $xml_info = yash_qx(qw/svn info --xml --non-interactive/, $O{svnurl});
    my $xs = XML::Simple->new();
    my $info = $xs->XMLin($xml_info);
    die "Странный XML от svn info:\n" . $xml_info unless $info && ref($info) eq 'HASH' && $info->{entry} && ref($info->{entry}) eq 'HASH';
    return $info->{entry};
}

=head2 svn_get_merge_info

    svnurl
    rev

    $VAR1 = {
          '/branches/xls_for_all' => '12858-12918,12975',
          '/branches/geo-campaigns' => '19667-20015',
          '/trunk' => '25255-28972',
    };

=cut

sub svn_get_merge_info
{
    my %O = @_;

    die 'Не указан svnurl' unless $O{svnurl};
    my $rev = $O{rev} || 'HEAD';

    my $merge_info = yash_qx(qw/svn propget svn:mergeinfo --non-interactive -r/, $rev, $O{svnurl});
    return { map { split /:/ } split /\r?\n/, $merge_info };
}

=head2 expand_svn_rev_list

    @list = expand_svn_rev_list('10-12,17,18-20,23,24');
    # @list => (10,11,12,17,18,19,20,23,24);

=cut

sub expand_svn_rev_list
{
    my ($rev_list) = @_;

    die unless $rev_list;

    return map { my @x = split /-/; @x > 1 ? $x[0]..$x[1] : $x[0] } split /,/, $rev_list;
}

=head2 get_direct_configuration

    {
        'direct_version' => '1.28967.29147-1',
        'db_info' => {...},
    }

=cut

sub get_direct_configuration
{
    my $json_config_data = get($DIRECT_CONFIGURATION_URL);
    die "Не удалось получить данные с $DIRECT_CONFIGURATION_URL" unless $json_config_data;

    return decode_json($json_config_data);
}

=head2 get_last_merged_trunk_rev

    svnurl
    rev

=cut

sub get_last_merged_trunk_rev
{
    my (%O) = @_;

    die unless $O{svnurl};

    # если в бранч подмердживался транк, берём старшую подмердженную ревизию
    # TODO: это сломается, если подмердживать ревизии выборочно. стоит проверять непрерывность ряда ревизий
    my $merge_info = svn_get_merge_info(svnurl => $O{svnurl}, rev => $O{rev});
    my $branch_trunk_revs = $merge_info->{'/trunk'};
    if ($branch_trunk_revs) {
        my @revlist = expand_svn_rev_list($branch_trunk_revs);
        return max @revlist;
    } else {
        # если транк не мерджился, ищем ревизию, от которой создан бранч
        my $revisions_str = yash_qx(qw/svn mergeinfo --show-revs eligible --non-interactive/, svn_url('trunk'), $O{svnurl});
        my @revlist = map {/^r(\d+)$/ && $1} split /\n+/, $revisions_str;
        my $min_rev_to_merge = min @revlist;
        # последняя подмердженная ревизия -- предыдущая к той, что svn собирается мерджить первой
        return $min_rev_to_merge - 1;
    }
}

=head2 checkout_source

    svnurl
    rev

    returns File::Temp object

=cut

sub checkout_source
{
    my (%O) = @_;

    die unless $O{svnurl};
    my $rev = $O{rev} || 'HEAD';

    my $tmpdir = File::Temp->newdir();
    my $svnurl = $O{svnurl} . '@' . $rev;
    my $tmpdir_dirname = $tmpdir->dirname(); # ->dirname() возвращяет путь со слешем в конце. это важно :)
    yash_system(qw/direct-svn-checkout/, $svnurl, $tmpdir_dirname);

    return $tmpdir;
}


=head2 build_packages

    root_dir

    returns built package version

=cut

sub build_packages
{
    my (%O) = @_;

    die unless $O{root_dir};

    yash_system(qw/svn up/, $O{root_dir});
    yash_system(qw/svn-reset/, $O{root_dir});
    yash_system('echo test | gpg --sign > /dev/null');

    my $package_version;
    {
        local $CWD = $O{root_dir} . '/packages';
        local $Yandex::Shell::CHDIR = 0;
        # собираем пакеты и кладём их в репозиторий
        my $text = yash_qx('make ' . yash_quote('SVN_ROOT='.$O{root_dir}) . ' all'); # 2>&1
        print $text if $text;
        if ($text =~ /source version *([^\n]*)/sm) {
            $package_version = $1;
        } else {
            die 'Не нашёл версию собранного пакета';
        }
    }

    return $package_version;
}

=head2 run_unit_tests

    root_dir

=cut

sub run_unit_tests
{
    my (%O) = @_;

    die unless $O{root_dir};

    # запускам юнит-тесты. в случае проблем yash_system умрёт
    my $dir = getcwd;
    chdir $O{root_dir};
    yash_system("/usr/local/bin/direct-mk", "test-full");
    chdir $dir;
}

=head2 find_tracker_ppclim_ticket

    branch_name

=cut

sub find_tracker_ppclim_ticket
{
    my (%O) = @_;

    die unless $O{branch_name};
    die unless $O{limtest_num};

    my $startrek = Startrek::Client::Easy->new(startrek => $ProjectSpecific::STARTREK_INSTANCE);
    my $summary_to_search = $O{branch_name}.'@ppclim'.$O{limtest_num};
    my $admin_project = jira_limtest_ticket_project();
    my $found_tickets = $startrek->get(query => qq/Queue: $admin_project Summary: "$summary_to_search"/, limit => 20, array => 1);
    if ($found_tickets && ref($found_tickets) eq 'ARRAY' && @$found_tickets > 0) {
        my @tickets = grep {$_->{summary} && $_->{summary} =~ /^\Q$summary_to_search\E/} @$found_tickets;
        if (@tickets == 1) {
            return $tickets[0];
        } elsif (@tickets > 1) {
            die 'Нашли больше одного тикета про бранч в трекере: ' . join(', ', (map {$_->{key}} @tickets));
        }
    }
    return undef;
}

=head2 get_version_data_from_summary

    $version_data = get_version_data_from_summary('my-branch@ppclim1: выложить на среду ограниченного тестирования версию 1.12345.67890~my-branch-1');
    $version_data => {
        package_version => '1.12345.67890~my-branch-1',
        base_rev => 12345,
        last_rev => 67890,
        branch_name => 'my-branch',
        limtest_num => 1,
    };

=cut

sub get_version_data_from_summary
{
    my ($summary) = @_;

    die unless $summary;

    my ($package_version, $base_rev, $last_rev, $branch_name) = ($summary =~ /(\d+\.(\d+)(?:\.(\d+))?(?:~([\w\-]+))?-\d+)$/);
    my ($limtest_num) = ($summary =~ /^\Q$branch_name\@ppclim(\d+)\E/);
    return {
        package_version => $package_version,
        base_rev => $base_rev,
        last_rev => $last_rev,
        branch_name => $branch_name,
        limtest_num => $limtest_num,
    };
}

=head2 need_update_tracker_ticket

    tracker_deploy_ticket
    branch_svnurl

    returns need_update

=cut

sub need_update_tracker_ticket
{
    my (%O) = @_;

    my $svninfo = svn_info(svnurl => $O{branch_svnurl});
    my $trunk_rev = get_last_merged_trunk_rev(svnurl => $O{branch_svnurl});

    my $ticket_version_data = get_version_data_from_summary($O{tracker_deploy_ticket}->{summary});
    my $ticket_base_rev = $ticket_version_data->{base_rev};
    my $ticket_last_rev = $ticket_version_data->{last_rev};
    if ($ticket_base_rev > $trunk_rev) {
        die "Ревизия транка в тикете ($ticket_base_rev) больше, чем в только что собранном пакете ($trunk_rev)";
    } elsif ($ticket_last_rev > $svninfo->{revision}) {
        die "Ревизия последнего коммита в тикете ($ticket_last_rev) больше, чем в только что собранном пакете (" . $svninfo->{revision} . ')';
    } elsif ($ticket_base_rev == $trunk_rev && $ticket_last_rev > $svninfo->{revision}) {
        return 0;
    }

    return 1;
}

=head2 make_ticket_summary

    branch_name
    package_version
    limtest_num

=cut

sub make_ticket_summary
{
    my (%O) = @_;

    die unless $O{branch_name};
    die unless $O{package_version};
    die unless $O{limtest_num};

    return $O{branch_name} . '@ppclim' . $O{limtest_num} . ': выложить на среду ограниченного тестирования limtest' . $O{limtest_num} .' версию ' . $O{package_version};
}

=head2 update_tracker_ppclim_ticket

    ticket
    branch_name
    package_version
    tracker_func_ticket
    limtest_num

    returns ticket name

=cut

sub update_tracker_ppclim_ticket
{
    my (%O) = @_;

    die unless $O{branch_name};
    die unless $O{package_version};
    die unless $O{tracker_func_ticket};
    die unless $O{limtest_num};

    my $new_ticket = {};

    if ($O{ticket} && $O{ticket}->{key}) { # правим уже существующий тикет
        $new_ticket->{key} = $O{ticket}->{key};
        $new_ticket->{comment} = 'Версия обновлена до %%' . $O{package_version} . '%%';
        if ($O{ticket}->{status} eq 'closed') {
            $new_ticket->{actions} = ['reopen'];
        }
    } else {
        $new_ticket->{create} = 1;
        $new_ticket->{description} = 'Среду ограниченного тестирования limtest' . $O{limtest_num} . ' надо залочить на версию %%' . $O{package_version} . '%% для тестирования задачи ' . $O{tracker_func_ticket} . '.';
        $new_ticket->{internal} = 1;
        $new_ticket->{type} = 'task';
        $new_ticket->{queue} = jira_limtest_ticket_project();
        $new_ticket->{components} = ["Limtest"];
    }
    $new_ticket->{summary} = make_ticket_summary(branch_name => $O{branch_name}, package_version => $O{package_version}, limtest_num => $O{limtest_num});

    my $startrek = Startrek::Client::Easy->new(startrek => $ProjectSpecific::STARTREK_INSTANCE);
    my $created_ticket_key = $startrek->do(%$new_ticket);
    if ($created_ticket_key) {
        my $tracker_ppclim_ticket_url = get_tracker_ticket_url(ticket => $created_ticket_key);
        print "\nСоздал тикет на выкладывание версии $O{package_version} на limtest$O{limtest_num}: $tracker_ppclim_ticket_url\n";
        return $created_ticket_key;
    } else {
        return $O{ticket}->{key};
    }
}

=head2 send_upload_notify

    upload_ticket
    package_version
    branch_name
    limtest_num

=cut

sub send_upload_notify
{
    my (%O) = @_;

    die unless $O{upload_ticket};
    die unless $O{package_version};
    die unless $O{branch_name};
    die unless $O{limtest_num};

    my $ticket_key = (ref($O{upload_ticket}) eq '') ? $O{upload_ticket} : $O{upload_ticket}->{key};
    my $ticket_summary = make_ticket_summary(branch_name => $O{branch_name}, package_version => $O{package_version}, limtest_num => $O{limtest_num});
    my $subject = "Ready To Deploy: ($ticket_key) $ticket_summary";
    my $ticket_url = get_tracker_ticket_url(ticket => $O{upload_ticket});

    my $content = "По тикету $ticket_key ( $ticket_url ) необходимо залочить среду ограниченного тестирования limtest$O{limtest_num} на версию $O{package_version}";
    sendmail(get_upload_notify_email_to(), get_upload_notify_email_from(), $subject, \$content);
}

=head2 svn_url_exists

    svnurl
    rev

=cut

sub svn_url_exists
{
    my (%O) = @_;

    die unless $O{svnurl};
    my $rev = $O{rev} || 'HEAD';

    my $cmd = join(' ', (qw/svn ls --non-interactive --revision/, yash_quote($rev), yash_quote($O{svnurl}), '2>&1 1>/dev/null'));
    my $err = qx($cmd);
    if (!$? && !$err) {
        return 1;
    } elsif ($? && $err && $err =~ /non-existent in ([a-z ]+)?revision/) {
        return 0;
    } else {
        die $err;
    }
}

=head2 svn_copy

    from
    to
    rev
    message

=cut

sub svn_copy
{
    my (%O) = @_;

    die unless $O{from};
    die unless $O{to};
    die unless $O{message};
    my $rev = $O{rev} || 'HEAD';

    yash_system(qw/svn copy --revision/, $rev, '--message', $O{message}, $O{from}, $O{to});
}

=head2 svn_merge

    source
    source2
    working_copy
    reintegrate

=cut

sub svn_merge
{
    my (%O) = @_;

    die unless $O{source};
    die unless $O{working_copy};

    yash_system(qw/svn merge --accept postpone/, '--ignore-ancestry', $O{source}, $O{source2} || (), $O{working_copy});
    svn_check_conflicts(working_copy => $O{working_copy});
}

=head2 svn_check_conflicts

    working_copy

=cut

sub svn_check_conflicts
{
    my (%O) = @_;

    die unless $O{working_copy};

    my $status_xml = yash_qx(qw/svn status --xml --quiet/, $O{working_copy});
    my $xs = XML::Simple->new();
    my $info = $xs->XMLin($status_xml);
    die unless $info && ref($info) eq 'HASH' && $info->{target} && $info->{target}->{entry};
    my $entries = (ref($info->{target}->{entry}) eq 'ARRAY') ? $info->{target}->{entry} : [$info->{target}->{entry}];

    if (any { ($_->{'wc-status'}->{item} || '') eq 'conflicted' || ($_->{'wc-status'}->{'tree-conflicted'} || '') eq 'true' } @$entries) {
        die 'CONFLICT';
    }
}

=head2 svn_commit

    working_copy
    message

=cut

sub svn_commit
{
    my (%O) = @_;

    die unless $O{working_copy};
    die unless $O{message};

    yash_system(qw/svn commit --non-interactive --message/, $O{message}, $O{working_copy});
}

=head2 svn_up

    working_copy
        rev
        accept => postpone|base|mine-conflict|theirs-conflict|mine-full|theirs-full|edit|launch

=cut

sub svn_up {
    my (%O) = @_;

    die unless $O{working_copy};
    my $rev = $O{rev} || 'HEAD';
    my $accept = $O{accept} || 'postpone';

    yash_system(qw/svn up --non-interactive --ignore-externals --accept/, $accept, $O{working_copy});
}


=head2 usage

=cut

sub usage
{
    my ($exitval) = @_;
    # HelpMessage на старом перле/Getopt::Long/etc дважды кодирует utf8, портя вывод
    # Getopt::Long::HelpMessage(-exitval => $exitval, -verbose => 1, utf8 => 1);
    yash_system('podselect -section NAME -section SYNOPSIS -section DESCRIPTION -section ARGUMENTS ' . yash_quote($0). ' | pod2text-utf8 >&2');
    exit($exitval);
}

1;
