# -*- coding:utf-8 -*-
import re
from django.db import models
from datetime import datetime
from releaser.users.models import User
from releaser.svnlog.models import SvnLog, SvnLogHotfix, SvnPath
from releaser.versionica.models import HostProperty, Host, Property, HostGroup

from django.conf import settings

class AutoBeta(models.Model):
    STATUS_TO_CREATE      =  1
    STATUS_CREATING       =  5
    STATUS_WORK           = 10
    STATUS_PAUSE          = 20
    STATUS_TO_DELETE      = 50
    STATUS_DELETED        = 55
    STATUS_CHOICES = (
            (STATUS_TO_CREATE,   'to create'),
            (STATUS_WORK,        'work'),
            (STATUS_PAUSE,       'pause'),
            (STATUS_TO_DELETE,   'to delete'),
            (STATUS_DELETED,     'deleted'),
            (STATUS_CREATING,    'creating'),
            )
    SOURCE_TYPE_BRANCH    = 1
    SOURCE_TYPE_HOST      = 2
    SOURCE_TYPE_HOSTGROUP = 3
    SOURCE_TYPE_WC        = 4
    SOURCE_CHOICES = (
            (SOURCE_TYPE_BRANCH,    'branch'),
            (SOURCE_TYPE_HOST,      'host'),
            (SOURCE_TYPE_HOSTGROUP, 'hostgroup'),
            (SOURCE_TYPE_WC,        'working copy'),
            )
    port = models.IntegerField( default=-1 )
    source_type = models.IntegerField( choices=SOURCE_CHOICES, default=SOURCE_TYPE_BRANCH)
    source_branch = models.ForeignKey( SvnPath, null=True )
    source_host = models.ForeignKey(Host, null=True, related_name='+')
    source_hostgroup = models.ForeignKey(HostGroup, null=True)
    source_working_copy = models.CharField( max_length = 200, default='' )
    jira_issue = models.CharField( max_length = 50, db_index=True, default='' )
    author = models.ForeignKey( User )
    target_host = models.ForeignKey( Host, related_name='+' )
    comment  = models.CharField( max_length = 500 )
    update_period = models.IntegerField( default=-1 )
    current_revision = models.IntegerField( default=-1 )
    last_update_time = models.DateTimeField(null=True)
    status = models.IntegerField( choices=STATUS_CHOICES, default=STATUS_TO_CREATE )
    conf_name = models.CharField( max_length = 50, default='' )
    working_copy_path = models.CharField( max_length = 200, default='' )
    actions = models.CharField( max_length = 200, default='' )
    expiration_time = models.DateTimeField(null=True)

    def __unicode__(self):
        return "%s:%s (%s)" % (self.target_host, self.port, self.source_branch.path)

    def check_source(self):
        if self.source_type == SOURCE_TYPE_BRANCH and self.source_branch == None:
            raise Exception("missing branch")
        elif self.source_type == SOURCE_TYPE_HOST and self.source_host == None:
            raise Exception("missing host")
        elif self.source_type == SOURCE_TYPE_HOSTGROUP and self.source_hostgroup == None:
            raise Exception("missing hostgroup")
        elif self.source_type == SOURCE_TYPE_WC and self.source_working_copy == '':
            raise Exception("missing source working copy")
        else:
            pass
        return

    def to_dict(self):
        d = {
                'id': self.id,
                'port': self.port,
                'source_type': self.get_source_type_display(),
                'source_branch': self.source_branch.path,
                'status': self.get_status_display(),
                'path': self.working_copy_path,
                'target_host': self.target_host.name,
                'actions': self.actions,
                'expiration_time': self.expiration_time_to_str(),
                }
        return d

    def status_nicename(self):
        return self.get_status_display()

    def last_update(self):
        if self.last_update_time is None:
            return ''
        return self.last_update_time.strftime('%Y-%m-%d %H:%M:%S')

    def expiration_time_to_str(self):
        if self.expiration_time is None:
            return ''

        return self.expiration_time.strftime('%Y-%m-%d %H:%M:%S')

    def url(self):
        if self.port < 0:
            return ''
        try:
            return "https://%s.%s" % (self.port, settings.AUTOBETA_URL[self.target_host.name])
        except KeyError:
            return 'unknown'

    def actions_count(self):
        return len([a for a in re.split(u'[,:;]+', self.actions) if a != '' ])




class AutoBetaLog(models.Model):
    TYPE_INFO     =  1
    TYPE_ERROR    = 10
    TYPE_CHOICES = (
            (TYPE_INFO,    'info'),
            (TYPE_ERROR,  'error'),
            )
    # autobeta_id не ForeignKey для того, чтобы
    # можно было записывать логи, не привязанные
    # к конкретной автобете
    autobeta_id = models.IntegerField( default=-1, db_index = True )
    logtime = models.DateTimeField(db_index = True)
    type = models.IntegerField( choices=TYPE_CHOICES, default=TYPE_ERROR)
    logtext = models.TextField(max_length=32*1024*1024, null=True)
    def save(self, *args, **kwargs):
            self.logtime = datetime.now()
            super(AutoBetaLog, self).save(*args, **kwargs)
    def __unicode__(self):
        return "%s: %s %s" % (self.autobeta, self.logtime, self.get_type_display())
    def time(self):
        return self.logtime.strftime('%Y-%m-%d %H:%M:%S')
