# -* encoding: utf-8 -*-
from django.http import HttpResponse, Http404
from django.shortcuts import render_to_response, redirect
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.db.models import Q
import json

import math, sys, os, random, time, datetime, string, yaml, re
from datetime import datetime, timedelta, date

from django.conf import settings
from releaser.flagman.models import *

from releaser.rights.tools import allowed


def index(r):
    flags = Flag.objects.all().order_by('name')
    return render_to_response('flagman/index.html',
            {
                'flags': flags,
            },
            context_instance=RequestContext(r)
            )

def bookmarks(r):
    flags = Flag.objects.all().filter(name__regex='^bookmark').order_by('name')
    return render_to_response('flagman/bookmarks.html',
            {
                'flags': flags,
            },
            context_instance=RequestContext(r)
            )

def get(r):
    name = r.GET.get('name', '').strip()
    if name == '':
        raise Http404

    try:
        flag = Flag.objects.get(name=name)
    except Flag.DoesNotExist:
        raise Http404

    res = {
            'name': flag.name,
            'valid': 1 if flag.valid() else 0
            }

    if res['valid'] == 1:
        res['value'] = flag.value
        res['start'] = flag.start()
        res['end'] = flag.end()
        res['duration'] = flag.duration
        res['now'] = datetime.now().strftime('%Y-%m-%d %H:%M:%S')

    json_data = json.dumps(res, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


@login_required
@allowed(['release_manager', 'tabula_superuser', 'manage_flags'])
def set(r):
    name = r.GET.get('name', '').strip()
    value = r.GET.get('value', '').strip()
    duration = r.GET.get('duration', '').strip()
    if name == '':
        raise Http404

    try:
        flag = Flag.objects.get(name=name)
    except Flag.DoesNotExist:
        raise Http404

    if not flag.online_editable:
        raise Http404

    flag.starttime = datetime.now()
    flag.duration = duration if duration != '' else flag.default_duration
    flag.value = value
    flag.save()

    return redirect('/flagman')


@login_required
@allowed(['release_manager', 'tabula_superuser', 'manage_flags'])
def delete(r):
    name = r.GET.get('name', '').strip()
    if name == '':
        raise Http404

    try:
        flag = Flag.objects.get(name=name)
    except Flag.DoesNotExist:
        raise Http404

    flag.delete()

    return redirect('/flagman')


@login_required
@allowed(['developer'])
def refresh_flags(r):
    flags = yaml.load(open(settings.FLAGMAN_FLAGS_FILE))

    for f in flags:
        rec, new = Flag.objects.get_or_create(
                name = f['name'],
                defaults={
                    'starttime': datetime.now(),
                    }
                )
        if not new:
            continue
        if 'online_editable' in f:
            rec.online_editable = f['online_editable'] != 0
        for param in ['value', 'duration', 'default_duration', 'description']:
            if param in f:
                setattr(rec, param, f[param])
        rec.save()

    return redirect('/flagman' )

def list_docs(r):
    res = os.listdir(settings.FLAGMAN_DOCS_DIR)

    json_data = json.dumps(res, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


# wget 'http://direct-dev.yandex-team.ru:8800/flagman/getdoc?name=2'
def get_doc(r):
    name = r.GET.get('name', '').strip()
    if not re.match(r'^[a-zA-Z0-9\._-]+$', name):
        raise Http404
    file = settings.FLAGMAN_DOCS_DIR + "/" + name
    try:
        f = open(file , 'r')
        res = f.read()
    except IOError:
        raise Http404

    return HttpResponse(res, mimetype="text/plain; charset=utf-8")

