# -*- coding:utf-8 -*-

from django.conf import settings

from textwrap import dedent

from releaser.hotfixing.models import LimtestRequest
from releaser.subscriptions.events_base import BaseEvent, event
from releaser.subscriptions.models import Subscription
from releaser.versionica.models import GroupIncludesHost, PropertyGroup, HostProperty, HostGroup

@event
class LimtestRequestEvent(BaseEvent):
    event_type = 'limtest_request'

    @staticmethod
    def display_subscription(sub):
        return u'Новый запрос на limtest'

    def current_versions_message(self):
        if hasattr(self, 'current_versions_message_cache'):
            return self.current_versions_message_cache

        limtest_hosts = GroupIncludesHost.objects.filter(group__name='limtest').values_list('host', flat=True)
        default_package = PropertyGroup.objects.get(name = "packages").default_property
        package_records = HostProperty.objects.filter( host__in = limtest_hosts, group__name = "packages", property__name__regex = default_package )

        version_template = "{name:>30s} -- {value}"

        current_versions = [ version_template.format(name=rec.host.name, value=rec.value) for rec in package_records ]

        production_main_host = HostGroup.objects.get(name = 'production').main_host
        production_version = HostProperty.objects.get(
            host=production_main_host,
            group__name='packages',
            property__name__regex=default_package,
            ).value

        current_versions += [version_template.format(name=production_main_host.name, value=production_version)]
        self.current_versions_message_cache = "\n".join(current_versions)

        return self.current_versions_message_cache

    def message_text(self):
        lr = self.event_data['request']
        issue_status = self.event_data['issue_status']
        versionica_url = self.event_data['versionica_url']

        msg = dedent(u"""\
            (Табула) требуется выкладка на limtest

            автор: {author}

            бранч: {branch}
            тикет: {ticket} ({ticket_status})

            тесты:
            {unittests_report}

            комментарий:
            {comment}

            текущие версии:
            {current_versions}

            Версионика:
            {versionica_url}
            """).format(
            author=lr.applicant.login,
            branch=lr.branch,
            ticket=lr.tracker_issue,
            ticket_status=issue_status,
            unittests_report=lr.unittests_report,
            comment=lr.applicant_message,
            current_versions=self.current_versions_message(),
            versionica_url=versionica_url,
            )

        return msg

    def notification_content_for_yamb(self, recipient):
        return self.message_text()

    def notification_content_for_email(self, recipient):
        lr = self.event_data['request']
        subject = u'Запрос на выкладку на limtest: {branch} / {ticket}'.format(
            branch=lr.branch,
            ticket=lr.tracker_issue,
            )
        return { 'subject' : subject, 'body' : self.message_text() }

@event
class LimtestDeployedEvent(BaseEvent):
    event_type = 'limtest_deployed'

    @staticmethod
    def display_subscription(sub):
        lr = LimtestRequest.objects.get(lr_id=int(sub.params))
        return u'Изменения из бранча {branch} попали на limtest ({ticket})'.format(branch=lr.branch, ticket=lr.tracker_issue)

    def notification_content(self, notification_type, recipient):
        lr = self.event_data['request']
        hostname = self.event_data['hostname']

        # версия пакетов содержит тильду (~), которая в вики-синтаксисе специальный символ;
        # чтобы в комментарии версия показывалась правильно, нужно перед версией ставить ещё одну тильду
        package_version_display = '~{package_version}' if notification_type == Subscription.NOTIFICATION_TYPE_STARTREK else '{package_version}'

        # если уведомление отправляем в стартрек, ссылку на тикет он сделает сам
        # (пусть даже избыточную: комментарий пишется в тот же тикет, на который ссылка);
        # иначе нужно сформировать адрес
        ticket_display = '{ticket}' if notification_type == Subscription.NOTIFICATION_TYPE_STARTREK else settings.STARTREK_DISPLAY_URL + '/{ticket}'

        message_template = dedent(u"""\
            (Табула) изменения из бранча попали на limtest

            бранч: {branch}
            тикет: """ + ticket_display + u"""

            сервер: {hostname}
            версия пакетов: """ + package_version_display + u"""

            Версионика: {versionica_url}
            """)

        message = message_template.format(
            branch=lr.branch,
            ticket=lr.tracker_issue,
            hostname=hostname,
            package_version=self.event_data['package_version'],
            versionica_url=u'https://{tabula_hostname}/versionica/property?group=packages&host=%5E{hostname}%24'.format(
                tabula_hostname=settings.RELEASER_DNS,
                hostname=hostname,
                )
            )

        if notification_type == Subscription.NOTIFICATION_TYPE_EMAIL:
            subject = u'Изменения из бранча {branch} попали на limtest ({ticket})'.format(branch=lr.branch, ticket=lr.tracker_issue)
            return { 'subject': subject, 'body': message }

        return message
