# -*- coding:utf-8 -*-
from django.db import models

from releaser.svnlog.models import SvnLog
from releaser.svnrelease.models import SvnRelease
from datetime import datetime

from django.conf import settings
from releaser.utils import make_display_url

class Issue(models.Model):
    issue_id = models.CharField(max_length=20, primary_key=True )
    assignee = models.CharField(max_length=100, default="")
    qa_engineer = models.CharField(max_length=100, default="")
    summary = models.CharField(max_length=200)
    commits = models.ManyToManyField(SvnLog, db_column='rev')
    releases = models.ManyToManyField(SvnRelease, through='Problem',
                                           symmetrical=False,
                                           related_name='problem_of_release')
    def __unicode__(self):
        return self.issue_id
    def save(self, *args, **kwargs):
        if self.assignee == None:
            self.assignee = ''
        super(Issue, self).save(*args, **kwargs)
    def display_url(self):
        return make_display_url(self.issue_id)


class Problem(models.Model):
    # Важно!
    # Константы для origin используются также в fetch_release_problems.pl
    # Если константы меняются -- надо править и его тоже
    ORIGIN_DESCRIPTION   =  1
    ORIGIN_HOTFIX        = 10
    ORIGIN_COMMENT       = 20
    ORIGIN_AFTER_RELEASE = 50
    ORIGIN_CHOICES = (
            (ORIGIN_DESCRIPTION,   'description'),
            (ORIGIN_HOTFIX,        'hotfix'),
            (ORIGIN_COMMENT,       'comment'),
            (ORIGIN_AFTER_RELEASE, 'created after release'),
            )

    RELEVANCE_UNKNOWN  = 0
    RELEVANCE_NO  = 10
    RELEVANCE_YES = 20
    RELEVANCE_CHOICES = (
            (RELEVANCE_UNKNOWN, 'n/a'),
            (RELEVANCE_NO, 'No'),
            (RELEVANCE_YES, 'Yes'),
            )

    SIGNIFICANCE_UNKNOWN  = 0
    SIGNIFICANCE_NO  = 10
    SIGNIFICANCE_YES = 20
    SIGNIFICANCE_CHOICES = (
            (SIGNIFICANCE_UNKNOWN, 'n/a'),
            (SIGNIFICANCE_NO, 'No'),
            (SIGNIFICANCE_YES, 'Yes'),
            )

    release = models.ForeignKey( 'svnrelease.Svnrelease', related_name = 'releases' )
    issue = models.ForeignKey( Issue, related_name = 'issues' )
    origin = models.IntegerField(choices=ORIGIN_CHOICES)
    context_details = models.CharField( max_length = 200 )
    relevance = models.IntegerField(choices=RELEVANCE_CHOICES, default=RELEVANCE_UNKNOWN)
    significance = models.IntegerField(choices=SIGNIFICANCE_CHOICES, default=SIGNIFICANCE_UNKNOWN)
    class Meta:
        unique_together = (("release", "issue", "origin", "context_details"),)
    def __unicode__(self):
        return "%s - %s - %s - %s" % (self.release, self.issue, self.origin_nickname(), self.context_details)
    def origin_nickname(self):
        return self.get_origin_display()
    def relevance_nickname(self):
        return self.get_relevance_display()
    def significance_nickname(self):
        return self.get_significance_display()


class IssueStatusHistory(models.Model):
    issue = models.ForeignKey( Issue )
    logtime = models.DateTimeField(db_index = True)
    prev_run_time = models.DateTimeField()
    new_status = models.CharField(max_length=60)

    def save(self, *args, **kwargs):
        self.logtime = datetime.now()
        super(IssueStatusHistory, self).save(*args, **kwargs)
    def __unicode__(self):
        return "" % (self.issue_id, self.logtime, self.new_status)

