# -* encoding: utf-8 -*-
from django.http import HttpResponse, Http404, HttpResponseBadRequest
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.db.models import Q
from django.db import connection
from django.core.cache import cache
from django.utils.dateparse import parse_datetime
from django.core.management import call_command
from django.core.urlresolvers import reverse
from django.conf import settings

from releaser.rights.tools import allowed, has_right
from releaser.svnlog.models import *
from releaser.svnlog.tools import get_recent_releases
from releaser.svnrelease.models import *
from releaser.releaseplanning.models import TaskToRelease, TASK_ST_TAG
from releaser.metatracker.models import *
from releaser.metatracker.tools import status_changes, pseudo_status_changes
from releaser.logcmd.models import LogCmd
from releaser.utils import make_display_url
from releaser.startrek.tools import get_startrek_robot_token, get_recent_releases_st
import releaser.common.apps_conf as AppsConf

import math, sys, time, re, hashlib
from startrek_client import Startrek
import dateutil.parser
import datetime
import hashlib
import json
from urllib import urlencode


TIMELINE_CACHE_TIMEOUT = 600

RELEVANT_PROP = "relevant"
SIGNIFICANT_PROP = "significant"
ALL_PROPS = [RELEVANT_PROP, SIGNIFICANT_PROP]

RELEVANT_UNKNOWN = "NA"
RELEVANT_YES = "Yes"
RELEVANT_NO = "No"
SIGNIFICANT_UNKNOWN = "NA"
SIGNIFICANT_YES = "Yes"
SIGNIFICANT_NO = "No"

ALL_VALUES = {
    RELEVANT_PROP: [
        RELEVANT_UNKNOWN,
        RELEVANT_YES,
        RELEVANT_NO
    ],
    SIGNIFICANT_PROP: [
        SIGNIFICANT_UNKNOWN,
        SIGNIFICANT_YES,
        SIGNIFICANT_NO
    ]
}

CHECKSUM_KEY = "checksum"
FORMAT_KEY = "format"
FORMAT_VERSION = "1"
COMMENT_KEY = "comment"
COMMENT_VALUE = u"Данное поле редактируется ТОЛЬКО сервисом metatracker в табуле Директа"
DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S"
RELEASES_LIMIT_NUMBER = 10


def str_problems_dict(problems_dict):
    return json.dumps(problems_dict, indent=2, sort_keys=True, ensure_ascii=False).encode('utf-8')


def count_problems_checksum(problems_dict):
    return hashlib.md5(str_problems_dict(
        {k: v for k, v in problems_dict.iteritems() if k != CHECKSUM_KEY}
    )).hexdigest()


def has_checksum_error(problems_dict):
    return CHECKSUM_KEY in problems_dict and count_problems_checksum(problems_dict) != problems_dict[CHECKSUM_KEY] or \
           CHECKSUM_KEY not in problems_dict and problems_dict


def recent_releases_for_template(recent_releases):
    return [{
        'key': release.key,
        'createdAt': dateutil.parser.parse(release.createdAt),
        'summary': release.summary,
        'status': release.status.display
    } for release in recent_releases]


def selected_release_for_template(selected_release):
    return {
        'key': selected_release.key,
        'summary': selected_release.summary,
        'display_url': make_display_url(selected_release.key)
    }


def process_params(params, startrek, apps_conf):
    release_id = params.get('release_id', "-1")

    if release_id == "-1":
        app = params.get("app", "")
        if not app:
            app = settings.MAIN_APP

        recent_releases = get_recent_releases_st(startrek, apps_conf[app]['tracker-component'], limit=RELEASES_LIMIT_NUMBER)
        selected_release = recent_releases[0]
    else:
        try:
            selected_release = startrek.issues[release_id]
            if selected_release.type.key != "release":
                raise

            app = ""
            for cur_app in apps_conf:
                if apps_conf[cur_app].get("tracker-component", "") in [comp.display for comp in selected_release.components]:
                    app = cur_app
                    break

            if app:
                recent_releases = get_recent_releases_st(startrek, apps_conf[app]['tracker-component'], limit=RELEASES_LIMIT_NUMBER)
            else:
                raise
        except:
            raise Http404("bad release ticket key '%s'" % (release_id))

    return app, release_id, selected_release, recent_releases


@login_required
def index(request):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    apps_conf = AppsConf.get()

    app, release_id, selected_release, recent_releases = process_params(request.REQUEST, startrek, apps_conf)
    issues = {}

    # достаем тикеты из комментариев
    comments = selected_release.comments.get_all()
    issues.update({
        issue: {'origin': 'comment'} for comment in comments for issue in extract_issues_from_text(comment.text)
    })

    # достаем тикеты из описания
    issues.update({
        issue: {'origin': 'description'} for issue in extract_issues_from_text(selected_release.description)
    })

    problems = {}
    checksum_error = False

    if selected_release.testComment:
        problems = json.loads(selected_release.testComment)
        checksum_error = has_checksum_error(problems)

    assignee_dict = {}
    qa_engineer_dict = {}

    for issue in issues.keys():
        st_issue = startrek.issues[issue]

        # Зависимости релизов не показываем
        if '43460' in [ c.id for c in st_issue.components ]:
            del issues[issue]
            continue
        # Другие прилинкованные релизы не показываем
        if st_issue.type.name == 'Release':
            del issues[issue]
            continue

        issues[issue]['summary'] = st_issue.summary
        issues[issue]['display_url'] = "%s/%s" % (settings.STARTREK_DISPLAY_URL, st_issue.key)
        if not st_issue.assignee:
            issues[issue]['assignee'] = 'Unassigned'
        else:
            issues[issue]['assignee'] = st_issue.assignee.id

        if issues[issue]['assignee'] not in assignee_dict:
            assignee_dict[issues[issue]['assignee']] = 0
        assignee_dict[issues[issue]['assignee']] += 1

        if not st_issue.qaEngineer:
            issues[issue]['qa_engineer'] = 'Unknown'
        else:
            issues[issue]['qa_engineer'] = st_issue.qaEngineer.id

        if issues[issue]['qa_engineer'] not in qa_engineer_dict:
            qa_engineer_dict[issues[issue]['qa_engineer']] = 0
        qa_engineer_dict[issues[issue]['qa_engineer']] += 1

        if not checksum_error and issue in problems and RELEVANT_PROP in problems[issue]:
            issues[issue][RELEVANT_PROP] = problems[issue][RELEVANT_PROP]['status']
        else:
            issues[issue][RELEVANT_PROP] = RELEVANT_UNKNOWN

        if not checksum_error and issue in problems and SIGNIFICANT_PROP in problems[issue]:
            issues[issue][SIGNIFICANT_PROP] = problems[issue][SIGNIFICANT_PROP]['status']
        else:
            issues[issue][SIGNIFICANT_PROP] = SIGNIFICANT_UNKNOWN

    issues = sorted(issues.items(), key=lambda x: x[0])
    for pair in issues:
        pair[1]['key'] = pair[0]
    issues = [pair[1] for pair in issues]

    assignee_issues = [(key, assignee_dict[key]) for key in sorted(assignee_dict.keys())]
    qa_engineer_issues = [(key, qa_engineer_dict[key]) for key in sorted(qa_engineer_dict.keys())]

    release_qa = None
    if selected_release.qaEngineer:
        release_qa = selected_release.qaEngineer.id

    # можно отмечать некритичным если есть роль или QA-engineer-у из тикета
    set_properties_allowed = (request.user.username == release_qa) or has_right(
            request.user.username, ['release_manager', 'tabula_superuser', 'release_engineer', 'release_qa']
            )

    return render_to_response(
        'metatracker/index.html',
        {
             'app': app,
             'apps_list': apps_conf.keys(),
             'release': selected_release_for_template(selected_release),
             'release_id': release_id,
             'recent_releases': recent_releases_for_template(recent_releases),
             'issues': issues,
             'assignee_issues': assignee_issues,
             'qa_engineer_issues': qa_engineer_issues,
             'statuses': ['Open', 'Closed', 'Resolved', 'Beta-tested', 'Testing', 'Awaiting-release'],
             'app_statuses': ['not-affected', 'not-tested', 'tested'],
             'show_app_statuses': app in ['direct', 'dna', 'java-web'],
             'checksum_error': checksum_error,
             'set_properties_allowed': set_properties_allowed and not checksum_error,
        },
        context_instance=RequestContext(request)
    )


def extract_issues_from_text(text):
    return re.findall(ur'\b((?:DIRECT|TESTIRT|BALANCE|DIRECTADMIN|TESTBALANCE)-\d+)', text)


def get_prop_name_first(prop):
    return u"%s_first" % prop

def get_prop_name_last(prop):
    return u"%s_last" % prop


@login_required
def updateIssueProps(request):
    # надо бы проверять, что у текущего пользователя есть права на "некритичное"
    # но пока не заморачиваемся, потому что аккуратно -- непросто, а злоупотреблений на самом деле не опасаемся
    try:
        now_dt = datetime.datetime.now()

        release_id = request.REQUEST.get('release_id', '')
        issue_id = request.REQUEST.get('issue', '')
        prop = request.REQUEST.get('prop', '')
        value = request.REQUEST.get('value', '')

        if prop not in ALL_PROPS or value not in ALL_VALUES[prop]:
            raise

        startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
        release = startrek.issues[release_id]

        problems = {}
        if release.testComment:
            problems = json.loads(release.testComment)
            if has_checksum_error(problems):
                return HttpResponse(status=422)

        problems[FORMAT_KEY] = FORMAT_VERSION
        problems[COMMENT_KEY] = COMMENT_VALUE

        if issue_id not in problems:
            problems[issue_id] = {}

        if prop not in problems[issue_id]:
            problems[issue_id][prop] = {}

        data = {
            'author': request.user.username,
            'time': now_dt.strftime(DATETIME_FORMAT)
        }
        if get_prop_name_first(prop) not in problems[issue_id][prop]:
            problems[issue_id][prop][get_prop_name_first(prop)] = data
        problems[issue_id][prop][get_prop_name_last(prop)] = data
        problems[issue_id][prop]['status'] = value
        problems[CHECKSUM_KEY] = count_problems_checksum(problems)

        release.update(testComment=str_problems_dict(problems))
        return HttpResponse()
    except:
        return HttpResponseBadRequest()


def release_timeline_data(release_id):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    apps_conf = AppsConf.get()

    selected_release = startrek.issues[release_id]

    affected_app = None
    for (app, conf) in apps_conf.items():
        if conf.get("tracker-component", "") in [comp.display for comp in selected_release.components]:
            affected_app = conf.get("tracker-affected-app", None)
            break

    issue_keys = extract_issues_from_text(selected_release.description)
    comments = selected_release.comments.get_all()
    issue_keys += [issue_key for comment in comments for issue_key in extract_issues_from_text(comment.text)]

    issues = {}
    for issue_key in issue_keys:
        try:
            if issue_key in issues:
                continue

            st_issue = startrek.issues[issue_key]

            # Зависимости релизов не показываем в таймлайне
            if '43460' in [ c.id for c in st_issue.components ]:
                continue
            # Другие прилинкованные релизы не показываем в таймлайне
            if st_issue.type.name == 'Release':
                continue

            issues[issue_key] = {
                "assignee": st_issue.assignee.id if st_issue.assignee else "Unassigned",
                "summary": st_issue.summary if st_issue.summary else "",
                "mentioned": TASK_ST_TAG in st_issue.tags,
            }
        except:
            pass

    issue_keys = issues.keys()
    # для больших java-приложений, у которых есть специальные поля,
    # историю считаем по "псевдостатусам"
    #FORGIVEME мог бы участвовать apps-conf
    java_testing_field = {
            '31819': 'javaJobsTesting',
            '30820': 'javaIntapiTesting',
            '31053': 'javaApiTesting',
            '34789': 'javaWebTesting',
            }
    # если в компонентах встречаются известные из java_testing_field -- релизу нужны псевдостатусы
    intr = set(java_testing_field.keys()) & set([ c.id for c in selected_release.components ])
    if len(intr) > 0:
        # это приложение с java-testing полем, здесь учитываем и его, и Affected Apps
        field = java_testing_field[ next(iter(intr)) ]
        raw_data = pseudo_status_changes(issue_keys, field, affected_app)
    elif affected_app != None and affected_app not in ['perl', 'dna']:
        # Для perl-Директа и dna пока продолжаем показывать статусы. Поменяем после DIRECT-131678
        # это приложение без java-testing поля но с Affected Apps, здесь смотрим только на Affected Apps
        raw_data = pseudo_status_changes(issue_keys, '', affected_app)
    else:
        # это приложение без java-testing и без Affected Apps, здесь смотрим на статусы тикетов
        raw_data = status_changes(issue_keys)
    # для старых тикетов возвращаются не все смены статусов, подделываем данные для тикетов с документацией
    if 'DIRECT-2997' in raw_data:
        raw_data['DIRECT-2997'] = [{ 'timestamp': 0, 'toString': 'Closed' }]

    # начало и окончание тестирования релиза. Если релиз еще не дотестирован -- окончание устанавливаем в now()
    release_created_timestamp = int(
        time.mktime(dateutil.parser.parse(selected_release.createdAt).timetuple())
    )
    release_changes = status_changes([selected_release.key])[selected_release.key]
    release_end_timestamp = int(time.mktime(datetime.datetime.now().timetuple()))
    marker_type = "Now"
    for ch in release_changes:
        if ch["toString"] == 'Need Acceptance':
            release_end_timestamp = ch['timestamp']
            marker_type = "Need Acceptance"
            break

    release_end_time = int((release_end_timestamp - release_created_timestamp) / 60)

    # считываем данные о проблемах из тикета
    problems = {}
    checksum_error = False

    if selected_release.testComment:
        problems = json.loads(selected_release.testComment)
        checksum_error = has_checksum_error(problems)

    data = {
        "checksum_error": checksum_error,
        "time": time.strftime("%Y-%m-%d %H:%M:%S"),
        "time_span": release_end_time + 40 if release_end_time > 110 else 150,
        "markers": [{
            "time": release_end_time,
            "type": marker_type,
            "title": marker_type,
        }],
        "items": [],
        "scale": [],
    }

    data["scale"] += [{"time": 0, "title": '0'}]
    if data["time_span"] > 120:
        data["scale"] += [{"time": 120, "title": '2h'}]
    if data["time_span"] > 1440:
        data["scale"] += [{"time": 1440, "title": '1d'}]
    if data["time_span"] > 2880:
        data["scale"] += [{"time": 2880, "title": '2d'}]
    if data["time_span"] > 10080:
        data["scale"] += [{"time": 10080, "title": '1w'}]

    data_items = []
    for issue_key in raw_data:
        item = {
            "key": issue_key,
            "login": issues[issue_key]['assignee'],
            "summary": issues[issue_key]['summary'],
            "mentioned": issues[issue_key]['mentioned'],
            "additional": u'',
            "timeline": [],
            "display_url": make_display_url(issue_key)
        }

        unrelevant = (not checksum_error and issue_key in problems and
            RELEVANT_PROP in problems[issue_key] and
            problems[issue_key][RELEVANT_PROP]['status'] == RELEVANT_NO
        )

        unsignificant = (not checksum_error and issue_key in problems and
            SIGNIFICANT_PROP in problems[issue_key] and
            problems[issue_key][SIGNIFICANT_PROP]['status'] == SIGNIFICANT_NO
        )

        significant = (not checksum_error and issue_key in problems and
            SIGNIFICANT_PROP in problems[issue_key] and
            problems[issue_key][SIGNIFICANT_PROP]['status'] == SIGNIFICANT_YES
        )

        # для некритичных и не относящихся к релизу тикетов
        # проставляем окончание времени жизни
        if unsignificant:
            item['additional'] += u" (НЕКРИТИЧНЫЙ)"
            first_change = datetime.datetime.strptime(
                problems[issue_key][SIGNIFICANT_PROP][get_prop_name_first(SIGNIFICANT_PROP)]['time'],
                DATETIME_FORMAT
            )
            item['time_span'] = (int(time.mktime(first_change.timetuple())) - release_created_timestamp) // 60

        if unrelevant and not significant and not unsignificant:
            item['additional'] += u" (НЕ ОТНОСИТСЯ К РЕЛИЗУ)"
            first_change = datetime.datetime.strptime(
                problems[issue_key][RELEVANT_PROP][get_prop_name_first(RELEVANT_PROP)]['time'],
                DATETIME_FORMAT
            )
            item['time_span'] = (int(time.mktime(first_change.timetuple())) - release_created_timestamp) // 60

        for i in raw_data[issue_key]:
            if i['timestamp'] < release_created_timestamp:
                if len(item['timeline']) == 0:
                    item['timeline'] += [{}]
                item['timeline'][0] = {
                    'new_status': i["toString"],
                    'time': 0,
                }
                continue

            item["timeline"] += [{
                "new_status": i["toString"],
                "time": int((i['timestamp'] - release_created_timestamp) / 60)
            }]

        # финальные статусы -- на них история тикета заканчивается
        if not 'time_span' in item and item["timeline"][-1]['new_status'] in ['Closed', 'psJTestingDone', 'psJSkipTesting']:
            item['time_span'] = item["timeline"][-1]['time'] + 20
        item['title'] = "%s - %s - %s%s" % (item['key'], item['login'], item['summary'][0:40], item['additional'])
        data_items.append(item)

    data_items = sorted(data_items, cmp=timeline_item_cmp)
    data['items'] = data_items

    return data


@login_required
def releaseTimelineData(r):
    release_id = r.REQUEST.get('release_id', '')
    if release_id == '':
        return HttpResponseBadRequest('release_id field is missing')

    # проверяем наличие данных в кеше
    cache_key = "%s/releaseTimelineData/%s" % (r.META['HTTP_HOST'], release_id)
    if not r.REQUEST.get('no_cache', None):
        cached_data = cache.get(cache_key)
        if cached_data is not None:
            return HttpResponse(cached_data, mimetype="application/javascript")

    data = release_timeline_data(release_id)

    ret_json = json.dumps(data, sort_keys=True, indent=2)
    cache.set(cache_key, ret_json, TIMELINE_CACHE_TIMEOUT)

    return HttpResponse(ret_json, mimetype="application/javascript")


@login_required
def releaseTimeline(request):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    apps_conf = AppsConf.get()

    app, release_id, selected_release, recent_releases = process_params(request.REQUEST, startrek, apps_conf)

    return render_to_response(
        'metatracker/release_timeline.html',
        {
            'app': app,
            'apps_list': apps_conf.keys(),
            'timeline_base_template': 'base-bootstrap.html',
            'display_timeline': True,
            'release': selected_release_for_template(selected_release),
            'release_id': release_id,
            'recent_releases': recent_releases_for_template(recent_releases),
            'timeline_data_url': reverse(releaseTimelineData) + "?release_id=%s" % (selected_release.key),
        },
        context_instance=RequestContext(request)
    )


@login_required
def index_v1(request):
    issues = Issue.objects.all()

    recent_releases = get_recent_releases(15)

    release_id = -1

    if 'release_id' in request.REQUEST and request.REQUEST.get('release_id'):
        release_id = int(request.REQUEST.get('release_id'))

    if release_id == -1:
        release_id = recent_releases[0].release_id

    if release_id > 0:
        release = SvnRelease.objects.get(pk=release_id)
    else:
        release = SvnRelease()

    # raw_problems -- список проблем с повторениями (одна и та же может быть с разными контекстами)
    if release_id == -2:
        # для еще не созданного релиза составляем список фиктивных проблем,
        # которые потом превратятся в хорошие списки тикетов и коммитов
        release.release_id = -2
        last_release = SvnRelease.objects.get(pk=recent_releases[0].release_id)
        new_trunk_commits=SvnLog.objects.all().filter(rev__gt=last_release.base_rev).filter(svnlogbranch__branch__path='/trunk')
        issues = Issue.objects.all().filter(commits__in=new_trunk_commits).filter(commits__isnull=False)
        raw_problems = [ Problem(release=release, issue = i) for i in issues ]
    else:
        raw_problems = Problem.objects.all().filter(release=release_id).filter(Q(origin=Problem.ORIGIN_DESCRIPTION) | Q(origin=Problem.ORIGIN_HOTFIX) | Q(origin=Problem.ORIGIN_COMMENT)).select_related('issue__assignee', 'issue__qa_engineer', 'issue__summary')

    # надо: список problems без повторений
    # сначала собираем "хеш" (=словарь): {jira_id => $problem}
    problems_hash = {}
    for rp in raw_problems:
        if not rp.issue in problems_hash:
            problems_hash[rp.issue] = rp

    # словарь переделываем в обыкновенный список ( values %problems_hash )
    problems = sorted([ problems_hash[id] for id in problems_hash.keys() ], key=lambda p: p.issue_id)

    assignee_hash = {}
    qa_engineer_hash = {}
    for p in problems:
        #sys.stderr.write( 'issue_id: %s\n' % p.issue )
        commit_objects = p.issue.commits.all().filter(svnlogbranch__branch__path='/trunk')
        p.commits = [ c.rev for c in commit_objects ]
        # Разбираем коммиты на смерженные и несмерженные.
        # TODO со временем отделять коммиты "побывавшие в предыдущем релизе" и новые
        p.commits_current = []
        p.commits_unmerged = []
        for c in commit_objects:
            from_release = c.rev <= release.base_rev
            hotfixed = release.base_rev  in [ r.release_rev.rev for r in c.hotfix_releases_set.all() ]

            if from_release or hotfixed :
                p.commits_current.append(c.rev)
            else:
                p.commits_unmerged.append(c.rev)

        if p.issue.assignee in assignee_hash:
            assignee_hash[p.issue.assignee] += 1
        else:
            assignee_hash[p.issue.assignee] = 1
        if p.issue.qa_engineer == '':
            p.issue.qa_engineer = 'Unknown'
        if p.issue.qa_engineer in qa_engineer_hash:
            qa_engineer_hash[p.issue.qa_engineer] += 1
        else:
            qa_engineer_hash[p.issue.qa_engineer] = 1
        if p.relevance == Problem.RELEVANCE_UNKNOWN:
            p.relevant_class = 'NA'
        else:
            p.relevant_class = p.relevance_nickname

        if p.significance == Problem.SIGNIFICANCE_UNKNOWN:
            p.significant_class = 'NA'
        else:
            p.significant_class = p.significance_nickname

    assignee_issues = [ (a, assignee_hash[a]) for a in sorted(assignee_hash.keys()) ]
    qa_engineer_issues = [ (a, qa_engineer_hash[a]) for a in sorted(qa_engineer_hash.keys()) ]

    response = render_to_response('metatracker/index_v1.html',
            {   'issues': issues,
                'release': release,
                'release_id': release_id,
                'recent_releases': recent_releases,
                'problems': problems,
                'assignee_issues': assignee_issues,
                'qa_engineer_issues': qa_engineer_issues,
                'statuses': ['Open', 'Closed', 'Resolved', 'Beta-tested', 'Testing'],
                'set_properties_allowed': 1 if has_right(request.user.username, ['release_manager', 'tabula_superuser', 'release_engineer', 'release_qa']) else 0,
                },
            context_instance=RequestContext(request)
            )

    #for query in connection.queries:
    #    print query['sql']

    return response

@login_required
@allowed(['release_manager', 'tabula_superuser', 'release_engineer', 'release_qa'])
def updateIssueProps_v1(request):
    release_id = request.REQUEST.get('release_id', '')
    issue_id = request.REQUEST.get('issue', '');
    prop = request.REQUEST.get('prop', '');
    value = request.REQUEST.get('value', '');

    problems = Problem.objects.all().filter(release=release_id, issue=issue_id)

    for problem in problems:
        if prop == 'relevant':
            problem.relevance = Problem.RELEVANCE_NO if value == 'No' else Problem.RELEVANCE_YES if value == 'Yes'else problem.relevance
        else:
            problem.significance = Problem.SIGNIFICANCE_NO if value == 'No' else Problem.SIGNIFICANCE_YES if value == 'Yes'else problem.significance
        problem.save()

    return HttpResponse(json.dumps("OK", sort_keys=True), mimetype="application/javascript")


""" умное сравнение тикетов для сортировки их на таймлайне"""
def timeline_item_cmp(i1, i2):
    # одна задача закончилась, а другая -- еще нет
    if 'time_span' in i1 and not 'time_span' in i2:
        return 1
    # одна задача закончилась, а другая -- еще нет, в другом порядке
    if 'time_span' in i2 and not 'time_span' in i1:
        return -1
    # обе задачи закончились
    if 'time_span' in i1 and 'time_span' in i2:
        return cmp(i2['time_span'], i1['time_span'])
    # обе задачи еще не закончились
    if not 'time_span' in i1 and not 'time_span' in i2:
        # задачи в статусах иных, чем 'Beta-tested', вытаскиваем наверх
        if i1["timeline"][-1]['new_status'] == 'Beta-tested' and i2["timeline"][-1]['new_status'] != 'Beta-tested':
            return 1
        if i2["timeline"][-1]['new_status'] == 'Beta-tested' and i1["timeline"][-1]['new_status'] != 'Beta-tested':
            return -1
        return cmp(i1['timeline'][0]['time'], i2['timeline'][0]['time'])
    return 0


@login_required
def releaseTimelineData_v1(r):
    release_jira_id = r.REQUEST.get('release_jira_id', '')
    if release_jira_id == '':
        raise Exception("missing release_jira_id")

    # проверяем наличие данных в кеше
    cache_key = "%s/releaseTimelineData_v1/%s" % (r.META['HTTP_HOST'], release_jira_id)
    if not r.REQUEST.get('no_cache', None):
        cached_data = cache.get(cache_key)
        if cached_data is not None:
            return HttpResponse(cached_data, mimetype="application/javascript")

    data = release_timeline_data_v1(release_jira_id)

    ret_json = json.dumps(data, sort_keys=True, indent=2)
    cache.set(cache_key, ret_json, TIMELINE_CACHE_TIMEOUT)

    return HttpResponse(ret_json, mimetype="application/javascript")

def release_timeline_data_v1(release_tracker_id):
    release = SvnRelease.objects.get(jira_id=release_tracker_id)
    release_id = release.release_id
    issues = [rp.issue for rp in Problem.objects.all().filter(release=release_id)]
    issue_ids = [i.issue_id for i in issues]
    issue_ids = list(set(issue_ids))

    issue_by_id = {}
    for issue in issues:
        if not issue.issue_id in issue_by_id:
            issue_by_id[issue.issue_id] = issue

    raw_data = status_changes(issue_ids);
    # для старых тикетов возвращаются не все смены статусов, подделываем данные для тикетов с документацией
    if 'DIRECT-2997' in raw_data:
        raw_data['DIRECT-2997'] = [{ 'timestamp': 0, 'toString': 'Closed' }]

    # начало и окончание тестирования релиза. Если релиз еще не дотестирован -- окончание устанавливаем в now()
    release_created_timestamp = int(time.mktime(release.create_time.timetuple()))
    release_changes = status_changes([release_tracker_id])[release_tracker_id];
    release_end_timestamp = int(time.mktime(datetime.datetime.now().timetuple()))
    marker_type = "Now"
    for ch in release_changes:
        if ch["toString"] == 'Need Acceptance':
            release_end_timestamp = ch['timestamp']
            marker_type = "Need Acceptance"
            break

    release_end_time = int((release_end_timestamp-release_created_timestamp)/60)
    data = {
            "time": time.strftime("%Y-%m-%d %H:%M:%S"),
            "time_span": release_end_time + 40 if release_end_time > 110 else 150,
            "markers": [
                {
                    "time": release_end_time,
                    "type": marker_type,
                    "title": marker_type,
                    },
                ],
            "items": [],
            "scale":[],
            }
    data["scale"] += [{"time": 0, "title": '0',}]
    if data["time_span"] > 120:
        data["scale"] += [{"time": 120, "title": '2h',}]
    if data["time_span"] > 1440:
        data["scale"] += [{"time": 1440, "title": '1d',}]
    if data["time_span"] > 2880:
        data["scale"] += [{"time": 2880, "title": '2d',}]
    if data["time_span"] > 10080:
        data["scale"] += [{"time": 10080, "title": '1w',}]

    mentioned_tasks = set([x.jira_issue for x in TaskToRelease.objects.filter(jira_issue__in=raw_data.keys())])

    data_items = []
    for id in raw_data:
        display_url = make_display_url(id)
        item = {
                "jira": id,
                "mentioned": id in mentioned_tasks,
                "login": issue_by_id[id].assignee,
                "summary": issue_by_id[id].summary,
                "additional": u'',
                "timeline": [],
                "display_url": display_url
                }

        unsignificant = Problem.objects.all().filter(release=release_id, issue=id, significance = Problem.SIGNIFICANCE_NO).exists()
        significant = Problem.objects.all().filter(release=release_id, issue=id, significance = Problem.SIGNIFICANCE_YES).exists()
        unrelevant = Problem.objects.all().filter(release=release_id, issue=id, relevance = Problem.RELEVANCE_NO).exists()

        # для некритичных и не относящихся к релизу тикетов
        # проставляем окончание времени жизни
        if unsignificant:
            item['additional'] += u" (НЕКРИТИЧНЫЙ)"
            l = LogCmd.objects.raw('select * from logcmd_logcmd where cmd = "%s" and param rlike "release_id=%s&" and param rlike "issue=%s&" and param rlike "prop=significant" and param rlike "value=No" order by logtime limit 1' % (reverse(updateIssueProps_v1), release_id, id))[0]
            item['time_span'] = int((int(time.mktime(l.logtime.timetuple())) - release_created_timestamp)/60)

        if unrelevant and not significant and not unsignificant:
            item['additional'] += u" (НЕ ОТНОСИТСЯ К РЕЛИЗУ)"
            l = LogCmd.objects.raw('select * from logcmd_logcmd where cmd = "%s" and param rlike "release_id=%s&" and param rlike "issue=%s&" and param rlike "prop=relevant" and param rlike "value=No" order by logtime limit 1' % (reverse(updateIssueProps_v1), release_id, id))[0]
            item['time_span'] = int((int(time.mktime(l.logtime.timetuple())) - release_created_timestamp)/60)

        for i in raw_data[id]:
            if i['timestamp'] < release_created_timestamp:
                if len(item['timeline']) == 0:
                    item['timeline'] += [{}]
                item['timeline'][0] = {
                        'new_status': i["toString"],
                        'time': 0,
                        }
                continue
            item["timeline"] += [{
                "new_status": i["toString"],
                "time": int((i['timestamp']-release_created_timestamp)/60)
                }]
        if not 'time_span' in item and item["timeline"][-1]['new_status'] == 'Closed':
            item['time_span'] = item["timeline"][-1]['time'] + 20
        item['title'] = "%s - %s - %s%s" % (item['jira'], item['login'], item['summary'][0:40], item['additional'])
        data_items.append(item)

    data_items = sorted(data_items, cmp=timeline_item_cmp)
    data['items'] = data_items

    return data


# список "непродуктовых" тикетов в релизе, для ревью
def issuesToReview(r):
    release_jira_id = r.REQUEST.get('release_jira_id', '')
    if release_jira_id == '':
        raise Exception("missing release_jira_id")

    data = release_timeline_data_v1(release_jira_id)
    to_review = []
    for i in data['items']:
        if not i["timeline"][0]["new_status"] in ["Closed", "Resolved"]:
            continue
        to_review.append(i["jira"])

    ret_json = json.dumps(to_review, sort_keys=True, indent=2)

    return HttpResponse(ret_json, mimetype="application/javascript")


@login_required
def timelineDataByST(r):
    duration = r.REQUEST.get('duration', '').strip()
    duration_days = r.REQUEST.get('duration_days', '').strip()
    start_date = r.REQUEST.get('start_date', '').strip()
    base_issue_id = r.REQUEST.get('base_issue_id', '').strip()
    startrek_query = r.REQUEST.get('startrek_query', '').strip()
    base_issue_final_status = r.REQUEST.get('base_issue_final_status', 'Closed').strip()
    linked_issue_final_status = r.REQUEST.get('linked_issue_final_status', 'Tested').strip()

    #base_issue_id = "DIRECT-46908"
    #startrek_query = "Queue: DIRECT created: > 2015-09-29"

    #sys.stderr.write("%s\n\n" % r.REQUEST)

    #if base_issue_id == '':
    #    raise Exception("missing base_issue_id")
    if startrek_query == '':
        raise Exception("missing base_issue_id")

    # проверяем наличие данных в кеше
    cache_key = hashlib.md5(duration + duration_days + start_date + base_issue_id+base_issue_final_status+linked_issue_final_status+startrek_query).hexdigest()
    if not r.REQUEST.get('no_cache', None):
        cached_data = cache.get(cache_key)
        if cached_data is not None:
            return HttpResponse(cached_data, mimetype="application/javascript")

    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    issues = startrek.issues.find(startrek_query)
    if duration == "base_issue":
        base_issue = startrek.issues[base_issue_id]

    issue_by_id = {}
    for issue in issues:
        if not issue.key in issue_by_id:
            issue_by_id[issue.key] = issue
    if duration == "base_issue" and not base_issue.key in issue_by_id:
        issue_by_id[base_issue.key] = base_issue
    issue_ids = issue_by_id.keys()
    issues = None;

    raw_data = status_changes(issue_ids)
    # для старых тикетов возвращаются не все смены статусов, подделываем данные для тикетов с документацией
    if 'DIRECT-2997' in raw_data:
        raw_data['DIRECT-2997'] = [{ 'timestamp': 0, 'toString': 'Closed' }]

    if duration == "base_issue":
        # начало и окончание базового тикета. Если тикет еще не закончен -- окончание устанавливаем в now()
        start_timestamp = int(time.mktime(parse_datetime( base_issue.createdAt ).timetuple()))
        base_issue_changes = status_changes([base_issue_id])[base_issue_id];
        base_issue_end_timestamp = int(time.mktime(datetime.datetime.now().timetuple()))
        marker_type = "Now"
        for ch in base_issue_changes:
            if ch["toString"] == base_issue_final_status:
                base_issue_end_timestamp = ch['timestamp']
                marker_type = "Done"
                break

        end_time = int((base_issue_end_timestamp - start_timestamp)/60)

    if duration == "calendar":
        end_time = int(duration_days) * 1440
        marker_type = duration_days + "d"
        start_timestamp = int(time.mktime(time.strptime(start_date, '%Y-%m-%d')))

    if duration == "linked_issues":
        min_createdAt = min([i.createdAt for i in issue_by_id.values()])
        start_timestamp = int(time.mktime(parse_datetime( min_createdAt ).timetuple()))
        end_timestamp = None
        marker_type = "Now"
        for issue in issue_by_id.values():
            for ch in raw_data[issue.key]:
                issue_end_timestamp = int(time.mktime(datetime.datetime.now().timetuple()))
                if ch["toString"] == linked_issue_final_status:
                    issue_end_timestamp = ch['timestamp']
                    marker_type = "Done"
                    break
            if end_timestamp == None:
                # первый закончившийся тикет
                end_timestamp = issue_end_timestamp
            elif issue_end_timestamp > end_timestamp:
                # очередной тикет законичился позже, чем все предыдущие
                end_timestamp = issue_end_timestamp

        if end_timestamp == None:
            end_timestamp = int(time.mktime(datetime.datetime.now().timetuple()))
        end_time = int((end_timestamp - start_timestamp)/60)

    timeline_data = {
            "time": time.strftime("%Y-%m-%d %H:%M:%S"),
            "time_span": end_time + 40 if end_time > 110 else 150,
            "markers": [
                {
                    "time": end_time,
                    "type": marker_type,
                    "title": marker_type,
                    },
                ],
            "items": [],
            "scale":[],
            }
    timeline_data["scale"] += [{"time": 0, "title": '0',}]
    if timeline_data["time_span"] > 120 and timeline_data["time_span"] <= 2*1440:
        timeline_data["scale"] += [{"time": 120, "title": '2h',}]
    if timeline_data["time_span"] > 1440 and timeline_data["time_span"] <= 60*1440:
        timeline_data["scale"] += [{"time": 1440, "title": '1d',}]
    if timeline_data["time_span"] > 2880 and timeline_data["time_span"] <= 7*1440:
        timeline_data["scale"] += [{"time": 2880, "title": '2d',}]
    if timeline_data["time_span"] > 7*1440 and timeline_data["time_span"] <= 90*1440:
        timeline_data["scale"] += [{"time": 5*1440, "title": '5d',}]
    if timeline_data["time_span"] > 10080:
        for i in range(1, 300):
            mark_time = i*7*1440
            if mark_time >= timeline_data["time_span"]:
                break
            timeline_data["scale"] += [{"time": mark_time, "title": str(i)+'w',}]

    # TODO
    mentioned_tasks = set([]) #set([x.jira_issue for x in TaskToRelease.objects.filter(jira_issue__in=raw_data.keys())])

    for id in raw_data:
        display_url = make_display_url(id)
        assignee_id = None
        try:
            assignee_id = issue_by_id[id].assignee.id
        except:
            pass
        item = {
                "jira": id,
                "mentioned": id in mentioned_tasks,
                "login": assignee_id,
                "summary": issue_by_id[id].summary,
                "additional": u'',
                "timeline": [],
                "display_url": display_url
                }

        # TODO
        unsignificant = False #Problem.objects.all().filter(release=release_id, issue=id, significance = Problem.SIGNIFICANCE_NO).exists()
        significant = False #Problem.objects.all().filter(release=release_id, issue=id, significance = Problem.SIGNIFICANCE_YES).exists()
        unrelevant = False #Problem.objects.all().filter(release=release_id, issue=id, relevance = Problem.RELEVANCE_NO).exists()

        # для некритичных и не относящихся к релизу тикетов
        # проставляем окончание времени жизни
        if unsignificant:
            item['additional'] += u" (НЕКРИТИЧНЫЙ)"
            #l = LogCmd.objects.raw('select * from logcmd_logcmd where cmd = "/metatracker/updateIssueProps" and param rlike "release_id=%s&" and param rlike "issue=%s&" and param rlike "prop=significant" and param rlike "value=No" order by logtime limit 1' % (release_id, id))[0]
            #item['time_span'] = int((int(time.mktime(l.logtime.timetuple())) - release_created_timestamp)/60)

        if unrelevant and not significant and not unsignificant:
            item['additional'] += u" (НЕ ОТНОСИТСЯ К РЕЛИЗУ)"
            #l = LogCmd.objects.raw('select * from logcmd_logcmd where cmd = "/metatracker/updateIssueProps" and param rlike "release_id=%s&" and param rlike "issue=%s&" and param rlike "prop=relevant" and param rlike "value=No" order by logtime limit 1' % (release_id, id))[0]
            #item['time_span'] = int((int(time.mktime(l.logtime.timetuple())) - release_created_timestamp)/60)

        for i in raw_data[id]:
            if i['timestamp'] < start_timestamp:
                if len(item['timeline']) == 0:
                    item['timeline'] += [{}]
                item['timeline'][0] = {
                        'new_status': i["toString"],
                        'time': 0,
                        }
                continue
            item["timeline"] += [{
                "new_status": i["toString"],
                "time": int((i['timestamp']-start_timestamp)/60)
                }]
        if not 'time_span' in item and item["timeline"][-1]['new_status'] in [ linked_issue_final_status, "Closed" ]:
            item['time_span'] = item["timeline"][-1]['time'] + 20
        if 'time_span' in item and item['time_span'] > timeline_data["time_span"]:
            item['time_span'] = timeline_data["time_span"]
        item['title'] = "%s - %s - %s%s" % (item['jira'], item['login'], item['summary'][0:40], item['additional'])
        timeline_data['items'].append(item)

    timeline_data['items'] = sorted(timeline_data['items'], cmp=timeline_item_cmp)
    # базовый тикет вытаскиваем вперед
    head = []
    tail = []
    for i in timeline_data['items']:
        if i['jira'] == base_issue_id:
            head += [i]
        else:
            tail += [i]
    timeline_data['items'] = head + tail

    ret_json = json.dumps(timeline_data, sort_keys=True, indent=2)
    cache.set(cache_key, ret_json, TIMELINE_CACHE_TIMEOUT)

    return HttpResponse(ret_json, mimetype="application/javascript")


@login_required
def releaseTimeline_v1(r):
    recent_releases = get_recent_releases(15)
    release_id = -1
    if 'release_id' in r.REQUEST and r.REQUEST.get('release_id', ''):
        release_id = int(r.REQUEST.get('release_id'))
    if release_id == -1:
        release_id = recent_releases[0].release_id

    release_jira_id = r.REQUEST.get('release_jira_id', '')
    if release_jira_id != '':
        release = SvnRelease.objects.get(jira_id=release_jira_id)
        release_id = release.release_id
    else:
        release = SvnRelease.objects.get(pk=release_id)
        release_jira_id = release.jira_id

    return render_to_response('metatracker/release_timeline_v1.html',
            {
                'timeline_base_template': 'base-bootstrap.html',
                'display_timeline': True,
                'release_id': release_id,
                'release_jira_id': release_jira_id,
                'release': release,
                'recent_releases': recent_releases,
                'timeline_data_url': reverse(releaseTimelineData_v1) +"?release_jira_id=%s" % (release_jira_id),
                },
            context_instance=RequestContext(r)
            )


@login_required
def timelineByST(r):
    show = r.REQUEST.get('show', '').strip()
    duration = r.REQUEST.get('duration', '').strip()
    duration_days = r.REQUEST.get('duration_days', '').strip()
    start_date = r.REQUEST.get('start_date', '').strip()
    base_issue_id = r.REQUEST.get('base_issue_id', '').strip()
    startrek_query = r.REQUEST.get('startrek_query', '').strip()
    base_issue_final_status = r.REQUEST.get('base_issue_final_status', 'Closed').strip()
    linked_issue_final_status = r.REQUEST.get('linked_issue_final_status', 'Tested').strip()

    return render_to_response('metatracker/timeline_by_st.html',
            {
                'timeline_base_template': 'base-navbar-united.html' if settings.TABULA_UNITED else 'base-navbar-project.html',
                'start_date': start_date,
                'duration': duration,
                'duration_days': duration_days,
                'base_issue_id': base_issue_id,
                'startrek_query': startrek_query,
                'base_issue_final_status': base_issue_final_status,
                'linked_issue_final_status': linked_issue_final_status,
                'display_timeline': show != "" and show != "0",
                'timeline_data_url': reverse(timelineDataByST) + "?" + urlencode({
                    'start_date': start_date,
                    'duration': duration,
                    'duration_days': duration_days,
                    'base_issue_id': base_issue_id,
                    'base_issue_id': base_issue_id,
                    'base_issue_final_status': base_issue_final_status,
                    'linked_issue_final_status': linked_issue_final_status,
                    'startrek_query': startrek_query,
                    })
                },
            context_instance=RequestContext(r)
            )


@login_required
@allowed(['tabula_superuser', 'hotfix_accepter'])
def refresh_deployed_tickets(request):
    try:
        call_command('update_tracker_on_deploy')
        if settings.PROJECT == 'javadirect':
            call_command('update_tracker_on_deploy_dna')
    except:
        return HttpResponse(status=429)

    return HttpResponse('')

