# -*- encoding: utf-8 -*-
# django-admin.py process_releaseplans --settings=releaser.settings.direct --pythonpath=../

from __future__ import with_statement

from django.core.management.base import BaseCommand
from django.db import transaction, models
from django.core.files import locks

from pprint import pprint as p
import tempfile, re, string, os, time, sys, urllib2
from datetime import datetime
from optparse import make_option
from contextlib import contextmanager

import copy

from django.db.models import Q

from suds.transport.http import HttpTransport

from django.conf import settings

from releaser.svnlog.models import *
from releaser.svnrelease.models import *
from releaser.metatracker.models import Issue
from releaser.versionica.models import HostGroup
from releaser.versionica.installed_versions import has_commit

from releaser.releaseplanning.models import TaskToRelease
from releaser.releaseplanning.tools  import current_week, time_appropriate_for

@contextmanager

def locked_file(filename):
    full_filename = "%s/%s" % (settings.LOCKS_PATH, filename)
    f = open(full_filename, 'wb')
    locks.lock(f, locks.LOCK_EX|locks.LOCK_NB)
    try:
        yield
    finally:
        locks.unlock(f)
        f.close()
        os.unlink(full_filename)


class Command(BaseCommand):
    help = "Autoprocessing of release plans"
    debug = True

    option_list = BaseCommand.option_list + (
        make_option('--limit', action='store', dest='limit', type='int',
            default=10, help='Number of releases to fetch'),
        make_option('--debug', action='store_true', dest='debug',
            default=False, help='Print steps of command processing'),
    )


    def dprint(self, *args):
        if self.debug:
            print ", ".join(map(lambda x: unicode(x), args))


    @transaction.commit_on_success
    def process_new_commits(self, cw):
        # Для каждой задачи из текущей недели/любого будущего:
        # если раньше не было коммита И теперь появился
        # записать коммит в базу,
        # задаче выставить статус "committed"
        # Если к тому же задача была из будущего -- type = 'сверх плана', week = current_week()
        present_and_future_tasks = TaskToRelease.objects.filter(week__gte=cw, commit=-1)
        for task in present_and_future_tasks:
            try:
                issue_details = Issue.objects.filter(issue_id=task.jira_issue)[0]
            except IndexError:
                continue
            commits = issue_details.commits.filter(svnlogbranch__branch__path='/trunk')
            if len(commits) > 0:
                task.commit = commits[0].rev
                if task.status != TaskToRelease.STATUS_RELEASED:
                    task.status = TaskToRelease.STATUS_COMMITTED
                if task.week > cw:
                    task.week = cw
                    task.type = TaskToRelease.TYPE_ABOVE_PLAN
                task.save()
        # Если есть задачи из будущего, но к которым привязан коммит -- переносим на текущую неделю
        future_commited_tasks = TaskToRelease.objects.filter(week__gt=cw).filter(~Q(commit = -1))
        for task in future_commited_tasks:
            if TaskToRelease.objects.filter(jira_issue = task.jira_issue, week = cw).exists():
                continue
            task.week = cw
            task.status = TaskToRelease.STATUS_COMMITTED
            task.type = TaskToRelease.TYPE_ABOVE_PLAN
            task.save()
        return


    @transaction.commit_on_success
    def update_status_released(self, cw):
        # Для каждой задачи из текущей недели в статусе release|committed
        # В продакшене?
        #   + статус = released
        #   - статус = committed
        prod = HostGroup.objects.get(name='production')
        current_committed_tasks = TaskToRelease.objects.filter(week=cw, commit__gt=0).filter(Q(status=TaskToRelease.STATUS_RELEASED)|Q(status=TaskToRelease.STATUS_COMMITTED))
        for task in current_committed_tasks:
            #sys.stderr.write("%s\n" % task)
            if has_commit(prod, task.commit):
                task.status = TaskToRelease.STATUS_RELEASED
            else:
                task.status = TaskToRelease.STATUS_COMMITTED
            task.save()
        return


    @transaction.commit_on_success
    def update_status_notdone(self, cw):
        # Если конец недели
        # Для каждой задачи текущей недели
        # * в статусе none -- status = notdone
        #   + копия "в будущем" (если таковой еще нет)
        # * в статусе committed
        #   копия "на сл. неделе" (если таковой еще нет) status = committed
        #   если копия есть в любом будущем -- перенести на сл. неделю со статусом "закоммичено"
        current_tasks = TaskToRelease.objects.filter(week=cw).filter(~Q(status=TaskToRelease.STATUS_RELEASED))
        for task in current_tasks:
            future_copy, created = TaskToRelease.objects.get_or_create(
                jira_issue=task.jira_issue,
                week__gt=cw,
                defaults = { 'author': task.author }
                )
            if created:
                id = future_copy.id
                future_copy = copy.copy(task)
                future_copy.id = id
                future_copy.type = TaskToRelease.TYPE_ANY

            if task.status == TaskToRelease.STATUS_COMMITTED:
                future_copy.week = cw + 1
                future_copy.status = TaskToRelease.STATUS_COMMITTED
            else:
                task.status = TaskToRelease.STATUS_NOT_DONE
                task.save()
                future_copy.week = TaskToRelease.FUTURE_WEEK
                future_copy.status = TaskToRelease.STATUS_NONE
            future_copy.save()
        return


    def handle(self, **options):
        self.debug = options["debug"]
        with locked_file('releaseplans'):
            cw = current_week()

            self.process_new_commits(cw)

            self.update_status_released(cw)

            if time_appropriate_for('summarize_week'):
                self.update_status_notdone(cw)

            return

