# -* encoding: utf-8 -*-
from django.http import HttpResponse, Http404
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.db.models import Q
from django.shortcuts import redirect
import math
import sys
#import datetime
from datetime import datetime, timedelta, date
from time import time
import re

from django.conf import settings

from releaser.users.models import User
from releaser.metatracker.models import Issue

from releaser.releaseplanning.models import TaskToRelease, Membership, TASK_ST_TAG
from releaser.releaseplanning.tools import current_week, week_title, tasks_waiting_for_release, title_by_task, time_appropriate_for, change_nice
from releaser.releaseplanning.tasks import get_recent_tasks
from releaser.versionica.models import HostGroup
from releaser.ptera.models import NotifyOnDeploy
import releaser.ptera.notifications
from releaser.utils import sendmail

from releaser.rights.tools import has_right, allowed, exclusive_holder
from releaser.gap.tools import get_gaps

from startrek_client import Startrek
from releaser.startrek.tools import get_startrek_robot_token

@login_required
def index(r):
    vars = {
            'current_path': r.get_full_path(),
            'main_jira_project': settings.TRACKER_QUEUE,
            'indented_tasklist': 1,
            'may_edit_comments': has_right(r.user.username, 'release_manager'),
            'may_change_niceness': has_right(r.user.username, ['release_manager', 'tabula_superuser']),
    }
    tasks_info = get_recent_tasks(login=r.user.username, add_available_actions=True)
    vars = dict(tasks_info.items() + vars.items())

    vars['notifications'] = releaser.ptera.notifications.user_notifications_by_issues(r.user.username)
    vars['gap'] = get_gaps()

    return render_to_response('releaseplanning/index.html',
            vars,
            context_instance=RequestContext(r)
            )


@login_required
def edit(r):
    id = r.REQUEST.get('id', '');

    task = TaskToRelease.objects.get(id=id)
    if len(task.title.strip()) == 0:
        task.title = title_by_task(task.jira_issue)

    return render_to_response('releaseplanning/edit_task.html',
            {
                'task': task
            },
            context_instance=RequestContext(r)
            )

@login_required
def add(r):
    # TODO С вечера пятницы до утра понедельника -- запретить
    jira_issue = r.REQUEST.get('jira_issue', '');
    m = re.search(settings.ISSUE_KEY_REGEXP, jira_issue)
    jira_issue = m.group(0)

    task = get_or_create_task(author=r.user.username, issue_key=jira_issue)
    return redirect('/releaseplanning/#%s' % task.id)


@login_required
def mention(r):
    rev = r.REQUEST.get('rev', '');

    jira_issues = Issue.objects.filter(commits__in=[rev])

    if len(jira_issues) == 0:
        return HttpResponse("Can't find in DB correspondent Jira issues, please try later.", mimetype="text/plain; charset=utf-8")

    for issue in jira_issues:
        get_or_create_task(author=r.user.username, rev=rev, issue_key=issue.issue_id)

    path = r.REQUEST.get('retpath', '/svnlog');
    return redirect(path)


def nice_above_max(w):
    try:
        nice = TaskToRelease.objects.filter(week = w).order_by('-niceness')[0].niceness + 10
    except IndexError:
        nice = 0
    return nice


def get_or_create_task(**O):
    if not 'author' in O:
        raise Exception("author undefined")
    if not 'issue_key' in O:
        raise Exception("issue key undefined")

    cw = current_week()

    try:
        task = TaskToRelease.objects.all().filter(jira_issue=O['issue_key']).filter(week__gte=cw)[0]
    except IndexError:
        title = title_by_task(O['issue_key'])[:100]
        author, _ = User.objects.get_or_create(login=O['author'])
        task = TaskToRelease(
            jira_issue=O['issue_key'],
            title=title,
            author=author,
            week = TaskToRelease.FUTURE_WEEK,
            niceness = nice_above_max(TaskToRelease.FUTURE_WEEK),
            )
        if 'rev' in O:
            task.commit = O['rev']
        task.save()
        startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.STARTREK_CLIENT_USER_AGENT)
        startrek.issues[O['issue_key']].update(tags=startrek.issues[O['issue_key']].tags + [TASK_ST_TAG])
        issue = startrek.issues[O['issue_key']]
        assignee = issue.assignee.id
        reporter = issue.createdBy.id
        try:
            task.add_members([assignee, reporter])
        except Exception, e:
            pass

    return task


@login_required
def save(r):
    id = r.REQUEST.get('id', '').strip()
    title = r.REQUEST.get('title', '').strip()
    comment = r.REQUEST.get('comment', '').strip()
    members_str = r.REQUEST.get('members', '').strip()

    member_logins = re.split(u'[ ,:;]+', members_str)
    member_logins = list(set(member_logins))

    task = TaskToRelease.objects.get(id=id)
    task.title = title
    if 'comment' in r.REQUEST:
        task.comment = comment
    task.save()

    task.add_members(member_logins)

    return redirect('/releaseplanning/#%s' % task.id)


def titles_for_release(r):
    titles = "\n".join(tasks_waiting_for_release())
    return HttpResponse(titles, mimetype="text/plain; charset=utf-8")


@login_required
def move(r):
    cweek = current_week()
    id = r.REQUEST.get('id', '').strip()
    week_from = int(r.REQUEST.get('week_from', '').strip())
    week_to = int(r.REQUEST.get('week_to', '').strip())

    if week_to != cweek + 1 and week_to != TaskToRelease.FUTURE_WEEK and week_to != TaskToRelease.FAR_FUTURE_WEEK:
        raise Exception("WrongWeekNumber")

    if week_from != cweek + 1 and week_from != TaskToRelease.FUTURE_WEEK and week_from != TaskToRelease.FAR_FUTURE_WEEK:
        raise Exception("WrongWeekNumber")

    if (week_from == cweek or week_to == cweek) and not time_appropriate_for('change_next_week_tasks'):
        raise Exception("InappropriateTime")

    task = TaskToRelease.objects.get(pk=id)
    if task.week != week_from:
        raise Exception("Inconsistent week_to parameter")

    existed_tasks = TaskToRelease.objects.filter(jira_issue=task.jira_issue, week=week_to)

    if len(existed_tasks) > 0:
        raise Exception("IssueAlreadyExists")

    if week_to == TaskToRelease.FUTURE_WEEK:
        task.niceness=nice_above_max(week_to)

    task.week = week_to

    task.save()

    return redirect('/releaseplanning/#%s' % task.id)


@login_required
def delete(r):
    id = r.REQUEST.get('id', '').strip()
    cweek = current_week()

    task = TaskToRelease.objects.get(pk=id)

    if task.author.login != r.user.username:
        raise "PermissionDenied"

    appr_time = task.week > cweek + 1 or task.week == cweek + 1 and time_appropriate_for('change_next_week_tasks')
    if not appr_time:
        raise Exception("CantDelete")

    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.STARTREK_CLIENT_USER_AGENT)
    startrek.issues[task.jira_issue].update(tags=[tag for tag in startrek.issues[task.jira_issue].tags if tag != TASK_ST_TAG])
    task.delete()

    return redirect('releaser.releaseplanning.views.index')


@login_required
@allowed(['release_manager', 'tabula_superuser'])
def nice(r):
    cweek = current_week()
    id = r.REQUEST.get('id', '').strip()
    to = r.REQUEST.get('to', '').strip()

    cweek = current_week()
    task = TaskToRelease.objects.get(pk=id)

    if task.week <= cweek:
        raise Exception("Task should be in future")

    if to == 'up':
        change_nice(task, 1)
    elif to == 'down':
        change_nice(task, -1)
    elif to == 'top':
        change_nice(task, 1000)
    elif to == 'bottom':
        change_nice(task, -1000)
    else:
        raise Exception("Unknown instruction '%s'" % to)

    return redirect('/releaseplanning/#%s' % task.id)


@login_required
@allowed(['release_engineer', 'tabula_superuser'])
def release_warning(r):
    when = r.REQUEST.get('when', 'tomorrow').strip()
    RE_login = exclusive_holder('release_engineer').login
    if when == "tomorrow":
        if date.weekday(date.today()) == 4:   # пятница
            when_str = u'в понедельник'
        else:
            when_str = u'завтра'
    elif when == 'tonight':
        when_str = u'сегодня вечером'
    else:
        raise Exception("Unknown time '%s'" % when)

    subject = u'%s: релиз %s' % (settings.NAME, when_str)
    body = u'''Всем привет.

%s ожидается релиз, пожалуйста, учитывайте его в своих планах.

Предполагаемые задачи и уже состоявшиеся коммиты можно смотреть в Табуле.


Это письмо отправлено автоматически.
Основание: в Табуле нажата кнопка "Скоро релиз".
Вопросы задавайте, пожалуйста, релиз-инженеру (%s).


Всего хорошего!
''' % ( when_str.capitalize(), RE_login )
    sendmail(
            "%s@yandex-team.ru" % (r.user.username),
            settings.RELEASE_WARNING_EMAIL,
            subject.encode('utf-8'),
            body.encode('utf-8'),
            cc=None
            )
    return redirect('/releaseplanning/release_warning_sent')


@login_required
@allowed(['release_engineer', 'tabula_superuser'])
def release_warning_sent(r):
    message = u"Вы нажали кнопку 'Скоро релиз'. Сообщение отправлено на %s" % (settings.RELEASE_WARNING_EMAIL)
    return HttpResponse(message, mimetype="text/plain; charset=utf-8")


