# -* encoding: utf-8 -*-
from django.db.models import Q
import math
import sys
import re

from django.conf import settings
from releaser.rights.models import Right, HasRight
from releaser.users.models import User

from functools import wraps
from django.http import HttpResponseForbidden


def exclusive_holder(right_name, **kwargs):
    """
    Для исключительного права right_name (строка)
    возвращает пользователя, который этим правом обладает.

    Если право не исключительное -- выбрасывает исключение,
    если нашлось больше одного пользователя -- исключение,
    если нет такого пользователя -- исключение,
    если нет такого права -- исключение.

    Параметры именованные
    default -- если не нашелся пользователь или право,
        то вместо исключения будет возвращено это значение
    """

    try:
        right = Right.objects.get(right=right_name)
    except Right.DoesNotExist:
        if 'default' in kwargs:
            return kwargs['default']
        else:
            raise Exception("there is no right '%s'" % right_name)

    if not right.exclusive:
        raise Exception("'%s' isn't an exclusive right" % right_name)

    right_records = HasRight.objects.filter(right=right)

    users = [r.user for r in right_records]
    if len(users) > 1:
        raise Exception("there are more than one user with right '%s'" % right_name)

    if len(users) == 0:
        if 'default' in kwargs:
            return kwargs['default']
        else:
            raise Exception("there is no user with right '%s'" % right_name)

    return users[0]


def has_right(login, rights):
    """
    Имеет ли указанный пользователь указанное право
    """
    if type(rights) != list:
        rights = [rights]

    try:
        user = User.objects.get(login = login)
    except User.DoesNotExist:
        return False

    allowed = HasRight.objects.filter(user=user, right__right__in = rights)
    #sys.stderr.write("user: %s, rights: %s, allowed: %s \n" % (user, rights, allowed))

    return len(allowed) > 0


TEAM_RIGHTS = ['developer', 'release_manager', 'manager', 'test_engineer', 'admin', 'tabula_superuser', 'subscriber']


def user_rights(login):
    """
    Словарь всех прав пользователя
    """
    try:
        user = User.objects.get(login = login)
    except User.DoesNotExist:
        return []

    rights = {}
    for rec in HasRight.objects.filter(user=user):
        rights[rec.right.right] = True

    return rights


def allowed(rights):
    """
    Декоратор для проверки прав.
    Обязательный параметр -- список (массив) прав.
    Если пользователь обладает хотя бы одним из перечисленных прав -- OK,
    иначе получает страницу "Нет прав".
    Имеет смысл использовать вместе с декоратором @login_required
    """
    def inner_allowed(fn):
        def wrapped(r, *args, **kwargs):
            login = r.user.username
            sys.stderr.write("login: %s, rights: %s\n" % (login, rights))
            if not has_right(login, rights):
                # TODO красивую страничку со ссылкой
                return HttpResponseForbidden(u'<h1>Нет прав</h1>К сожалению, у вас нет прав для выполнения этой операции.<br>Если вы считаете, что права нужны -- обратитесь к любому из разработчиков, он может выдать права.')
            return fn(r, *args, **kwargs)
        return wraps(fn)(wrapped)
    return inner_allowed

