# -*- coding:utf-8 -*-

from django.core.management.base import BaseCommand
from django.db import models
from django.conf import settings

from releaser.yambclient.client import YambClient
from releaser.svnlog.models import SvnLog
from releaser.svnlog.tools import get_new_release_base
from releaser.startrek.tools import get_startrek_robot_token, get_sign_comment
from releaser.statistics.models import ReleaseStatistics
from startrek_client import Startrek
from releaser.statistics.views import format_time, STATISTICS_TAG, get_new_releases

import os
import json
import datetime
import urllib2
import dateutil.parser
import re
from collections import defaultdict
import pytz
from optparse import make_option
import copy


class Command(BaseCommand):
    help = "Get statistics for releases"

    option_list = BaseCommand.option_list + (
        make_option(
            "-l",
            "--live",
            dest = "live",
            action = 'store_true',
            default = False,
            help = "получить 'live' статистику по открытым релизам",
        ),
    )

    NOT_FINISHED = False

    startrek_client = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)

    @classmethod
    def get_release_testing_times(self, ticket=None):
        changes = self.startrek_client.issues[ticket.key].changelog.get_all(sort='asc', field='status')

        start = None
        end = None

        for change in changes:
            for field in change.fields:
                if not start and field['field'].id == 'status' and field['to'].key == 'readyForTest':
                    start = dateutil.parser.parse(change.updatedAt)

                if not end and field['field'].id == 'status' and field['from'] and field['from'].key == 'testing':
                    end = dateutil.parser.parse(change.updatedAt)

        return start, end


    def get_closing_date(self, ticket=None, testing_end=None):
        result = {'ticket': ticket.key}
        changes = self.startrek_client.issues[ticket.key].changelog.get_all(sort='asc', field='status')

        for change in changes:
           for field in change.fields:
                if field['field'].id == 'status' and field['to'].key == 'closed':
                    result['value'] = dateutil.parser.parse(change.updatedAt)
                    return result

        if self.NOT_FINISHED:
            result['value'] = testing_end
            result['not_finished'] = True
            return result

        return None


    def count_delta(self, start=None, end=None):
        value = int((end - start).total_seconds())

        if (start.weekday() < 5 and start.weekday() > end.weekday()) or (end - start).days > 5:
            value -= 2 * 24 * 60 * 60
        elif end.weekday() > 4:
            value -= int((end - end.replace(day=end.day - end.weekday() % 5, hour=0, minute=0, second=0)).total_seconds())

        return max(value, 0)


    @classmethod
    def count_testing_duration(self, ticket=None, metric=None, testing_start=None, testing_end=None):
        result = {metric: {'value': self.count_delta(testing_start, testing_end)}}

        if self.NOT_FINISHED:
            result[metric]['not_finished'] = True

        return result


    def get_regression_tickets(self, ticket=None):
        comments = self.startrek_client.issues[ticket.key].comments.get_all()

        result = []

        for comment in comments:
            if re.search(ur'Созданы тикеты на регрессию:', comment.text, re.U): # comment.createdBy.id == 'direct-handles'
                result.extend(re.findall(ur'(?:DIRECT|TESTIRT)-[0-9]+', comment.text))

        return list(set(result))


    @classmethod
    def count_regression_tickets_lifetime(self, ticket=None, metric=None, testing_start=None, testing_end=None):
        summary2name = {
                           u'INTAPI,': '_intapi',
                           u'API:': '_api',
                           u'cmd': '_cmd',
                           u'веб': '_web',
                           u'DNA': '_dna',
                       }

        result = defaultdict(dict)

        comments = self.startrek_client.issues[ticket.key].comments.get_all()

        reg_tickets = self.get_regression_tickets(ticket)

        for reg_ticket in reg_tickets:
            issue = self.startrek_client.issues[reg_ticket]

            for summary in summary2name:
                if re.search(summary, issue.summary, re.U):
                    if not issue.resolvedAt:
                        value = self.count_delta(testing_start, testing_end)
                        if self.NOT_FINISHED:
                            result[metric + summary2name[summary]]['not_finished'] = True
                    else:
                        value = self.count_delta(testing_start, min(testing_end, dateutil.parser.parse(issue.resolvedAt)))

                    result[metric + summary2name[summary]]['value'] = max(result[metric + summary2name[summary]]['value'] \
                                                                          if (metric + summary2name[summary]) in result and 'value' in result[metric + summary2name[summary]] else 0,
                                                                          value)
                    result[metric + summary2name[summary]]['comment'] = reg_ticket
                    break

        return result


    @classmethod
    def count_tickets_closed_time(self, ticket=None, metric=None, testing_start=None, testing_end=None):
        metrics = ['tasks_closed_time', 'tasks_closed_time_net', 'bugs_closed_time']
        tickets_to_save = 3
        value = None

        tickets_to_check = re.findall(ur'^(DIRECT-[0-9]+):', ticket.description, re.I | re.U | re.M)

        comments = self.startrek_client.issues[ticket.key].comments.get_all()

        for comment in comments:
            if comment.createdBy.id != 'direct-handles' and dateutil.parser.parse(comment.createdAt) < testing_end:
                tickets_to_check.extend(re.findall(ur'DIRECT-[0-9]+', comment.text, re.I | re.U | re.M))

        tickets_to_check = list(set(tickets_to_check))

        result = {}
        res_tickets = {}
        reg_tickets = set(self.get_regression_tickets(ticket))

        for metric in metrics:
            res_tickets[metric] = []

        for ticket_to_check in tickets_to_check:
            try:
                issue = self.startrek_client.issues[ticket_to_check]
            except:
                continue

            close_dt = self.get_closing_date(issue, testing_end)

            for metric in metrics:
                if issue.summary.startswith(u'Ревью непродуктовых коммитов') or not close_dt or \
                   ((metric == 'tasks_closed_time' or metric == 'tasks_closed_time_net') and dateutil.parser.parse(issue.createdAt) > testing_start) or \
                   (metric == 'bugs_closed_time' and dateutil.parser.parse(issue.createdAt) < testing_start) or \
                   (metric == 'tasks_closed_time_net' and u'limited_by_release_regression' in issue.tags) or \
                   close_dt['value'] > testing_end or \
                   ticket_to_check in reg_tickets:
                    continue

                res_tickets[metric].append(close_dt)

        for metric in metrics:
            res_tickets[metric] = sorted(res_tickets[metric], key=lambda x: x['value'])

            if len(res_tickets[metric]) > tickets_to_save:
                res_tickets[metric] = res_tickets[metric][-tickets_to_save:]

            if not res_tickets[metric]:
                result.update({metric: {"value": 0, "comment": ""}})
            else:
                value = self.count_delta(testing_start, res_tickets[metric][-1]['value'])
                new_el = {metric: {"value": value, "comment": ", ".join([res_ticket['ticket'] for res_ticket in res_tickets[metric][::-1]])}}
                if 'not_finished' in res_tickets[metric][-1]:
                    new_el[metric]['not_finished'] = True

                result.update(new_el)

        return result


    def get_correct_values(self, raw_values, ticket, start_testing):
        correct_values = {}

        start_testing_utc = self.dt_utc_local(start_testing)
        if start_testing_utc.hour < 8:
            next_morning = start_testing_utc.replace(hour=8, minute=0, second=0)
            today_night = start_testing_utc
        else:
            next_morning = (start_testing_utc + datetime.timedelta(days=1)).replace(hour=8, minute=0, second=0)
            today_night = max(start_testing_utc, start_testing_utc.replace(hour=20, minute=0, second=0))

        value_to_sub = int((next_morning - today_night).total_seconds())

        skip_correct = raw_values['testing_duration']['value'] > 30 * 60 * 60 or (start_testing_utc.hour >= 8 and start_testing_utc.hour < 16)

        for metric in raw_values:
            new_metric = metric + u'__minus_night'
            correct_values[new_metric] = copy.deepcopy(raw_values[metric])

            if skip_correct:
                continue

            correct_values[new_metric]['value'] = max(0, correct_values[new_metric]['value'] - value_to_sub)


        return correct_values


    def dt_utc_local(self, dt):
        return dt.replace(tzinfo=pytz.utc).astimezone(pytz.timezone('Europe/Moscow'))


    def comment_regression_tickets(self, result, start_testing, release):
        dt_format = "%Y-%m-%d %H:%M:%S"

        regression_metric_prefix = "regression_tickets_lifetime_"

        for metric in result:
            if metric.startswith(regression_metric_prefix) and metric.find('__minus_night') == -1:
                comment = u"Подсчитана статистика по релизу %s.\n" % release
                comment += u"Начало релиза: %s\n" % self.dt_utc_local(start_testing).strftime(dt_format)
                comment += u"Окончание регрессии %s: %s\n" % (metric[len(regression_metric_prefix):], self.dt_utc_local(dateutil.parser.parse(self.startrek_client.issues[result[metric]['comment']].resolvedAt)).strftime(dt_format))
                comment += u"Длительность: %s\n" % format_time(result[metric]['value'], with_seconds=False)
                comment += u"Длительность с ночной коррекцией: %s\n" % format_time(result[metric + "__minus_night"]['value'], with_seconds=False)

                if settings.PROJECT == 'javadirect':
                    comment += u"\nПолная статистика: https://javadirect-dev.yandex-team.ru/statistics/releases_table"
                else:
                    comment += u"\nПолная статистика: https://direct-dev.yandex-team.ru/statistics/releases_table"

                self.startrek_client.issues[result[metric]['comment']].comments.create(text=comment + u"\n" + get_sign_comment(os.path.basename(__file__)))


    metrics_dict = {
                       'testing_duration': count_testing_duration.__func__,
                       'regression_tickets_lifetime': count_regression_tickets_lifetime.__func__,
                       'tasks_closed_time': count_tickets_closed_time.__func__,
                   }


    def handle(self, *args, **options):
        LIVE = bool(options['live'])
        releases = list(get_new_releases(self.startrek_client, live=LIVE))
        full_result = {}

        for release in releases:
            testing_start, testing_end = self.get_release_testing_times(release)

            if LIVE:
                if not testing_end:
                    testing_end = datetime.datetime.utcnow().replace(tzinfo=pytz.utc)
                    self.NOT_FINISHED = True
                else:
                    continue

            if not testing_start or not testing_end:
                continue

            result = {}
            for metric in self.metrics_dict:
                result.update(self.metrics_dict[metric](self=self, ticket=release, metric=metric, testing_start=testing_start, testing_end=testing_end))

            result.update(self.get_correct_values(result, release, testing_start))

            if os.environ.get('COMMENT_REGRESSIONS'):
                self.comment_regression_tickets(result, testing_start, release.key)

            if not result:
                continue

            for key in result:
                record, _ = ReleaseStatistics.objects.get_or_create(
                    ticket = release.key,
                    metric_name = key,
                    defaults = {
                        'value': result[key]['value'],
                        'comment': result[key]['comment'] if 'comment' in result[key] else "",
                    },
                )

                record.value = result[key]['value']
                record.comment = result[key]['comment'] if 'comment' in result[key] else ""
                record.save()

            if not LIVE:
                self.startrek_client.issues[release.key].update(tags=self.startrek_client.issues[release.key].tags + [STATISTICS_TAG])

        return
