# -* encoding: utf-8 -*-
import json
from django.http import HttpResponse, Http404
from django.shortcuts import render_to_response, redirect
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.db.models import Q, Min, Count
from django.db import connection
from django.core.cache import cache
from django.core.management import call_command
from django.utils.dateparse import parse_datetime
import math, sys, os, random, time, string, re
from datetime import date, datetime, timedelta
from socket import getaddrinfo, getnameinfo
from urllib import urlencode
from collections import defaultdict
import dateutil.parser
import yaml

from django.conf import settings
from releaser.versionica.models import *
from releaser.versionica.installed_versions import check_commits
from releaser.releaseplanning.models import *
from releaser.svnlog.models import SvnLog
from releaser.statistics.models import ReleaseStatistics, ReleaseStatisticsComments
from releaser.svnrelease.models import SvnRelease

from startrek_client import Startrek
from releaser.startrek.tools import startrek_status_key_to_str, get_startrek_robot_token, last_releases_query
import releaser.common.apps_conf as AppsConf

from releaser.rights.tools import allowed, has_right
from io import BytesIO

TESTING_TIME_CACHE_TIMEOUT = 600

CORRECTED_NO = ['','0', 'no']

charts = {
        'cifront_fail_duration': {
            'template':'statistics/chart.html',
            'data_url': '/statistics/cifront_fail_duration',
            'title': u'длительность зафейленных тестов',
            'subtitle': u'(в часах)',
            'y_axis_text': u'время, часы',
            },
        'hotfixes': {
            'template':'statistics/chart.html',
            'data_url': '/statistics/hotfixes-data',
            'title': u'количество хотфиксов',
            'subtitle': u'(штуки)',
            'y_axis_text': u'хотфиксы',
            },
        'release_delay':{
            'template':'statistics/chart.html',
            'data_url': '/statistics/release-delay-data',
            'title': u'Время от коммита до выкладки',
            'subtitle': u'Квантили, в днях',
            'y_axis_text': 'дни',
            },
        'release_events':{
            'template':'statistics/chart.html',
            'data_url': '/statistics/release-events-data',
            'title': u'Релизы (релизные события / 3)',
            'subtitle': u'cкользящая сумма за N дней',
            'y_axis_text': 'релизные события, шт.',
            },
        }

# в секундах
time2color = (
    (4 * 60 * 60, "#00884f"),
    (6 * 60 * 60, "#00ff4d"),
    (8 * 60 * 60, "#ceff4d"),
    (12 * 60 * 60, "#ffff52"),
    (18 * 60 * 60, "#ffe652"),
    (24 * 60 * 60, "#ffc452"),
    (36 * 60 * 60, "#ff8585"),
    (10000 * 60 * 60, "#ff0000"),
)

STATISTICS_TAG = u'statistics_counted'


@login_required
def index(r):
    global charts
    return render_to_response('statistics/index.html',
            {
                'request': r,
                'charts': charts.keys(),
            },
            context_instance=RequestContext(r)
            )

@login_required
def release_age_chart(r):
    return render_to_response('statistics/release_age_chart.html',
            {
                'request': r,
            },
            context_instance=RequestContext(r)
            )


@login_required
def add_review_comment(r):
    ticket = r.REQUEST.get('ticket', '')
    metric = r.REQUEST.get('metric', '')
    comment = r.REQUEST.get('comment', '')

    if metric.endswith('__minus_night'):
        metric = metric[:metric.find('__minus_night')]

    for cur_metric in [metric, metric + u"__minus_night"]:
        try:
            stats = ReleaseStatistics.objects.get(ticket=ticket, metric_name=cur_metric)
        except:
            stats = None

        if not ticket or not cur_metric  or not stats:
            return HttpResponse(json.dumps({'error': 'bad parameters'}), mimetype="application/javascript")

        # значит хотим удалить из базы
        if not comment:
            ReleaseStatisticsComments.objects.filter(release_stats=stats).delete()
            continue

        record, created = ReleaseStatisticsComments.objects.get_or_create(
            release_stats=stats,
            defaults = {
                'review_comment': comment,
            },
        )

        if not created:
            record.review_comment = comment
            record.save()

    return HttpResponse(json.dumps({}), mimetype="application/javascript")


@login_required
def releases_pie_chart(r):
    app = r.REQUEST.get('app', 'all')
    apps_list, extra_query_args = process_app_param(app)

    return render_to_response('statistics/releases_pie_chart.html',
            {
                'request': r,
                'form': {
                    'block_size': str(max(1, int(r.REQUEST.get('block_size', '')))) if r.REQUEST.get('block_size', '').isdigit() else "5",
                    'number': str(max(3, int(r.REQUEST.get('number', '')))) if r.REQUEST.get('number', '').isdigit() else "3",
                    'show_corrected': r.REQUEST.get('show_corrected', '').strip() not in CORRECTED_NO,
                    'app': app,
                },
                'apps_list': apps_list,
                'project': settings.PROJECT,
            },
            context_instance=RequestContext(r)
            )


def release_statistics_pie_chart_data(r):
    block_size = int(r.REQUEST.get('block_size', 5))
    number = int(r.REQUEST.get('number', 3))
    show_corrected = r.REQUEST.get('show_corrected', '').strip() not in CORRECTED_NO
    app = r.REQUEST.get('app', 'all')

    apps_list, extra_query_args = process_app_param(app)

    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)

    releases = [ticket.key for ticket in get_releases(startrek, extra_query_args)][:number * block_size]
    metrics = get_metrics(show_corrected)
    rows = ReleaseStatistics.objects.filter(ticket__in=releases, metric_name__in=metrics).values()

    data_dict = defaultdict(dict)

    for row in rows:
        data_dict[row['ticket']][row['metric_name']] = row['value']

    data = {'charts': [], 'colors': [el[1] for el in time2color]}

    for metric in metrics:
        el = {'name': u"%s" % metric, 'blocks': []}

        for ind in xrange(0, len(releases), block_size):
            block = [{'name': color_name(time2color[i][0], with_seconds=False), 'y': 0} for i in xrange(len(time2color))]
            for j in xrange(ind, min(ind + block_size, len(releases))):
                if metric not in data_dict[releases[j]]:
                    data_dict[releases[j]][metric] = 0

                for k in xrange(0, len(time2color)):
                    if data_dict[releases[j]][metric] <= time2color[k][0]:
                        block[k]['y'] += 1
                        break

            el['blocks'].append({'values': block, 'title': u"%s -- %s" % (releases[min(ind + block_size - 1, len(releases) - 1)], releases[ind])})

        data['charts'].append(el)

    json_data = json.dumps(data, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


@login_required
def releases_chart(r):
    app = r.REQUEST.get('app', 'all')
    apps_list, extra_query_args = process_app_param(app)

    return render_to_response('statistics/releases_chart.html',
            {
                'request': r,
                'form': {'number': r.REQUEST.get('number', '') if r.REQUEST.get('number', '').isdigit() else "40",
                         'show_corrected': r.REQUEST.get('show_corrected', '').strip() not in CORRECTED_NO,
                         'app': app,
                        },
                'apps_list': apps_list,
                'project': settings.PROJECT,
            },
            context_instance=RequestContext(r)
            )


def release_statistics_data(r):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)

    show_corrected = r.REQUEST.get('show_corrected', '').strip() not in CORRECTED_NO
    app = r.REQUEST.get('app', 'all')

    apps_list, extra_query_args = process_app_param(app)

    releases = [ticket.key for ticket in get_releases(startrek, extra_query_args)][:int(r.REQUEST.get('number', 40))]
    metrics = get_metrics(show_corrected)
    rows = ReleaseStatistics.objects.filter(ticket__in=releases, metric_name__in=metrics).values()

    data_dict = defaultdict(dict)

    for row in rows:
        data_dict[row['ticket']][row['metric_name']] = row['value']

    data = {'tickets': ["%s (%s)" % (release, dateutil.parser.parse(startrek.issues[release].createdAt).strftime("%Y-%m-%d")) for release in releases], 'series': []}

    for metric in metrics:
        data['series'].append({'name': metric, 'data': [data_dict[release][metric] if release in data_dict and metric in data_dict[release] else 0 for release in releases]})

    json_data = json.dumps(data, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


def get_info_data_for_releases_table(name, my_cmp, data, metrics):
    data_info = {'aggregator': name, 'values': [0 for i in xrange(len(metrics))]}

    for row in data:
        ind = 0

        # последняя колонка всегда должна быть "длительностью тестирования"
        for i in xrange(len(row['metrics']) - 1):
            if my_cmp(row['metrics'][i]['raw_value'], row['metrics'][ind]['raw_value']):
                ind = i

        data_info['values'][ind] += 1

    return data_info


def format_time(seconds, with_seconds=True):
    days = seconds / (60 * 60 * 24)
    hours = (seconds % (60 * 60 * 24)) / (60 * 60)
    if with_seconds:
        return "%dd %dh (%ds)" % (days, hours, seconds)
    else:
        return "%dd %dh" % (days, hours)


def color_name(time, with_seconds=True):
    return "< %s" % format_time(time, with_seconds) if time != time2color[-1][0] else "> %s" % format_time(time2color[-2][0], with_seconds)


def get_metrics(show_corrected=None):
    if show_corrected is None:
        return [metric['metric_name'] for metric in ReleaseStatistics.objects.order_by('metric_name').values('metric_name').distinct()]
    elif show_corrected:
        return [metric['metric_name'] for metric in ReleaseStatistics.objects.filter(metric_name__contains="__minus_night").order_by('metric_name').values('metric_name').distinct()]
    else:
        return [metric['metric_name'] for metric in ReleaseStatistics.objects.exclude(metric_name__contains="__minus_night").order_by('metric_name').values('metric_name').distinct()]


def get_new_releases(startrek, extra=u"", live=False):
    if live:
        if settings.PROJECT == 'javadirect':
            return startrek.issues.find(
                u'Queue: DIRECT Type: Release Status: !Closed Tags: !"%s" Components: "Releases: JavaDirect" %s "Sort by": key desc' % (
                    STATISTICS_TAG, extra
                )
            )
        else:
            return startrek.issues.find(
                u'Queue: DIRECT Type: Release Status: !Closed Tags: !"%s" Components: "Releases: Direct" %s "Sort by": key desc' % (
                    STATISTICS_TAG, extra
                )
            )
    else:
        if settings.PROJECT == 'javadirect':
            return startrek.issues.find(
                u'Queue: DIRECT Type: Release Status: Closed, "Need Acceptance" Tags: !"%s" Components: "Releases: JavaDirect" Created: >=2017-01-01 %s "Sort by": key desc' % (
                    STATISTICS_TAG, extra
                )
            )
        else:
            return startrek.issues.find(
                u'Queue: DIRECT Type: Release Status: Closed, "Need Acceptance" Tags: !"%s" Components: "Releases: Direct" Created: >=2017-01-01 %s "Sort by": key desc' % (
                    STATISTICS_TAG, extra
                )
            )


def get_releases(startrek, extra=u""):
    if settings.PROJECT == 'javadirect':
        return startrek.issues.find(u'Queue: DIRECT Type: Release Tags: "%s" Components: "Releases: JavaDirect" Created: >=2017-01-01 %s "Sort by": key desc' % (STATISTICS_TAG, extra))
    else:
        return startrek.issues.find(u'Queue: DIRECT Type: Release Tags: "%s" Components: "Releases: Direct" Created: >=2017-01-01 %s "Sort by": key desc' % (STATISTICS_TAG, extra))


def process_app_param(app):
    extra_query_args = u""
    apps_list = ["all"]

    if settings.PROJECT == 'direct':
        app = "all"
    else:
        apps_conf = AppsConf.get()
        if app in apps_conf:
            extra_query_args = 'Components: "%s"' % apps_conf[app]['tracker-component']
        else:
            app = 'all'
        apps_list.extend(apps_conf.keys())

    return apps_list, extra_query_args


def get_data_from_db(releases, metrics):
    rows = ReleaseStatistics.objects \
                            .select_related('releasestatisticscomments__review_comment') \
                            .filter(ticket__in=[release.key for release in releases], metric_name__in=metrics) \
                            .values('ticket', 'value', 'metric_name', 'comment', 'releasestatisticscomments__review_comment')

    data_dict = defaultdict(dict)

    for row in rows:
        data_dict[row['ticket']][row['metric_name']] = {
            'value': row['value'],
            'comment': re.sub(
                ur'((?:TESTIRT|DIRECT)-[0-9]+)', ur'<a target="_blank" href="https://st.yandex-team.ru/\1">\1</a>',
                row['comment']
            ),
            'review_comment': row['releasestatisticscomments__review_comment'] if row['releasestatisticscomments__review_comment'] else u""
        }

    data = []

    for release in releases:
        row = {'ticket_name': release.key,
               'ticket_summary': release.summary,
               'ticket_closed_time': dateutil.parser.parse(release.createdAt).strftime("%Y-%m-%d"),
               'metrics': [],
              }

        for metric in metrics:
            new_value = {'value': 0, 'raw_value': 0, 'metric': metric}
            if metric in data_dict[release.key]:
                new_value = data_dict[release.key][metric]
                new_value['raw_value'] = new_value['value']
                new_value['value'] = format_time(new_value['value'])
                new_value['metric'] = metric

            row['metrics'].append(new_value)

        for cell in row['metrics']:
            for t in time2color:
                if cell['raw_value'] <= t[0]:
                    cell['color'] = t[1]
                    break

        data.append(row)

    return data


@login_required
def releases_table(r):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)

    number = int(r.REQUEST.get('number', '')) if r.REQUEST.get('number', '').isdigit() else 40
    show_corrected = r.REQUEST.get('show_corrected', '').strip() not in CORRECTED_NO
    app = r.REQUEST.get('app', 'all')

    apps_list, extra_query_args = process_app_param(app)

    releases = list(get_releases(startrek, extra_query_args))[:number]
    metrics = get_metrics(show_corrected)
    data = get_data_from_db(releases, metrics)

    return render_to_response('statistics/releases_table.html', {'metrics': [metric.replace("_", " ") for metric in metrics],
                                                                 'data': data,
                                                                 'info_data': [get_info_data_for_releases_table(u"Сколько раз закончилось первым", lambda x, y: x < y, data, metrics),
                                                                               get_info_data_for_releases_table(u"Сколько раз закончилось последним", lambda x, y: x > y, data, metrics)],
                                                                 'form': {'number': str(number), 'show_corrected': show_corrected, 'app': app},
                                                                 'colors_table': [(color_name(time), color) for time, color in time2color],
                                                                 'apps_list': apps_list,
                                                                 'project': settings.PROJECT,
                                                                }, context_instance=RequestContext(r))


@login_required
def current_release(r):
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)

    # пересчитываем значения
    call_command('get_release_statistics', live=True)
    releases = list(get_new_releases(startrek, live=True))
    metrics = get_metrics(None)
    data = get_data_from_db(releases, metrics)

    # обеъединяем обычные метрики с соответствующими им minus_night
    for ticket in data:
        merged_metrics = {}

        for metric in ticket['metrics']:
            minus_night = False
            metric_short = metric['metric']
            if metric['metric'].endswith(u"__minus_night"):
                minus_night = True
                metric_short = metric['metric'][:-len("__minus_night")]

            if metric_short not in merged_metrics:
                merged_metrics[metric_short] = [{} for i in xrange(2)]

            index = 1 if minus_night else 0
            merged_metrics[metric_short][index] = metric

        ticket['metrics'] = [merged_metrics[metric] for metric in metrics if not metric.endswith(u"__minus_night")]

    return render_to_response('statistics/current_release.html', {'column_names': ['Ticket', 'Metric', 'Value', 'Night corrected value'],
                                                                  'colors_table': [(color_name(time), color) for time, color in time2color],
                                                                  'data': data,
                                                                 }, context_instance=RequestContext(r))


def production_updates(project):
    ignore = {}
    if project == 'direct':
        ignore = {
                77555: 1,
                110676: 1,
                237969: 1,
                }
        log = HostPropertyLog.objects.filter(host__name='ppcback01f.yandex.ru').filter(property__name="yandex-direct")
    elif settings.PROJECT == 'directmod':
        log = HostPropertyLog.objects.filter(host__name='ppcmod01e.yandex.ru').filter(property__name="direct-moderate")

    log = [ l for l in log if not l.id in ignore ]
    return log

def release_age_data(r):

    log = production_updates(settings.PROJECT)

    releases = []
    prev_mv = -1
    prev_sv = -1
    for rec in log:
        if not re.match(r'^[~a-z0-9\.\-]+$', rec.value):
            continue

        m = re.match(r'^1\.(\d+).(\d{2,6})?', rec.value)
        if m:
            mv = int(m.groups()[0] or 0)
            sv = int(m.groups()[1] or 0)
        else:
            raise Exception("can't parse version %s" % rec.value)

        if settings.PROJECT == 'directmod' and mv < 2500:
            continue

        if mv < prev_mv:
            # откатывались на более раннюю версию
            continue
        elif mv == prev_mv:
            # хотфикс через мерж
            prev_sv = sv
            continue
        elif prev_sv == 0:
            # svn up + мерж: считаем, что хотфикс.
            # На самом деле здесь теряем часть релизов (если релиз до конца выкладывался из транка)
            # Чтобы не терять, надо брать историю версий из Стартрека
            prev_sv = sv
            prev_mv = mv
            continue
        else:
            releases += [rec]
            prev_sv = sv
            prev_mv = mv


    series = []
    for k in [1,5,15]:
        data = []
        for i in range( k, len(releases) ):
            c = releases[i]
            p = releases[i-k]
            td = c.logtime - p.logtime
            delta = td.seconds + td.days * 86400
            data += [[
                    [c.logtime.year, c.logtime.month -1, c.logtime.day],
                    '%.1f' % (delta / 86400.0 / k),

                    ]]
        series += [
                {
                    'data': data,
                    'name': (u"среднее по %s релизам" % k),
                    },
                ]

    print series
    json_data = json.dumps(series, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


@login_required
def release_testing_time_chart(r):
    return render_to_response('statistics/release_testing_time_chart.html',
            {
                'request': r,
            },
            context_instance=RequestContext(r)
            )


def release_testing_time_data_raw():
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    if settings.PROJECT == 'direct':
        query = last_releases_query(component=settings.STARTREK_DIRECT_RELEASE_COMPONENT)
    else:
        query = last_releases_query()

    release_generator = startrek.issues.find(query)

    stat = []
    for r in release_generator:
        if len(stat) > 30000:
            break
        s = {
            'key': r.key,
            'created': r.createdAt,
            }
        raw_changelog = startrek.issues[r.key].changelog.get_all(fields=('status'))
        for entry in raw_changelog:
            if not hasattr(entry, 'fields'):
                continue

            try:
                status_change = next((field for field in entry['fields'] if field['field'].id == 'status'))
            except StopIteration:
                continue

            new_status = startrek_status_key_to_str(status_change['to'].key)
            if new_status == "Need Acceptance":
                s['tested'] = entry.updatedAt
                break

        if not 'tested' in s:
            #sys.stderr.write("unknown: %s\n" % s)
            continue

        s['created'] = re.sub(r'\d{3}\+0000$', '000+0400', s['created'])
        s['created_timestamp'] = int(time.mktime(time.strptime( s['created'], '%Y-%m-%dT%H:%M:%S.000+0400')))
        s['tested'] = re.sub(r'\d{3}\+0000$', '000+0400', s['tested'])
        s['tested_timestamp'] = int(time.mktime(time.strptime( s['tested'], '%Y-%m-%dT%H:%M:%S.000+0400')))
        s['testing_time_seconds'] = s['tested_timestamp'] - s['created_timestamp']

        if s['testing_time_seconds'] > 60 * 86400:
            continue

        if settings.PROJECT == 'directmod' and s['created'] < "2013-01-01":
            continue

        stat += [s]

    stat = sorted(stat, key=lambda k: k['created_timestamp'])

    return stat


@login_required
def release_testing_time_data(r):
    stat = release_testing_time_data_raw()

    series = []
    for k in [1, 5, 15]:
        data = []
        for i in range( k, len(stat) ):
            testing_time = sum([ stat[i - j]['testing_time_seconds'] for j in range(0, k) ])
            data += [[
                    stat[i]['created_timestamp'],
                    '%.1f' % (testing_time / 86400.0 / k),
                    ]]
        series += [
                {
                    'data': data,
                    'name': (u"чистое время тестирования по %s релизам" % k),
                    },
                ]

    json_data = json.dumps(series, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


@login_required
def release_testing_time_data_plaintext(r):
    # проверяем наличие данных в кеше
    cache_key = "%s/release_testing_time_data_plaintext" % (r.META['HTTP_HOST'])
    if not r.REQUEST.get('no_cache', None):
        cached_data = cache.get(cache_key)
        if cached_data is not None:
            return HttpResponse(cached_data, content_type="text/plain; charset=utf-8")
    stat = release_testing_time_data_raw()

    ret_text = "".join([
        " ".join([
            '%.1f' % (s['testing_time_seconds'] / 86400.0),
            s['key'],
            s['created'],
            ]) + "\n"
        for s in reversed(stat)
        ])

    cache.set(cache_key, ret_text, TESTING_TIME_CACHE_TIMEOUT)
    return HttpResponse(ret_text, content_type="text/plain; charset=utf-8")


@login_required
def cifront_fail_duration(r):
    cursor = connection.cursor()
    cursor.execute("select max(logtime) as logtime, max(if(status!=10, 1, 0)) as failed from cifront_buildbotsinglebuild sb join cifront_buildbotbranch b on b.id = sb.branch_id where b.branch = 'trunk' group by rev order by logtime")
    builds = cursor.fetchall()

    stat = []
    prev_t = builds[0][0]
    prev_failed = builds[0][1]
    for b in builds:
        if b[1] == prev_failed:
            continue
        if b[1] == 0:
            delta = b[0] - prev_t
            delta_seconds = delta.seconds + delta.days * 86400
            stat += [
                    {
                        'timestamp': b[0].strftime('%s'),
                        'fail_duration': delta_seconds,
                        'comment': ("%s" % b[0])
                        }
                    ]

        prev_failed = b[1]
        prev_t = b[0]

    series = []
    for k in [1, 5, 15]:
        data = []
        for i in range( k, len(stat) ):
            fail_duration = sum([ stat[i - j]['fail_duration'] for j in range(0, k) ])
            data += [[
                    stat[i]['timestamp'],
                    '%.1f' % (fail_duration / 3600.0 / k),
                    ]]
        series += [
                {
                    'data': data,
                    'name': (u"среднее по %s фейлам" % k),
                    },
                ]
    json_data = json.dumps(series, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")


def date_range(start, end, delta):
    curr = start
    while curr <= end:
        yield curr
        curr += delta


@login_required
def hotfixes_data(r):
    cursor = connection.cursor()

    hotfixes= {}

    cursor.execute("select date(createtime), count(*) as cnt from hotfixing_hotfixrequest where urgency > 20 group by date(createtime) order by createtime")
    hotfixes['ts'] = list(cursor.fetchall())

    cursor.execute("select date(createtime), count(*) as cnt from hotfixing_hotfixrequest where urgency <= 20 group by date(createtime) order by createtime")
    hotfixes['prod'] = list(cursor.fetchall())

    cursor.execute("select date(revtime), count(*) from svnlog_svnloghotfix h join svnlog_svnlog l on l.rev = h.hotfix_rev where revtime > '2012-01-10' group by 1")
    hotfixes['total'] = list(cursor.fetchall())

    start_date = hotfixes['total'][0][0]
    end_date   = hotfixes['total'][-1][0]

    hotfix_types = ['ts', 'prod', 'total',]

    hotfixes_dict = {}
    for t in hotfix_types:
        hotfixes_dict[t] = dict(hotfixes[t])

    for t in hotfix_types:
        hotfixes[t] = []

    for d in date_range(start_date, end_date, timedelta(days=1)):
        for t in hotfix_types:
            try:
                cnt = hotfixes_dict[t][d]
            except KeyError:
                cnt = 0
            hotfixes[t] += [[d, cnt]]

    series = []
    for t in hotfix_types:
        for k in [1, 7, 30]:
            data = []
            for i in range( k, len(hotfixes[t]) ):
                cnt = sum([ hotfixes[t][i - j][1] for j in range(0, k) ])
                data += [[
                        hotfixes[t][i][0].strftime('%s'),
                        '%.2f' % (float(cnt) / k),
                        ]]
            series += [
                    {
                        'data': data,
                        'name': (u"%s: среднее по %s дням" % (t, k)),
                        },
                    ]

    json_data = json.dumps(series, sort_keys=True)
    return HttpResponse(json_data, mimetype="application/javascript")

@login_required
def chart(r):
    global charts
    chart_name = r.GET.get('chart_name', 'default').strip();
    data_url = charts[chart_name]['data_url']
    if settings.PROJECT == 'javadirect':
        app = r.GET.get('app', 'java-web').strip();
        project_title = "%s / %s" % (settings.PROJECT, app)
        data_url = "%s?app=%s" % (data_url, app)
    else:
        project_title = "%s" % (settings.PROJECT)

    return render_to_response('statistics/chart.html',
            {
                'request': r,
                'chart': charts[chart_name],
                'project_title': project_title,
                'data_url': data_url,
            },
            context_instance=RequestContext(r)
            )


def release_delay_data(r):
    prod = HostGroup.objects.get(name='production')
    # звездные задачи
    #tasks = TaskToRelease.objects.filter(status = 7)
    #commits = [ t.commit for t in tasks ]
    #commits = sorted(commits)

    # все коммиты
    if settings.PROJECT == 'direct':
        min_rev = 30000
    elif settings.PROJECT == 'directmod':
        min_rev = 2500
    else:
        min_rev = 1000
    commit_obects=SvnLog.objects.all().filter(rev__gt=min_rev).filter(svnlogbranch__branch__path='/trunk')
    commits = sorted( [ c.rev for c in commit_obects ] )

    delays = {}
    for c in commits:
        # когда сделан коммит
        delays[c] = {
                "commit_time": SvnLog.objects.get(rev = c).revtime
                }

    releases = production_updates(settings.PROJECT)
    for rl in releases:
        try:
            commits_existence = check_commits(prod, commits, rl.value)
            commits_deployed = [c for c in commits if commits_existence[c]]

            for c in commits_deployed:
                if not 'release_time' in delays[c] or rl.logtime < delays[c]['release_time']:
                    # когда случился релиз
                    delays[c]['release_time'] = rl.logtime
        except:
            continue

    for c in delays.keys():
        if not "release_time" in delays[c]:
            del delays[c]

    delays_arr = []
    for c in sorted(delays.keys(), key=lambda c: delays[c]["release_time"]):
        delta = (delays[c]["release_time"] - delays[c]["commit_time"]).total_seconds() / 86400.0
        delays[c]["delta"] = delta
        delays_arr += [ delays[c] ]

    if r.GET.get('raw', '') == '1':
        text = "".join([ "%s %.1f %d\n" % ( delays[rev]["release_time"].strftime('%Y-%m-%d %H:%M:%S'), delays[rev]['delta'], rev ) for rev in sorted(delays.keys())])
        return HttpResponse(text, content_type="application/javascript")

    series = []
    # считаем медиану по k коммитам. TODO: по N дням (30)
    k = 300
    for ts in [20, 50, 80, 90, 95, 98]:
        data = []
        for i in range( k, len(delays_arr) ):
            c = delays_arr[i]
            n = int(k*ts/100)
            q = sorted([delays_arr[i-j]["delta"] for j in range(0, k) ])[n]
            data += [[
                c["release_time"].strftime('%s'),
                '%.1f' % q ,
                ]]
            #sys.stderr.write("%s\n" % ( c["release_time"] ))
        series += [
                {
                    'data': data,
                    'name': (u"%s-квантиль от коммита до выкладки, дни" % ts),
                    },
                ]

    json_data = json.dumps(series, sort_keys=True)
    return HttpResponse(json_data, content_type="application/javascript")


def get_release_events_by_date(start_date = "2020-01-01", app="java-web"):
    apps_conf = AppsConf.get()
    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    if settings.PROJECT == 'direct':
        query = last_releases_query(component=settings.STARTREK_DIRECT_RELEASE_COMPONENT)
    else:
        component = apps_conf[app]['tracker-component']
        query = last_releases_query(component=component)

    release_generator = startrek.issues.find(query)

    releases = []
    for r in release_generator:
        if len(releases) > 3000:
            break
        s = {
            'key': r.key,
            'created': r.createdAt,
            }

        if settings.PROJECT == 'direct' and s['created'] < "2012-01-01":
            continue
        if settings.PROJECT == 'directmod' and s['created'] < "2013-01-01":
            continue
        if s['created'] < start_date:
            continue

        raw_changelog = startrek.issues[r.key].changelog.get_all(fields=('status'))
        # ищем первые переходы в Need Acceptance и Ready to deploy
        for entry in raw_changelog:
            if not hasattr(entry, 'fields'):
                continue

            try:
                status_change = next((field for field in entry['fields'] if field['field'].id == 'status'))
            except StopIteration:
                continue

            new_status = startrek_status_key_to_str(status_change['to'].key)
            if new_status == "RM Acceptance" and not "tested" in s:
                s['tested'] = entry.updatedAt
            if new_status == "Ready to deploy" and not "predeployed" in s:
                s['predeployed'] = entry.updatedAt
            if "tested" in s and "predeployed" in s:
                break

        releases += [s]

    events_by_date = {}
    for rl in releases:
        for event in ['created', 'tested', 'predeployed']:
            if not event in rl:
                continue
            event_date = parse_datetime( rl[event] ).date()
            event_tuple = [rl['key'], event, rl[event]]
            if not event_date in events_by_date:
                events_by_date[ event_date ] = []

            events_by_date[ event_date ] += [ event_tuple ]

    #for d in sorted( events_by_date.keys() ):
    #    sys.stderr.write("%s: %s\n" % (d, events_by_date[d]))
    return events_by_date


def release_events_data(r):
    app = r.REQUEST.get('app', 'java-web')
    events_by_date = get_release_events_by_date(app=app)

    start_date = min( events_by_date.keys() )
    end_date   = max( events_by_date.keys() )

    events_stat_by_date = []
    for d in date_range(start_date, end_date, timedelta(days=1)):
        try:
            cnt = len(events_by_date[d]) / 3.0
        except KeyError:
            cnt = 0
        events_stat_by_date += [[d, cnt]]

    series = []
    for k in [1, 7, 28]:
        data = []
        for i in range( k, len(events_stat_by_date) ):
            cnt = sum([ events_stat_by_date[i - j][1] for j in range(0, k) ])
            cnt = round(cnt, 2)
            data += [[
                    # +60*60*12 -- подозрительный хак, чтобы Highcharts показывал правильную дату, не смещенную на 3 часа назад
                    int( time.mktime( events_stat_by_date[i][0].timetuple())) + 60 * 720,
                    cnt,
                    ]]
        series += [
                {
                    'data': data,
                    'name': (u"сумма по %s дням" % (k)),
                    },
                ]

    json_data = json.dumps(series, sort_keys=True)

    return HttpResponse(json_data, content_type="application/javascript")

def release_events_by_date_data(r):
    events = get_release_events_by_date((datetime.now() - timedelta(days=25)).strftime('%Y-%m-%d'))
    events_str = {}
    for d in events:
        d_str = d.strftime('%Y-%m-%d')
        events_str[d_str] = events[d]

    json_data = json.dumps(events_str, sort_keys=True, indent=2)
    return HttpResponse(json_data, content_type="application/javascript")

