# -*- coding:utf-8 -*-

from django.conf import settings

import os
from importlib import import_module
import json
from startrek_client import Startrek

from releaser.yambclient.client import YambClient
from releaser.startrek.tools import get_startrek_robot_token, get_sign_comment
from releaser.subscriptions.models import Subscription, NotificationLog
from releaser.utils import sendmail

__all__ = ['BaseEvent', 'event', 'event_registry']

event_registry = {}

def event(event_class):
    event_type = event_class.event_type

    if event_type in event_registry:
        raise RuntimeError('duplicate event name: %s' % event_type)

    event_registry[event_type] = event_class
    return event_class


class BaseEvent(object):
    def __init__(self, event_data):
        self.event_data = event_data

    @staticmethod
    def display_subscription(sub):
        raise NotImplementedError('must implement display_subscription in a subclass')

    def find_subscriptions(self):
        return Subscription.objects.filter(event_type=self.event_type)

    def send_notifications(self, subscriptions=None):
        if subscriptions is None:
            subscriptions = self.find_subscriptions()

        yamb_client = None

        for sub in subscriptions:
            content = self.notification_content(sub.notification_type, sub.recipient)

            if sub.notification_type == Subscription.NOTIFICATION_TYPE_YAMB:
                if yamb_client is None:
                    yamb_client = YambClient()

                yamb_client.send_message(yamb_client.format_login(sub.recipient), content)
            elif sub.notification_type == Subscription.NOTIFICATION_TYPE_EMAIL:
                email_from = 'ppc@yandex-team.ru'
                if 'email_from' in content:
                    email_from = content['email_from']

                sendmail(email_from, sub.recipient, content['subject'].encode('utf-8'), content['body'].encode('utf-8'))
            elif sub.notification_type == Subscription.NOTIFICATION_TYPE_STARTREK:
                startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
                startrek.issues[sub.recipient].comments.create(text=content + u"\n" + get_sign_comment(os.path.basename(__file__)))
            else:
                raise RuntimeError(u'Unknown notification type {notification_type}'.format(notification_type=notification_type))

            NotificationLog.objects.create(
                event_type=self.event_type,
                event_data=json.dumps(self.event_data, default=unicode, ensure_ascii=False),
                subscription_event_type=sub.event_type,
                subscription_params=sub.params,
                notification_type=sub.notification_type,
                recipient=sub.recipient,
                notification_content=json.dumps(content, ensure_ascii=False),
                )

            if sub.notify_once:
                sub.delete()

    def notification_content(self, notification_type, recipient):
        if notification_type == Subscription.NOTIFICATION_TYPE_YAMB:
            return self.notification_content_for_yamb(recipient)
        if notification_type == Subscription.NOTIFICATION_TYPE_EMAIL:
            return self.notification_content_for_email(recipient)
        if notification_type == Subscription.NOTIFICATION_TYPE_STARTREK:
            return self.notification_content_for_startrek(recipient)

    def notification_content_for_yamb(self, recipient):
        raise NotImplementedError('must implement notification_content or notification_content_for_yamb in a subclass')

    def notification_content_for_email(self, recipient):
        raise NotImplementedError('must implement notification_content or notification_content_for_email in a subclass')

    def notification_content_for_startrek(self, recipient):
        raise NotImplementedError('must implement notification_content or notification_content_for_startrek in a subclass')


for app in settings.INSTALLED_APPS:
    if not app.startswith('releaser.'):
        continue

    try:
        import_module(app + '.events')
    except ImportError as error:
        if str(error) != 'No module named events':
            raise
