# -*- encoding: utf-8 -*-

from __future__ import with_statement

from django.core.management.base import BaseCommand
from django.db import transaction, models
from django.core.files import locks

from pprint import pprint as p
import pysvn, tempfile, re, string, os, time, sys, errno
from datetime import datetime
from optparse import make_option
from contextlib import contextmanager

from django.conf import settings
from releaser.svnlog.models import SvnLog, SvnAuthor, SvnBranch, SvnLogBranch, SvnLogHotfix, SvnPath, SvnDiff

from releaser.utils import decode_str
import releaser.common.apps_conf as AppsConf

@contextmanager
def locked_file(filename, retries=None, timeout=None):
    full_filename = "%s/%s" % (settings.LOCKS_PATH, filename)
    if retries is None:
        retries = 0
    if timeout is None:
        timeout = 1
    i = 0
    f = open(full_filename, 'wb')
    while True:
        try:
            locks.lock(f, locks.LOCK_EX|locks.LOCK_NB)
            break
        except (IOError), (e):
            if i >= retries or e.errno not in (errno.EACCES, errno.EAGAIN):
                raise e
            i = i + 1
            time.sleep(timeout)
    try:
        yield
    finally:
        locks.unlock(f)
        f.close()

class Command(BaseCommand):
    help = "Fetch svn log entries from svn to database"
    debug = True

    option_list = BaseCommand.option_list + (
        make_option('--limit', action='store', dest='limit', type='int',
            default=0, help='Number of svn log entries to fetch'),
        make_option('--lock-retries', action='store', dest='lock_retries', type='int',
            default=0, help='Number of retries to get file lock'),
        make_option('--lock-timeout', action='store', dest='lock_timeout', type='int',
            default=1, help='Delay in seconds to sleep betweeen retries of get file lock'),
        make_option('--debug', action='store_true', dest='debug',
            default=False, help='Print steps of command processing'),
    )

    def dprint(self, *args):
        if self.debug:
            print ", ".join(map(lambda x: unicode(x), args))

    def last_rev(self):
        try:
            last_rev = SvnLog.objects.order_by('-rev')[0].rev
        except IndexError:
            last_rev = 0
        return last_rev

    def new_log_entries(self, limit=0):
        if limit > 0:
            limit = limit + 1
        last_rev = self.last_rev()
        svnlog = pysvn.Client().log(settings.RELEASER_SVN + ('/'+settings.RELEASER_SVN_DIR if settings.RELEASER_SVN_DIR else ''),
                                    revision_start=pysvn.Revision(pysvn.opt_revision_kind.number, last_rev),
                                    revision_end=pysvn.Revision(pysvn.opt_revision_kind.head),
                                    discover_changed_paths=True,
                                    limit=limit,
                                    )
        if svnlog and svnlog[0]['revision'].number == last_rev:
            svnlog = svnlog[1:]

        if settings.PROJECT == 'javadirect':
            svnlog_release_branch = pysvn.Client().log(settings.RELEASER_SVN + settings.SVN_RELEASE_BRANCH,
                                                       revision_start=pysvn.Revision(pysvn.opt_revision_kind.number, last_rev),
                                                       revision_end=pysvn.Revision(pysvn.opt_revision_kind.head),
                                                       discover_changed_paths=True,
                                                       limit=limit,
                                                       )
            if svnlog_release_branch and svnlog_release_branch[0]['revision'].number == last_rev:
                svnlog_release_branch = svnlog_release_branch[1:]

            svnlog.extend(svnlog_release_branch)
            svnlog = sorted(svnlog, key=lambda x: x['revision'].number)

        if limit > 0 and len(svnlog) > limit - 1:
            svnlog = svnlog[0:limit - 1]

        return svnlog


    def svn_diff(self, rev, url=settings.RELEASER_SVN, new=False):
        # по-хорошему - переписать без использования diff в случае new
        try:
            tempdir = "%s/releaser-" % tempfile.gettempdir()
            try:
                os.makedirs(tempdir)
            except OSError as e:
                if e.errno != errno.EEXIST:
                    raise
            # чтобы этот код работал, нужно в .subversion/config записать diff-cmd = /usr/local/bin/diff01
            # /usr/bin/diff не вполне подходит, т.к. возвращает 2 на бинарные файлы, svn этого не любит.
            # Если в новых версиях diff научится так не делать -- можно использовать его
            diff_text = pysvn.Client().diff(tempdir,
                                        url,
                                        pysvn.Revision( pysvn.opt_revision_kind.number, (rev-1 if not new else 0) ),
                                        url,
                                        pysvn.Revision( pysvn.opt_revision_kind.number, rev ),
                                        diff_options=['-u', '-d', '-U', '5', '-F', '^sub'],
                                        recurse=False,
                                        )
        except pysvn.ClientError as e:
            if "Can't convert string from 'UTF-8' to native encoding" in str(e):
                diff_text = "!!! diff error: " + str(e) + '!!!'
            else:
                raise e
        return diff_text

    def file_branch(self, filename):
        if settings.PROJECT == 'javadirect':
            for app in AppsConf.get():
                match = re.match(r"^/branches/direct/release/%s/([0-9]+)/arcadia" % app, filename)
                if match:
                    return match.group(0)
            match =  re.match(r"^/branches/direct/release/perl/(release-[0-9]+)", filename)
            if match:
                return match.group(0)

        match = re.match(r'^/?(trunk|branches/[^/]+|releases/[^/]+|tags/[^/]+)', filename)
        if match:
            return match.group(0)

        match = re.match(r'^/?([^/]+)', filename)
        if match:
            return match.group(0)
        raise RuntimeError( "Can't determine branch for '%s'" % filename )

    def save_log_diffs(self, svn_log, log):
        for cp in svn_log.changed_paths:
            # в этом бранче пути слишком длинные и не помещаются в таблицы; mysql warning вызывает падение скрипта.
            if re.match('/branches/DIRECT-54491-prod_sozdanie_smart_menedzherom_oshibka_pri_sozdanii_kampanii_posle_srabotavshey_validatsii_na_pervom_shage', cp.path):
                continue
            if re.match('/branches/DIRECT-55445-v_cmd_ajaxretargetingcondestimate_ispolzovat_rbac_get_client_uids_by_clientid_dlya', cp.path):
                continue
            if re.match('/branches/Directmod-6389_-_Poluchenie_dannyh_o_kliente_statistika_chernyj_spisok_klientov_i_licenzijah_na_osnove_ClientID_v_zaprose', cp.path):
                continue
            if re.match('/branches/Directmod-6286_-_Posle_peremoderacii_bannera_cherez_poisk_-_razmorozhennaja_vizitka_ne_pojavljaetsja_v_ocheredi_kampanij_na_proverku', cp.path):
                continue

            self.dprint(cp.path, cp.action, cp.copyfrom_path)
            branch, _ = SvnBranch.objects.get_or_create(path=self.file_branch(cp.path))
            SvnLogBranch.objects.get_or_create(rev=log, branch=branch)
            path, _ = SvnPath.objects.get_or_create(path_hash=SvnPath.hash(cp.path), path=cp.path)

            diff = None
            if cp.action == 'M':
                try:
                    diff = self.svn_diff(log.rev, url=settings.RELEASER_SVN + cp.path)
                except pysvn.ClientError, e:
                    if unicode(e).find('was not found in the repository at revision') >= 0:
                        pass
                    else:
                        raise
            if cp.action == 'A' and log.rev > 1 \
                    or cp.action == 'M' and diff is None:
                diff = self.svn_diff(log.rev, url=settings.RELEASER_SVN + cp.path, new=True)
            if diff is None:
                diff = ''

            if len(diff) > settings.RELEASER_MAX_DIFF_LEN:
                diff = diff[0:settings.RELEASER_MAX_DIFF_LEN-4] + '...'
            if cp.copyfrom_path != None and len(cp.copyfrom_path) >= 200:
                copyfrom_path = cp.copyfrom_path[0:196] + '...'
            else:
                copyfrom_path = cp.copyfrom_path


            diff_lines = diff.split("\n")[4:]
            lines_added = len([None for l in diff_lines if l.startswith('+')])
            if lines_added > 64000:
                lines_added = 64000
            lines_deleted = len([None for l in diff_lines if l.startswith('-')])
            if lines_deleted > 64000:
                lines_deleted = 64000

            SvnDiff.objects.create(rev=log,
                                   path=path,
                                   action=cp.action,
                                   copyfrom_path=copyfrom_path,
                                   copyfrom_rev=cp.copyfrom_revision.number if cp.copyfrom_revision else None,
                                   lines_added=lines_added,
                                   lines_deleted=lines_deleted,
                                   diff=decode_str(diff))

    # на входе - сообщение
    # на выходе - список ревизий, которые мерджились
    def parse_hotfix_message(self, message):
        lines = message.split('\n')
        if len(lines) <= 1:
            return []
        if not lines[0].startswith("RELEASE:"):
            return []
        # смерженные ревизии находим рассчитывая либо на чистый svn log (r123 |...), либо на вывод svn_release.pl (r123: ...)
        return map(lambda m: int(m.group(1)), filter(lambda m: m, [re.match(r"r(\d+) *(:|\|)", str) for str in message.split('\n')]))

    def save_hotfix_info(self, svn_log, log):
        # смотрим, пришёлся ли коммит в какой-нибудь релиз
        releases_set = set()
        for cp in svn_log.changed_paths:
            m = re.match(r"/releases/release-(\d+)", self.file_branch(cp.path))
            if m:
                releases_set.add(int(m.group(1)))

            if settings.PROJECT == 'javadirect':
                for app in AppsConf.get():
                    m = re.match(r"^/branches/direct/release/%s/([0-9]+)/arcadia" % app, self.file_branch(cp.path))
                    if m:
                        releases_set.add(int(m.group(1)))
                m = re.match(r"^/branches/direct/release/perl/release-([0-9]+)", self.file_branch(cp.path))
                if m:
                    releases_set.add(int(m.group(1)))

        if not releases_set:
            return
        # находим промердженные ревизии
        # В принципе, возможно не парсить коммит-сообщение, а смотреть на mergeinfo, pysvn.Client.propget
        hotfix_revisions = self.parse_hotfix_message(svn_log.message)
        for hotfix_rev in hotfix_revisions:
            if hotfix_rev in [5068600, 967787, 1024613, 6022986, 6602127, 8713037, 8716480, 8951454]:
                # коммиты не в direct/, которых нет в базе svnlog, но который есть в коммит-сообщении к релизным бранчам
                # 6022986 - коммит важный для работы java-jobs, но не в Директе
                # Учитывать их не принципиально (произвольный ничего не меняющий в Директе коммит), поэтому просто пропускаем
                # появлялись такие штуки, когда в команду hotfix попадал номер ревью вместо ревизии
                continue
            for release_rev in releases_set:
                # актуально для javadirect табулы
                # если в svnlog нет коммита с ревизией release_rev, то создаем его с любыми not null значениями
                if not SvnLog.objects.filter(rev=release_rev).exists():
                    author, _ = SvnAuthor.objects.get_or_create(login=svn_log.get('author', '---'))
                    SvnLog.objects.create(rev=release_rev,
                                          revtime=datetime.fromtimestamp(svn_log.get('date')),
                                          author=author,
                                          message='для табулы')
                SvnLogHotfix.objects.get_or_create(hotfix_rev=SvnLog.objects.get(rev=hotfix_rev), release_rev=SvnLog.objects.get(rev=release_rev), defaults={'commit_rev': log})

    @transaction.commit_manually
    def handle(self, **options):
        self.debug = options["debug"]
        with locked_file('svnlog_svnfetch', retries=options["lock_retries"], timeout=options["lock_timeout"]):
            for log_entry in self.new_log_entries(limit=options["limit"]):
                try:
                    rev = log_entry.get('revision').number
                    author, _ = SvnAuthor.objects.get_or_create(login=log_entry.get('author', '---'))
                    log = SvnLog(rev=rev,
                                 revtime=datetime.fromtimestamp(log_entry.get('date')),
                                 author=author,
                                 message=decode_str(log_entry.get('message')),
                                 )
                    self.dprint(log)
                    log.save()
                    # разбираемся с изменением
                    self.save_log_diffs(log_entry, log)
                    # парсим и записываем информацию о хотфиксах
                    self.save_hotfix_info(log_entry, log)
                    self.dprint()
                except:
                    transaction.rollback()
                    raise
                finally:
                    transaction.commit()
        # если не было новых коммитов - делаем ролбэк
        transaction.rollback()

