# -* encoding: utf-8 -*-

from django.http import HttpResponse
from django.shortcuts import render_to_response
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.db.models import Q
from django.db import connection
import math, re, sys
from collections import defaultdict

from datetime import datetime, timedelta
from django.conf import settings
from releaser.svnlog.models import *
from releaser.svnreview.models import *
from releaser.svnrelease.models import *
from releaser.svnlog.tools import get_recent_releases, get_new_release_base, get_svnlog_stat

from releaser.ptera.models import NotifyOnDeploy
import releaser.ptera.notifications
from releaser.versionica.models import HostGroup
from releaser.versionica.tools import prod_group_for_app
from releaser.users.models import User
from releaser.releaseplanning.models import TaskToRelease
from releaser.hotfixing.models import HotfixRequest
from releaser.rights.tools import user_rights, has_right, TEAM_RIGHTS

from releaser.migrations.models import Migration, MigrationText
from releaser.cifront.models import BuildBotSingleBuild

from startrek_client import Startrek
from releaser.startrek.tools import get_startrek_robot_token
import releaser.common.apps_conf as AppsConf

@login_required
def index(request):
    svnlog_list = SvnLog.objects.all()
    limit = int(request.REQUEST.get('limit', '1000'))
    releases_limit = 20

    app_id = int(request.REQUEST.get('app_id', settings.SVNLOG_DEFAULT_APP_ID))
    app_name = settings.APPS_ID2NAME.get(app_id, "")
    dest_group = None
    if app_name:
        dest_group = prod_group_for_app(app_name)
    if app_id == -1:
        releases_list = []
    else:
        releases_list = get_recent_releases(releases_limit, app_id=app_id)

    since = None
    if 'since' in request.REQUEST and request.REQUEST.get('since'):
        since = request.REQUEST.get('since')
    till = None
    if 'till' in request.REQUEST and request.REQUEST.get('till'):
        till = request.REQUEST.get('till')
    author = None
    if 'author' in request.REQUEST and request.REQUEST.get('author'):
        author = request.REQUEST.get('author')
    days = None
    if 'days' in request.REQUEST and request.REQUEST.get('days'):
        days = int(request.REQUEST.get('days'))
    startrek_query = None
    if 'startrek_query' in request.REQUEST and request.REQUEST.get('startrek_query'):
        startrek_query = request.REQUEST.get('startrek_query')
    file_regexp = ''
    if 'file_regexp' in request.REQUEST and request.REQUEST.get('file_regexp'):
        file_regexp = request.REQUEST.get('file_regexp')
    branch = request.REQUEST.get('branch', settings.SVN_DEFAULT_BRANCH)

    release_id = -2
    if 'release_id' in request.REQUEST and request.REQUEST.get('release_id'):
        release_id = int(request.REQUEST.get('release_id'))

    if app_id == -1:
        release_id = -1
        if days == None:
            days = 30

    svnlog_stat = {}
    release = None
    if release_id > 0:
        # конкретный релиз
        try:
            release = SvnRelease.objects.get(pk=release_id, app_id=app_id)
            try:
                prev_release = SvnRelease.objects.all().filter(base_rev__lt=release.base_rev, app_id=app_id).order_by('-base_rev')[0]
                prev_base_rev = prev_release.base_rev
            except IndexError:
                # это -- самый-самый первый релиз
                prev_base_rev = 1
            release_branch = '/releases/release-%d' % release.base_rev
            svnlog_list = svnlog_list.filter(Q(svnlogbranch__branch__path = settings.SVN_DEFAULT_BRANCH, rev__range=(prev_base_rev+1, release.base_rev)) | Q(svnlogbranch__branch__path = release_branch))
        except SvnRelease.DoesNotExist:
            release_id = -2


    if release_id == -2:
        # новый(несуществующий) релиз
        base = get_new_release_base(app_id=app_id)
        svnlog_list = svnlog_list.filter(svnlogbranch__branch__path = settings.SVN_DEFAULT_BRANCH, rev__gt=base)
    elif release_id == -1:
        # без фильтра
        if since:
            svnlog_list = svnlog_list.filter(rev__gt=since)
        if till:
            svnlog_list = svnlog_list.filter(rev__lt=till)
        if author:
            svnlog_list = svnlog_list.filter(author__login=author)
        if branch:
            svnlog_list = svnlog_list.filter(svnlogbranch__branch__path = branch)
        if days:
            svnlog_list = svnlog_list.filter(revtime__gt=datetime.today() - timedelta(days=days))

    svnlog_list = svnlog_list.order_by('-rev')
    if limit is not None:
        svnlog_list = svnlog_list[:limit]
    svnlog_list = svnlog_list.select_related('author')
    svnlog_list = list(svnlog_list)

    if startrek_query:
        svnlog_list = filter_svnlog_by_issues(svnlog_list, startrek_query = startrek_query)
    if file_regexp:
        svnlog_list = filter_svnlog_by_file_regexp(svnlog_list, file_regexp = file_regexp)

    if release_id != -1:
        svnlog_stat = get_svnlog_stat(svnlog_list)

    # делаем массовые запросы ко всем нужнам свойствам
    revs = [l.rev for l in svnlog_list]
    task_to_release_by_rev = {}
    for t in TaskToRelease.objects.filter(commit__in=revs):
        task_to_release_by_rev[t.commit] = None
    reviews_by_rev = defaultdict(list)
    for r in SvnReview.objects.filter(rev__in=revs):
        reviews_by_rev[r.rev_id].append(r)
    hotfix_releases_by_rev = defaultdict(list)
    for h in SvnLogHotfix.objects.select_related('release_rev').filter(hotfix_rev__in=revs):
        hotfix_releases_by_rev[h.hotfix_rev_id].append(h)
    hotfix_requests_by_rev = {}
    for h in HotfixRequest.objects.select_related('applicant').filter(commit__in=revs):
        hotfix_requests_by_rev[h.commit_id] = h
    diffs_by_rev = defaultdict(list)
    for d in SvnDiff.objects.filter(rev__in=revs):
        diffs_by_rev[d.rev_id].append(d)

    notifications_by_rev = {}
    if dest_group:
        notifications_by_rev = releaser.ptera.notifications.commit_notifications_mass(revs, [dest_group])

    build_results = BuildBotSingleBuild.objects.filter(rev__in=revs).select_related('builder__name')
    build_results_by_rev = {}
    for b in build_results:
        if not b.rev in build_results_by_rev:
            build_results_by_rev[b.rev] = {}
        if not b.builder.name in build_results_by_rev[b.rev]:
            build_results_by_rev[b.rev][b.builder.name] = []
        build_results_by_rev[b.rev][b.builder.name].append(b)

    for l in svnlog_list:
        l.mentioned = l.rev in task_to_release_by_rev
        l.reviews = reviews_by_rev[l.rev]
        l.hotfix_releases = hotfix_releases_by_rev[l.rev]
        l.hotfix_request = hotfix_requests_by_rev.get(l.rev, None)
        diffs = diffs_by_rev[l.rev]
        l.files_changed = len(diffs)
        l.lines_added = sum([d.lines_added for d in diffs])
        l.lines_deleted = sum([d.lines_deleted for d in diffs])

        if dest_group:
            l.subscribers = [
                u.login for u in notifications_by_rev.get(l.rev, {}).get(dest_group, [])
                if u.login != request.user.username
            ]

        if l.rev in build_results_by_rev:
            newest_results = [ build_results_by_rev[l.rev][b][-1] for b in build_results_by_rev[l.rev]]
            build_cnt = len( newest_results )
            success_cnt = len([ br for br in newest_results if br.status == BuildBotSingleBuild.BUILD_STATUS_SUCCESS ])
            fail_cnt = len([ br for br in newest_results if br.status == BuildBotSingleBuild.BUILD_STATUS_FAIL ])
            if fail_cnt == 0 and success_cnt > 0:
                summary = 'success'
            else:
                summary = 'fail'

            l.build_result = {
                    'summary': summary,
                    'success_cnt': success_cnt,
                    'fail_cnt': fail_cnt,
                    'build_cnt': build_cnt,
                    }
        else:
            l.build_result = {
                    'summary': 'unknown',
                    'success_cnt': 0,
                    'fail_cnt': 0,
                    'build_cnt': 0,
                    }

    deploy_diffs = SvnDiff.objects.all().filter(path__path__contains='/deploy/', action='A').filter(rev__in = revs)
    deploy_files = []
    for d in deploy_diffs:
        m = re.search(r'deploy/([^/]+)$', d.path.path)
        if m:
            deploy_files.append(m.group(1))

    deploys = []
    for df in deploy_files:
        text_ids = (migr.text_id for migr in Migration.objects.filter(filename__in = [df], rev__in = revs))
        texts = MigrationText.objects.filter(migration_text_id__in = text_ids)
        #sys.stderr.write("%s\n\n" % texts)
        if len(texts) == 0:
            texts = [{'filename':df, 'text': 'к сожалению, сейчас текст этой миграции недоступен'}]
        else:
            for t in texts:
                if len(t.text) > 1024:
                    t.text = t.text[:1024]
                    t.text += u"\n\n\n...SOME TEXT SKIPPED..."
        deploys += texts

    tmpl_vars = {'svnlog_list': svnlog_list,
                 'releases_list': releases_list,
                 'deploys': list(deploys),
                 'release_id': release_id,
                 'release': release,
                 'branch': branch,
                 'since': since,
                 'till': till,
                 'author': author,
                 'days': days,
                 'startrek_query':startrek_query,
                 'file_regexp':file_regexp,
                 'current_url': request.build_absolute_uri(),
                 'current_path': request.get_full_path(),
                 'user_rights': user_rights(request.user.username),
                 'svnlog_stat': svnlog_stat,
                 'notifications': releaser.ptera.notifications.user_notifications(request.user.username, [dest_group]),
                 'user_from_team': has_right(request.user.username, TEAM_RIGHTS),
                 'app_id': app_id,
                 'app_name': app_name,
                 'apps_list': AppsConf.get(add_perl_direct=True),
                 'name2dest': {u"Продакшен %s" % app_name: dest_group} if dest_group else {}
                 }

    return render_to_response('svnlog/index.html',
                              tmpl_vars,
                              context_instance=RequestContext(request)
                              )


def filter_svnlog_by_issues(svnlog, startrek_query=None):
    """
    из списка коммитов svnlog оставить только те, которые сделаны по тикетам, попадающим под запрос startrek_query
    """
    if startrek_query == None:
        raise Exception("query expected")

    rev2issues = {}
    for c in svnlog:
        rev2issues[ c.rev ] = re.findall(r'DIRECT-[0-9]+', c.message)

    svnlog_issues = [ i for rev in rev2issues for i in rev2issues[rev] ]

    startrek_query += " key: " + ", ".join(svnlog_issues)

    startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
    required_issues = set( i.key for i in startrek.issues.find(startrek_query) )

    filtered_svnlog = []
    for c in svnlog:
        for i in rev2issues[c.rev]:
            if i in required_issues:
                filtered_svnlog += [c]
                break

    return filtered_svnlog


def filter_svnlog_by_file_regexp(svnlog, file_regexp=None):
    """
    из списка коммитов оставить только те, которые затрагивают файлы, попадающие под регвыр file_regexp
    """
    if file_regexp == None:
        raise Exception("regexp expected")

    revs = [ c.rev for c in svnlog ]

    if len(revs) <= 0:
        return svnlog

    cursor = connection.cursor()
    # параметры в запрос подставляем в два хода:
    # сначала простой подстановкой -- список ревизий (он безопасный, и нет хорошего способа подставить его через плейсхолдинг)
    # потом через плейсхолдинг -- небезопасный file_regexp
    sql_with_revs = "select d.rev from svnlog_svndiff d join svnlog_svnpath p on p.path_id = d.path_id where d.rev in (%s) and p.path rlike %%s group by d.rev" % ( ", ".join([str(r) for r in revs]))
    cursor.execute(sql_with_revs, [file_regexp])
    filtered_revs = [item[0] for item in cursor.fetchall()]

    filtered_revs_dict = {}
    for r in filtered_revs:
        filtered_revs_dict[r] = 1

    filtered_svnlog = [ c for c in svnlog if c.rev in filtered_revs_dict ]

    return filtered_svnlog

