# -*- encoding: utf-8 -*-

from __future__ import with_statement

from django.core.management.base import BaseCommand
from django.db import transaction, models
from django.core.files import locks

from pprint import pprint as p
import tempfile, re, string, os, time, sys
from datetime import datetime
from optparse import make_option
from contextlib import contextmanager

from django.conf import settings
from releaser.svnrelease.models import SvnRelease

from releaser.utils import decode_str

import warnings
with warnings.catch_warnings():
    warnings.filterwarnings("ignore", category=DeprecationWarning)
    from startrek_client import Startrek
import logging
logging.getLogger("startrek_client.collections").addHandler(logging.NullHandler())
from releaser.startrek.tools import startrek_status_key_to_str, get_startrek_robot_token, last_releases_query
import releaser.common.apps_conf as AppsConf

@contextmanager
def locked_file(filename):
    full_filename = "%s/%s" % (settings.LOCKS_PATH, filename)
    f = open(full_filename, 'wb')
    locks.lock(f, locks.LOCK_EX|locks.LOCK_NB)
    try:
        yield
    finally:
        locks.unlock(f)
        f.close()
        os.unlink(full_filename)

class Command(BaseCommand):
    help = "Fetch releases from tracker and save them to database"
    debug = True

    option_list = BaseCommand.option_list + (
            # TODO вернуть default=10, когда накопится 10 релизов perl-Директа из Аркадии
        make_option('--limit', action='store', dest='limit', type='int',
            default=3, help='Number of releases to fetch'),
        make_option('--debug', action='store_true', dest='debug',
            default=False, help='Print steps of command processing'),
    )

    def dprint(self, *args):
        if self.debug:
            print ", ".join(map(lambda x: unicode(x), args))

    @transaction.commit_manually
    def handle(self, **options):
        self.debug = options["debug"]
        with locked_file('svnlog_svnrelease'):
            startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
            apps_conf = AppsConf.get(add_perl_direct=True)

            for ra in apps_conf:
                if ra == 'dna':
                    continue
                if not 'id' in apps_conf[ra]:
                    # приложения без id игнорируем, все равно не заимпортируем ничего
                    continue
                # если у приложения нет компоненты, то скорее всего оно удалено или новое
                # в любом случае скрипт упадет, если такие приложения не пропустить
                if not apps_conf[ra].get('tracker-component', ''):
                    continue

                query = last_releases_query(component=apps_conf[ra]['tracker-component'])
                release_generator = startrek.issues.find(query, per_page=options['limit'])
                releases = []
                try:
                    for r in release_generator:
                        releases.append(r)
                        if len(releases) >= options['limit']:
                            break
                except StartrekError as err:
                    # в Django 1.5 и выше перевод строки добавляется автоматически, нужно будет поменять при обновлении
                    if err.code == 500:
                        self.stderr.write('Startrek server error' + '\n')
                    else:
                        self.stderr.write(err + '\n')

                # как парсить заголовок релизного тикета: по умолчанию -- регексп из settings, но можно переопределить
                # переопределение нужно например для perl-Директа
                release_summary_regexp = settings.JIRA_RELEASE_REGEXP
                if 'release_summary_regexp' in apps_conf[ra]:
                    release_summary_regexp = apps_conf[ra]['release_summary_regexp']

                for r in releases:
                    key = r.key
                    summary = decode_str(r.summary)
                    # преобразуем время из "%Y-%M-%DT%H:%M:%S.NNN+NNNN" в "%Y-%M-%D %H:%M:%S"
                    create_time = re.sub(r'^([0-9-]+)T([0-9:]+)\.[0-9]{3}\+[0-9]{4}$', r'\1 \2', r.createdAt)
                    status = startrek_status_key_to_str(r.status.key)
                    try:
                        release = SvnRelease.objects.get(jira_id=key)
                    except SvnRelease.DoesNotExist:
                        release = SvnRelease(jira_id=key, app_id = apps_conf[ra]['id'])
                    #self.dprint(r)
                    m = re.match(release_summary_regexp.decode('utf-8'), summary)
                    if not m:
                        print summary
                        raise Exception("can't parse %s" % summary)
                    release.title=m.group(1)
                    release.base_rev = int(m.group(2))
                    release.create_time = create_time
                    if len(m.groups()) > 2:
                        release.tip_rev = int(m.group(3) or release.base_rev)
                    else:
                        release.tip_rev = int(release.base_rev)
                    self.dprint(summary)
                    release.jira_status = status
                    release.deploy=''
                    release.deploy_notes=''
                    release.save()
                    transaction.commit()
