# -*- encoding: utf-8 -*-

import datetime
from collections import defaultdict
from django.http import HttpResponse, HttpResponseRedirect, HttpResponseForbidden, Http404
from django.template import RequestContext
from django.contrib.auth.decorators import login_required
from django.shortcuts import render_to_response, get_object_or_404
import json
from django.core.management import call_command

import releaser.ptera.notifications
from releaser.svnreview.models import *
from releaser.svnlog.models import *
from releaser.utils import sendmail
from django.conf import settings
from releaser.rights.tools import user_rights, has_right, TEAM_RIGHTS
from releaser.releaseplanning.models import TaskToRelease
from releaser.cifront.models import BuildBotSingleBuild

def _get_svnlog_or_404(rev):
    try:
        log = SvnLog.objects.get(pk=rev)
    except SvnLog.DoesNotExist:
        call_command('svnfetch', debug=False, limit=1000, lock_retries=30, lock_timeout=1)
        try:
            log = SvnLog.objects.get(pk=rev)
        except SvnLog.DoesNotExist:
            raise Http404
    return log

@login_required
def add(r, rev, error=None):
    log = _get_svnlog_or_404(rev)
    log.files_diffs = sorted(log.svndiff_set.all(), key=lambda x: x.path.path)

    log.mentioned = TaskToRelease.objects.filter(commit=log.rev).count() > 0
    if log.mentioned:
        log.mentiond_id = TaskToRelease.objects.filter(commit=log.rev).order_by('-id')[0].id
    log.reviews = log.svnreview_set.all()
    log.hotfix_releases = log.hotfix_releases_set.all()
    try:
        log.hotfix_request = log.hotfix_requests_set.all()[0]
    except IndexError:
        log.hotfix_request = None

    builds = BuildBotSingleBuild.objects.filter(rev=rev).select_related('builder__name')
    if builds:
        results_by_builder = defaultdict(list)
        for build in builds:
            results_by_builder[build.builder.name].append(build)

        newest_results = [ results_by_builder[b][-1] for b in results_by_builder ]

        build_cnt = len( newest_results )
        success_cnt = len([ br for br in newest_results if br.status == BuildBotSingleBuild.BUILD_STATUS_SUCCESS ])
        fail_cnt = len([ br for br in newest_results if br.status == BuildBotSingleBuild.BUILD_STATUS_FAIL ])

        if fail_cnt == 0 and success_cnt > 0:
            summary = 'success'
        else:
            summary = 'fail'

        log.build_result = {
                'summary': summary,
                'success_cnt': success_cnt,
                'fail_cnt': fail_cnt,
                'build_cnt': build_cnt,
                }

    return render_to_response('svnreview/add.html', {
            'logs': [log],
            'log': log,
            'retpath': r.REQUEST.get('retpath', '/'),
            'resolution': r.REQUEST.get('resolution', None),
            'message': r.REQUEST.get('message', None),
            'error': error,
            'user_rights': user_rights(r.user.username),
            'user_from_team': has_right(r.user.username, TEAM_RIGHTS),
            'current_path': r.get_full_path(),
            'notifications': releaser.ptera.notifications.user_notifications(r.user.username),
            }, context_instance=RequestContext(r))

@login_required
def save(r):
    msg = r.REQUEST.get('message', '').strip()
    res = r.REQUEST.get('resolution', None)
    if not res:
        return add(r, r.REQUEST['rev'], error=u'Выскажите мнение')
    elif res in ['U', 'N'] and not msg:
        return add(r, r.REQUEST['rev'], error=u'При отрицательной резолюции должно быть непустое сообщение')
    log = _get_svnlog_or_404(r.REQUEST['rev'])
    review = None
    if 'overwrite' in r.REQUEST:
        try:
            review = SvnReview.objects.all().filter(rev=log, reviewer=r.user).order_by('-review_time')[0]
        except IndexError:
            pass
    if not review:
        review = SvnReview(rev=log, reviewer=r.user)
    review.resolution = res
    review.review_time = datetime.datetime.now()
    review.message = msg
    review.save()
    if msg:
        if len(msg) > 40:
            short_msg = msg[:40] + '...'
        else:
            short_msg = msg
        subject = u"мнение о r%d - %s" % (log.rev, short_msg)
        if settings.REVIEW_SUBJ_PREFIX:
            subject = settings.REVIEW_SUBJ_PREFIX + subject
        res_text = {'Y': u'можно выкатывать', 'U': u'имею мнение', 'N': u'выкатывать нельзя'}[res]
        body = u"%s: %s, %s\nhttps://%s/svnreview/add/%d" % (r.user.username, res_text, msg, settings.RELEASER_DNS, log.rev)
        cc = None if r.REQUEST.get('no_cc', '0') == '1' else settings.REVIEW_CC_EMAIL
        sendmail("%s@yandex-team.ru" % r.user.username, "%s@yandex-team.ru" % log.author.login, subject.encode('utf-8'), body.encode('utf-8'), cc=cc)
    if 'retpath' in r.REQUEST:
        query = '?retpath=%s' % r.REQUEST['retpath']
    else:
        query = ''
    return HttpResponseRedirect('/svnreview/add/%s%s' % (r.REQUEST['rev'], query))

@login_required
def delete(r, review_id):
    review = get_object_or_404(SvnReview, pk=review_id)
    if r.user == review.reviewer:
        review.delete()
    else:
        return HttpResponseForbidden()
    if 'retpath' in r.REQUEST:
        query = '?retpath=%s' % r.REQUEST['retpath']
    else:
        query = ''
    return HttpResponseRedirect('/svnreview/add/%s%s' % (review.rev.rev, query))

#@login_required
def get(r, rev):
    reviews = SvnReview.objects.all().filter(rev=rev)
    reviews = [{'dt': rw.review_time.strftime('%Y-%m-%d:%H:%M:%S'), 'resolution': rw.resolution, 'reviewer': rw.reviewer.username, 'message': rw.message} for rw in reviews]
    return HttpResponse(json.dumps({'reviews': reviews, 'auth': r.user.is_authenticated()}), mimetype='application/javascript')

