# -*- encoding: utf-8 -*-

from django.core.management.base import BaseCommand
from django.db import models
from django.conf import settings


from startrek_client import Startrek
from releaser.startrek.tools import get_startrek_robot_token, get_sign_comment, startrek_status_key_to_str
from releaser.conductor_client.client import *
from releaser.zookeeper_client.client import ZookeeperClient
from releaser.infra_client.client import InfraClient
import releaser.common.apps_conf as AppsConf

import yaml
import re
import apt
import apt_pkg
import os
import time
import datetime
import dateutil.parser
import pytz


class Command(BaseCommand):
    help = u"Закрываем релизный тикет, если пакет нужной версии установлен на всей группе машин"

    ST_QUERY = (
        'Queue: DIRECT Type: Release ((Status: "Ready to Deploy" Created: > today() - "3month") ' +
        'OR (Status: Closed Created: > today() - "1month")) "Sort by": key desc'
    )

    def check_update_time(self, startrek, ticket):
        change = list(startrek.issues[ticket.key].changelog.get_all(sort='desc', field='status'))[0]
        for field in change.fields:
            if field['field'].id == 'status':
                update_dt = dateutil.parser.parse(change.updatedAt)
                update_dt = update_dt.replace(tzinfo=pytz.utc).astimezone(pytz.timezone('Europe/Moscow'))
                now_dt = datetime.datetime.utcnow().replace(tzinfo=pytz.utc).astimezone(pytz.timezone('Europe/Moscow'))
                return (now_dt - update_dt).total_seconds() > 2 * 60

        return False


    def do_all_work(self):
        zk_client = ZookeeperClient()
        apps_conf = AppsConf.get()
        startrek = Startrek(token=get_startrek_robot_token(), useragent=settings.USER_AGENT)
        tickets = [issue for issue in startrek.issues.find(self.ST_QUERY) if self.check_update_time(startrek, issue)]

        component2app = {
            apps_conf[app]['tracker-component']: app
            for app in apps_conf if 'tracker-component' in apps_conf[app]
        }

        for ticket in tickets:
            try:
                cur_app = ''
                for component in ticket.components:
                    if component.name in component2app:
                        cur_app = component2app[component.name]
                        break

                if not cur_app or cur_app not in apps_conf:
                    continue

                to_do_mark_deployed = apps_conf[cur_app]['tracker-deployed-tag'] not in startrek.issues[ticket.key].tags
                to_do_close = startrek_status_key_to_str(startrek.issues[ticket.key].status.key) != "Closed"

                if not to_do_close and not to_do_mark_deployed:
                    continue

                ticket_version = re.search(r'[0-9]+\..+$', ticket.summary).group(0)

                hosts = ConductorRestClient().groups2hosts(apps_conf[cur_app]['conductor_groups'])
                versions = zk_client.get_versions(cur_app, hosts)

                versions_dict = {}
                for version in versions:
                    if version not in versions_dict:
                        versions_dict[version] = 0
                    versions_dict[version] += 1

                length = len(versions)
                deb_version = ''
                for version in versions_dict:
                    # берем версию релиза, которая в данный момент стоит хотя бы на 80 процентах машин
                    if versions_dict[version] * 10 >= length * 8:
                        deb_version = version
                        break

                if apt_pkg.version_compare(deb_version, ticket_version) >= 0:
                    if to_do_mark_deployed:
                        startrek.issues[ticket.key].update(
                            tags=startrek.issues[ticket.key].tags + [apps_conf[cur_app]['tracker-deployed-tag']]
                        )

                    comment = u'Пакет версии %s установлен на группе продакшеновых машин\n%s' % (
                        ticket_version, get_sign_comment(os.path.basename(__file__))
                    )
                    if to_do_close:
                        startrek.issues[ticket.key].transitions['deploy'].execute(comment=comment, resolution='fixed')
                        InfraClient.create_event(
                            227,
                            310,
                            'maintenance',
                            'major',
                            u'Выложен %s %s: %s, версия %s' % (
                                u"релиз" if to_do_mark_deployed else u"хотфикс", cur_app, ticket.key, ticket_version
                            ),
                            int(time.time()),
                            int(time.time() + 10 * 60),
                            tickets=[ticket.key]
                        )
                    elif apt_pkg.version_compare(deb_version, ticket_version) == 0:
                        pass
                        # здесь ничего не делаем, т.к. для закрытого тикета не знаем,
                        # писали про его версию комментарий или еще нет. Читать комментарии не хотим.
                        #startrek.issues[ticket.key].update(comment=comment)

            except Exception as e:
                print "ERROR: release %s %s %s" % (ticket.key, type(e), e)

        return


    def handle(self, **options):
        self.do_all_work()

        return

