#!/usr/bin/env perl

=head1 NAME

    gencfg-new-direct-group - помошник для выделения новой группы в резервации Директа или ALL_DYNAMIC

=head1 SYNOPSIS

    gencfg-new-direct-group \
        --group-name=java-jobs-heavy --parent-group vla_ppc \
        --hosts-num 15 --memory-gb=40 --cpu-cores=15 \
        [--disk-gb 1000] [--hbf-macros _PPCNETS_] [--itype ppcvm] [--ipv4] \
        [--owners rivik,dspushkin,pe4kin,lena-san,yukaba[,emurav]] \
        [--location sas] [--force-include sas4-1312,...] [--slb]

    Ничего не делает с группами, печатает команды для gencfg-cli
    Все опции можно писать в lower case через дефис, они автоматически приводятся в нужный gencfg вид

    Опции:
        --parent-group
            sox-резервация {sas,man,vla,iva,...}_ppc или all_dynamic
        --location
            Локация (SAS,MAN,...) в динамике (--parent-group ALL_DYNAMIC) или в sox,
            если невозможно автоматически получить локацию из parent-group
        --disk-gb
            Общая квота на диск, квоты на /, /var/log и тд. расчитываются на ее основе
        --force-include
            Не искать хосты автоматикой, использовать указанные
        --slb
            Список балансеров, для которых нужно поднимать туннели на
            инстансах группы (через зяпятую)

    Менять ресурсы существующих групп лучше через рекластеризацию:
    https://wiki.yandex-team.ru/jandekspoisk/sepe/gencfg/console/#uvelicheniemoshhnostibezdobavlenijanovyxinstansov

=cut

use strict;
use warnings FATAL => 'all';
use Data::Dumper;
use Getopt::Long;
use utf8;
use open qw(:std :encoding(utf8));

my (
    $parent_group,
    $itype,
    $hosts_num,
    $group_name,
    $mem_gb,
    $cpu_cores,
    $owners,
    $force_include,
    $hbf_macros,
    $ipv4,
    $location,
    $disk_gb,
    $slb,
);

GetOptions(
    'parent-group=s' => \$parent_group,
    'hosts-num=i' => \$hosts_num,
    'group-name=s' => \$group_name,
    'memory-gb=i' => \$mem_gb,
    'disk-gb=i' => \$disk_gb,
    'cpu-cores=i' => \$cpu_cores,
    'owners=s' => \$owners,
    'itype=s' => \$itype,
    'ipv4' => \$ipv4,
    'hbf-macros=s' => \$hbf_macros,
    'force-include=s' => \$force_include,
    'location=s' => \$location,
    'slb=s' => \$slb,
);

if (! ($parent_group && $hosts_num && $group_name && $mem_gb && $cpu_cores)) {
    system("pod2text-utf8 <$0");
    exit(0);
}

$parent_group =~ s/-/_/g;
$parent_group = uc $parent_group;

$group_name =~ s/-/_/g;
$group_name = uc $group_name;
$group_name =~ s/^${parent_group}_//;

my $gencfg_prj_name = $group_name;
$gencfg_prj_name =~ s/_/-/g;
$gencfg_prj_name = lc $gencfg_prj_name;
$gencfg_prj_name = "direct-$gencfg_prj_name" if $gencfg_prj_name !~ /^direct-/;
$gencfg_prj_name =~ s/-\d+$//;

$group_name = "${parent_group}_$group_name";

my $cpu_power = $cpu_cores * 40;
my $cpu_power_total = $cpu_power * $hosts_num;

# снапшоты тут не учитываем, но и диск не всегда забит, и в снапшоте только рут участвует
$disk_gb //= 1000;
# дефолтные gencfg-тома <> и </logs>, не знаю, зачем они, но с ними работает и ладно
my $vol_gencfg_gb = 2;
my $vol_root_gb = 50;
my $vol_local_gb = 50;
my $vol_varlog_gb = int(($disk_gb - $vol_gencfg_gb - $vol_root_gb - $vol_local_gb) / 2);
my $vol_varwww_gb = $vol_varlog_gb;

$itype //= "ppcvm";
$owners //= "rivik,dspushkin,pe4kin,lena-san,yukaba";
$hbf_macros //= "_PPCNETS_";
$location //= (split(/_/, $parent_group))[0];
$location = uc $location;

my %new_hosts;
my $c = 0;
my @force_include_list = $force_include ? split(/,/, $force_include) : ();

for my $host (sort @force_include_list) {
    last if $c >= $hosts_num;
    $new_hosts{$host} = 1;
    $c++;
}

my $allocate_group_cmd = qq|./optimizers/dynamic/main.py -m $parent_group -a add -g $group_name -o $owners --verbose --location $location --itype $itype --ctype prod --prj '["$gencfg_prj_name"]' --metaprj banner --min_power $cpu_power_total --memory ${mem_gb}Gb --equal_instances_power True --disk ${disk_gb}Gb --min_replicas $hosts_num --max_replicas $hosts_num -y|;

my $create_group_cmd = qq(./utils/common/update_igroups.py -a addgroup --group $group_name --description "$gencfg_prj_name group" --owners $owners --instance-port-func auto --tags '{ "ctype" : "prod", "itype" : "$itype", "prj" : ["$gencfg_prj_name"], "metaprj" : "banner"}' --parent-group ${parent_group} --properties "reqs.instances.memory_guarantee=${mem_gb}Gb,reqs.instances.power=$cpu_power");

my $include_hosts_cmd = qq(./utils/common/update_igroups.py -a addslavehosts -g $group_name -s ) . join(',', sort(keys %new_hosts));
my $remove_full_host_prop_cmd = qq(./utils/common/update_card.py -g $group_name -k properties.full_host_group -v False -y);

print "######## checkout gencfg repo first: https://wiki.yandex-team.ru/jandekspoisk/sepe/gencfg/console/#rabotasgenerilkojjchekautobnovleniepushizmenenijj #########\n\n";
print "ya svn up; ya svn up db\n";

if ($force_include) {
    print "# создаем пустую группу с лимитами по памяти\n";
    print "$create_group_cmd\n";
    print "# включаем туда хосты\n";
    print "$include_hosts_cmd\n";
} else {
    printf "# ищем доступные хосты для заданных ресурсов\n";
    print "$allocate_group_cmd\n";
}

print "./utils/common/fix_cpu_guarantee.py -g $group_name -p $cpu_power\n";

#print "# ставим флаг mtn-ready\n";
#print "./utils/common/update_card.py -g $group_name -k properties.hbf_mtn_ready -v True -y\n";

print "# снимаем свойство full_host\n";
print "$remove_full_host_prop_cmd\n";

print "# ставим родительский макрос HBF\n";
print "./utils/common/update_card.py -g $group_name -k properties.hbf_parent_macros -v $hbf_macros -y\n";

if ($ipv4) {
    print "# включаем туннель во внешний мир\n";
    print "./utils/common/update_card.py -g $group_name -k properties.internet_tunnel -v True -y\n";
    print "./utils/common/update_card.py -g $group_name -k properties.ipip6_ext_tunnel_v2 -v True -y\n";
}

if ($slb) {
    my $mtn_prop = join(',', map { $_ = qq("$_") } split /,/, $slb);
    print "./utils/common/update_card.py -g $group_name -k properties.mtn.tunnels.hbf_slb_name -v '[$mtn_prop]' -y\n"; 
}

print "# добавляем volumes\n";
my $mvc = './utils/common/manipulate_volumes.py';

my $val = qq(-o "/" -p "quota=${vol_root_gb} Gb");
print qq($mvc -a modify -g $group_name $val\n);

$val = qq(-o "/local" -p "host_mp_root=/place,quota=${vol_local_gb} Gb");
print qq($mvc -a add -g $group_name $val || $mvc -a modify -g $group_name $val\n);
$val = qq(-o "/var/log" -p "host_mp_root=/place,quota=${vol_varlog_gb} Gb");
print qq($mvc -a add -g $group_name $val || $mvc -a modify -g $group_name $val\n);
$val = qq(-o "/var/www/ppc.yandex.ru/protected/logs" -p "host_mp_root=/place,quota=${vol_varwww_gb} Gb");
print qq($mvc -a add -g $group_name $val || $mvc -a modify -g $group_name $val\n);

print "ya svn diff db\n\n";
print "# тесты:\n";
print "./utils/common/update_igroups.py -a checkcards\n";
print "# совсем не обязательно ./gen.sh\n\n";
print "# убедиться, что никто не успел закомитить ничего с момента прошлого svn up (в нашу резервацию точно, но лучше - вообще, иначе может сломаться сборка), и быстро комитить:\n";
print "ya svn up db\n";
print "ya svn ci db -m 'New group $group_name'\n";
