#!/usr/bin/env python3
# -*- coding: utf8 -*-

import os, requests, ssl, sys
import logging
from datetime import datetime as dt
from OpenSSL import crypto
from subprocess import check_output, DEVNULL

def qloud_get_json(*args, **kwargs):
    url = 'https://qloud.yandex-team.ru'
    if kwargs.get('ext'):
        url = 'https://qloud-ext.yandex-team.ru'
    with open(os.path.expanduser('~/.qloud-token'), 'r') as f:
        token = f.read().strip()

    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(token)
    session.headers['Accept'] = 'application/json'
    session.headers['Content-Type'] = 'application/json'

    parts = (url, 'api', 'v1') + args
    url = '/'.join(parts)

    try:
        resp = session.get(url)
        resp.raise_for_status()
        result = resp.json()
    except Exception as e:
        logging.error('%s %s: %s' % (type(e), e, resp.text))
        raise

    return result

def qloud_get_envs(project_name, ext=False):
    di = qloud_get_json('project', project_name, ext=ext)
    envs = []
    for app in di['applications']:
        for env in app['environments']:
            envs.append(env['objectId'])
 
    return envs

def qloud_get_domains(project_name, ext=False):
    domains = {}
    for env in qloud_get_envs(project_name, ext=ext):
        try:
            res = qloud_get_json('domains', env, ext=ext)
            for domain in res:
                domains[domain['domainName']] = env
        except Exception as e:
            # может не быть доступа к окружению, это норма
            pass

    return domains

def qloud_check_certs(project_name, ext=False):
    bad = []
    domains = qloud_get_domains(project_name, ext=ext)
    now = dt.utcnow()
    for d, e in domains.items():
        try:
            raw_crt = check_output(['python3', '-c', 'import ssl, sys; print(ssl.get_server_certificate((sys.argv[1], sys.argv[2])))', d, '443'], stderr=DEVNULL, timeout=3)
        except:
            logging.error('cannot get certificate from %s (%s)' % (d, e))

        crt = crypto.load_certificate(crypto.FILETYPE_PEM, raw_crt)
        not_after = dt.strptime(crt.get_notAfter()[:-2].decode("utf-8"), '%Y%m%d%H%M%S')
        diff_days = (not_after - now).days

        if diff_days < 30:
            logging.info('CRIT: cert for %s (env %s) expires in %d days' % (d, e, diff_days))
            bad.append((d, e, diff_days))
        else:
            logging.info('OK: cert for %s (env %s) expires in %d days' % (d, e, diff_days))
    return bad

if __name__ == '__main__':
    logging.basicConfig(stream=sys.stdout, level=logging.INFO, format='%(message)s')

    qloud_check_certs('direct-infra')
    qloud_check_certs('advq', ext=True)
    qloud_check_certs('direct', ext=True)
    qloud_check_certs('mediaselling', ext=True)
