#!/usr/bin/perl

use strict;
use warnings;

=head1 NAME

=head1 SYNOPSIS

domain_change_history_client.pl --db_host=127.0.0.1 --db_port=3309 --db_name=ppclog --db_user=username --db_pass=password [--logdir=/some/dir] [--time_limit=8:00] [--service=http://service.url/handler] [--chunk_limit=1000000]

=head1 ARGUMENTS

db_host, db_port, db_name, db_user, db_password - параметры коннекта к базе данных, в которой хранится история изменения доменов

logdir - директория, в которой будет создан лог-файл domain_change_history.log.YYYYMMDD, в который будет писать скрипт. Необязательный параметр. По умолчанию /var/log/sales-dev-scripts - создаётся при установке пакета.

time_limit - необязательный параметр, который обозначает время в текущем дне, при наступлении которого скрипт не должен продолжать запрашивать и записывать новые данные. Должен быть в формате HH:MM.

service - необязательный параметр, указывающий на адрес веб-сервиса. Если не указан, то используется http://intapi.direct.yandex.ru/DomainChangeHistory

chunk_limit - необязательный параметр, Максимальное кол-во строк, которое, возможно, будет передаваться (это limit, который используется при выборке строк из базы, но кол-во передаваемых строк сервером как правило меньше первоначальной выборки). При слишком больших объёмах данных апач со стороны сервера будет падать с out of memory. Если параметр не указать, то он всё-равно будет применён со значением 1000000.


=head1 DESCRIPTION

Скрипт может запрашивать у веб-сервиса список логов из базы ppclog и забирать данные из этих логов.
Таблицы логов на стороне ppclog именуются следующим образом banners_bs_log_YYYYMMDD.
При заборе данных из конкретной таблицы, вебсервис агрегирует подряд идущие строки по BannerID и domain.
После получения данных скрипт смотрит в таблицу DomainChangeHistory и вставляет строки, при условии,
что либо последняя по дате строка с текущим BannerID имеет другой домен, либо строк с таким BannerID не существует.

=cut

use Net::INET6Glue::INET_is_INET6;

use Yandex::Log;
use Yandex::DBTools;
use Yandex::DateTime;

use URI;
use JSON;
use DateTime;
use Time::HiRes qw(tv_interval gettimeofday);
use Getopt::Long;
use Data::Dumper;
use LWP::UserAgent;
use Pid::File::Flock qw(:auto);

Pid::File::Flock->new();

my ($db_name, $host, $port, $user, $pass, $log_dir, $time_limit, $service_url, $chunk_limit);
$chunk_limit = 1000000;
GetOptions(
    "db_name=s" => \$db_name,
    "db_host=s" => \$host,
    "db_port=i" => \$port,
    "db_user=s" => \$user,
    "db_pass=s" => \$pass,
    "logdir=s" => \$log_dir,
    "time_limit=s" => \$time_limit,
    "service=s" => \$service_url,
    "chunk_limit=i" => \$chunk_limit,
);

unless($host && $port && $db_name && $user && $pass && $log_dir)
{
    print "usage:\n$0 --db_host=127.0.0.1 --db_port=3309 --db_name=ppclog --db_user=username --db_pass=password --logdir=/some/dir [--time_limit=08:00]\n";
    die;
}

$log_dir = '/var/log/sales-dev-scripts' unless $log_dir;

$service_url ||= 'http://intapi.direct.yandex.ru/DomainChangeHistory';

$Yandex::Log::LOG_ROOT = $log_dir;
my $log = Yandex::Log->new( log_file_name => 'domain_change_history.log', date_suf => '%Y%m%d', auto_rotate => 1);

$time_limit = setup_time_limit($time_limit) if $time_limit;

%Yandex::DBTools::DB_CONFIG = (
    utf8 => 1,
    CHILDS => {
        $db_name => {
            host => $host,
            port => $port,
            user => $user,
            pass => $pass
        }
    }
);

$Yandex::DBTools::QUOTE_DB = $db_name;

my $dbh = get_dbh($db_name);

main();

######################### SUBROUTINES ######################### 
sub setup_time_limit
{
    my $time_limit = shift;

    if($time_limit !~ /\d{1,2}:\d{2}/){
        $log->out("time_limit must be in format HH:MM.");
        die;
    }

    my $now = now();
    my ($hour, $minute) = split(':', $time_limit);
    my $time_limit_obj = $now->clone();

    eval {
        $time_limit_obj->set_hour($hour);
        $time_limit_obj->set_minute($minute);
        $time_limit_obj->set_second(0);
    };

    if($@){
        $log->out("incorrect time_limit $time_limit.");
        die;
    }

    if(DateTime->compare($now, $time_limit_obj) >= 0){
        $log->out("time_limit must be more than $now, but got $time_limit_obj.");
        die;
    }

    return $time_limit_obj;
}

sub main
{
    my $from_datetime = get_last_datetime();

    $log->out("Getting list of log table names...");
    my $logs_list = get_logs_list({ service_url => $service_url, from_datetime => $from_datetime || '2009-01-01 00:00:00'});
    $log->out(sprintf("Got %s tables.", scalar(@$logs_list)));

    process_log_list({logs_list => $logs_list, chunk_limit => $chunk_limit});
}

sub get_last_datetime
{
    return get_one_field_sql($dbh, "select max(changeDateTime) from DomainChangeHistory");
}

sub get_logs_list
{
    my $params = shift;

    my $url = URI->new($params->{service_url});

    if($params->{from_datetime}){
        $url->query_form( action => 'get_logs_list', from_datetime => $params->{from_datetime});
    } else {
        $url->query_form( action => 'get_logs_list');
    }

    my $response = get_data($url);
    return $response->{data};
}

sub process_log_list
{
    my $params= shift;

    for my $log_tablename(@{$params->{logs_list}}){
        $log->out("Getting data from $log_tablename...");

        my $options = { service_url => $service_url, log_tablename => $log_tablename };
        $options->{chunk_limit} = $params->{chunk_limit} if $params->{chunk_limit};

        my $inserted = 0;
        do {
            $options->{from_datetime} = get_last_datetime();
            my $data = get_log_data($options);
            $log->out(sprintf("Got %s rows.", scalar(@$data)));

            $log->out("Storing the data...");
            $inserted = store_data($data);
            $log->out(sprintf("Stored %s rows.", $inserted));
        } while($inserted);

        exit_if_exceed_time_limit() if $time_limit;
    }
}

sub get_log_data
{
    my $params = shift;

    my $url = URI->new($params->{service_url});

    if($params->{from_datetime}){
        $url->query_form( action => 'get_log_data', log_tablename => $params->{log_tablename}, from_datetime => $params->{from_datetime}, limit => $params->{chunk_limit});
    } else {
        $url->query_form( action => 'get_log_data', log_tablename => $params->{log_tablename}, limit => $params->{chunk_limit} ); 
    }

    my $response = get_data($url);
    return $response->{data};
}

sub get_data
{
    my $url = shift;

    my $ua = LWP::UserAgent->new();
    $ua->timeout(1800);
    my $response = $ua->get($url);
    $log->die("HTTP error for $url: ".$response->status_line) if !$response->is_success;

    my $result= eval { decode_json $response->content() };
    die "Response isn't in JSON format." unless $result;

    return $result;
}

sub store_data
{
    my $data = shift;

    my $inserted = 0;
    while(my @chunk = splice(@$data, 0, 10000)){
        my @BannerIDs = map { $_->{BannerID} } @chunk;
        my $last_history_hash = get_last_domains_by_BannerID(\@BannerIDs);

        my $rows_for_insert = [];
        for my $row (@chunk){
            my $domain = $last_history_hash->{$row->{BannerID}}->{domain};
            if( !$domain or $domain ne $row->{domain} ){
                push(@$rows_for_insert, [$row->{BannerID}, $row->{domain}, $row->{changeDateTime}]);
                $last_history_hash->{$row->{BannerID}}->{domain} = $row->{domain};
            }

        }

        $inserted += do_mass_insert_sql($dbh, 'insert into DomainChangeHistory (BannerID, domain, changeDateTime) values %s', $rows_for_insert);
    }

    return $inserted;
}

sub get_last_domains_by_BannerID
{
    my $BannerIDs_arrayref = shift;
    my $BannerIDs = join(',', @$BannerIDs_arrayref);

    my $select = qq{
        SELECT t.BannerID, h.domain
        FROM (
            SELECT BannerID, max(changedateTime) dt 
            FROM DomainChangeHistory
            WHERE BannerID in ($BannerIDs)
            GROUP BY BannerID
        ) t JOIN DomainChangeHistory h ON (h.BannerID = t.BannerID AND h.changeDateTime = t.dt)
    };

    return get_hashes_hash_sql($dbh, $select);
}

sub exit_if_exceed_time_limit
{
    if(DateTime->compare(now(), $time_limit) > 0){
        $log->out("Exceed time limit $time_limit. Stopped.");
        exit 0;
    }
}

