# -*- coding: utf-8 -*-

import time
from threading import Lock
from core.counter import Counter, Counters

import logging
logging.getLogger(__name__).addHandler(logging.NullHandler())

from copy import deepcopy
import subprocess
import MySQLdb

@Counters.register
class PPCMysqlCounter(Counter):


    def __init__(self, counter_config, args):
        super(PPCMysqlCounter, self).__init__(counter_config, args)
        self.lock = Lock()
        self.logger = logging.getLogger(__name__)
        self.sensors_to_send = []


    def get_lm_instances(self):
        instances = []
        try:
            instances = [ x.strip() for x in subprocess.check_output(['lm', '--complete']).split() ]
        except Exception as e:
            self.logger.error('No instances: %s - %s' % (type(e), e))

        return instances


    def get_mysql_status_var(self, instance, var):
        value = None
        try:
            with MySQLdb.connect(unix_socket='/var/run/mysqld.%s/mysqld.sock' % (instance,), user='root') as cursor:
                cursor.execute('SHOW GLOBAL STATUS LIKE %s', (var,))
                value = cursor.fetchall()[0][1]
                self.logger.debug('mysql status %s = %s' % (var, value))
        except Exception as e:
            self.logger.error('Cannot get mysql value num: %s - %s' % (type(e), e))

        return value


    def get_mysql_sensors(self):
        sensors = []

        for inst in self.get_lm_instances():
            for var in ('queries', 'innodb_rows_inserted'):
                value = self.get_mysql_status_var(inst, var)
                if not value:
                    continue
                # mode: 'deriv' работает только если не указывать timestamp
                # если хотим производную по агрегату, то это нужно делать при построении графика
                # метка 'sensor' должна присутствовать обязательно
                # делать длинные описания с префиксами mysql_ не обязательно, т.к.:
                # "Проект, кластер и сервис вместе определяют "шард" - единицу хранения, внутри которой строится дерево из необязательных меток"
                sensor = { 'labels': { 'mysql_status_var': var, 'mysql_instance': inst }, 'value': value }
                if self.counter_config.get('set_timestamp', True):
                    sensor['ts'] = int(time.time())
                sensors.append(sensor)

        return sensors


    def run(self):
        while True:
            start = time.time()
            sensors = self.get_mysql_sensors()
            self.logger.debug(str(sensors))
            duration = time.time() - start

            with self.lock:
                self.sensors_to_send = deepcopy(sensors)
 
            if self.poll_interval - duration < 0:
                self.logger.warning('Sensors retrieval took too long')

            time.sleep(max(self.poll_interval - duration, 1))
 

    def get(self):
        with self.lock:
            return self.sensors_to_send
