#!/usr/bin/perl

=head1 NAME

    sandbox-task-times.pl -- посчитать время, в котором задача находилась в разных состояниях

=head1 SYNOPSIS

    sandbox-task-times.pl --task-id 72167245

=head1 DESCRIPTION

    Выводит для задачи время, которое она пробыла в каждом состоянии.
    Пример вывода для задачи 72167245:

        DRAFT: 0
        ENQUEUING: 1
        ENQUEUED: 18
        PREPARING: 1
        EXECUTING: 187
        FINISHING: 6

    Аналог в веб-интерфейсе Sandbox: https://sandbox.yandex-team.ru/task/72167245/history

=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use feature 'state';
use feature 'say';

use Getopt::Long;
use HTTP::Request;
use JSON;
use LWP::UserAgent;
use Time::Piece;

our $SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0';
our $SANDBOX_TOKEN_FILE = $ENV{SANDBOX_TOKEN_FILE} || "$ENV{HOME}/.sandbox/token";
our $DEBUG = 0;

run() unless caller();

sub run {
    my $opt = parse_options();
    my $res = _sandbox_request(GET => "/task/$opt->{task_id}/audit");
    $_->{time} = Time::Piece->strptime(($_->{time} =~ s/\..*$//r), "%Y-%m-%dT%H:%M:%S")->epoch() for @$res;
    my ($status, $time);
    for my $item (grep { $_->{status} } @$res) {
        if ($status && $time) {
            printf "%s %d\n", $status, $item->{time} - $time;
        }
        $time = $item->{time};
        $status = $item->{status};
    }
    exit 0;
}

sub parse_options {
    my %O;
    GetOptions(
        'task-id=i' => \$O{task_id},
        'h|help' => sub { system "podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"; exit 0 },
    ) || die "can't parse options, stop";
    die "no task id\n" unless $O{task_id};
    return \%O;
}


# копипаста из sandbox-ya-make.pl без токена, для /task/<id>/audit не нужен
sub _sandbox_request {
    my ($method, $path, $data) = @_;
    $path = "/$path" if index($path, '/') != 0;
    print STDERR "sending request " . join(" ", $method, $path, $data ? to_json($data) : ()) . "\n" if $DEBUG;

    state $ua;
    $ua //= LWP::UserAgent->new();

#    state $token;
#    if (! $token) {
#        open my $fh, '<:encoding(UTF-8)', $SANDBOX_TOKEN_FILE or die "Can't open $SANDBOX_TOKEN_FILE: $!\n";
#        $token = <$fh>;
#        chomp $token;
#    }

    my $req = HTTP::Request->new($method => $SANDBOX_API_URL . $path);
#    $req->header('Authorization' => "OAuth $token");
    if ($data) {
        $req->header('Content-Type' => 'application/json; charset=utf-8');
        $req->content(encode_json($data));
    }
    my $res = $ua->request($req);
    die 'sandbox responded with status ' . $res->status . ': ' . $res->status_line . "\n" unless $res->is_success;
    print "sandbox response " . $res->decoded_content . "\n" if $DEBUG;
    return from_json($res->decoded_content) if $res->decoded_content;
}

1;
