#!/usr/bin/perl

=head1 DESCRIPTION

    Поставить задачу на сборку (ya make) в sandbox, дождаться завершения, забрать тарбол с результатом сборки и распаковать в нужное место.

=head1 SYNOPSIS

    ./sandbox-ya-make.pl --target direct/logviewer
    ./sandbox-ya-make.pl --target direct/logviewer --target direct/api5
    ./sandbox-ya-make.pl --target direct/logviewer --unpack-dir /var/www/beta.yukaba.8422/java

    ./sandbox-ya-make.pl --target direct/logviewer --revision 12345678
    ./sandbox-ya-make.pl --target direct/logviewer --arcadia-url arcadia:/arc/trunk/arcadia@12345678

    ./sandbox-ya-make.pl --target direct/logviewer --arcadia-url arcadia:/arc/branches/my-branch/arcadia@12345678
    ./sandbox-ya-make.pl --target direct/logviewer --branch my-branch --revision 12345678

    ./sandbox-ya-make.pl --target direct/logviewer --verbose

    ./sandbox-ya-make.pl --task-id 68128177

=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use feature 'state';
use feature 'say';

use Getopt::Long;
use HTTP::Request;
use JSON;
use LWP::UserAgent;

our $SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0';
our $SANDBOX_TOKEN_FILE = $ENV{SANDBOX_TOKEN_FILE} || "$ENV{HOME}/.sandbox/token";
our $DEBUG = 0;

run() unless caller();

sub run {
    my $opt = parse_options();
    my $task_id;
    if ($opt->{task_id}) {
        $task_id = $opt->{task_id};
    } else {
        my $task = _sandbox_request(POST => '/task',
            {
             type => "YA_MAKE",
             context => { checkout_arcadia_from_url => $opt->{arcadia_url}, targets => join(';', @{ $opt->{targets} })},
            }
        );
        $task_id = $task->{id};
        _sandbox_request(PUT => sprintf('/task/%d', $task_id),
            {
                priority => ["SERVICE", "NORMAL"],
                description => 'Build ' . join(', ', @{ $opt->{targets} }) . ' from ' . $opt->{arcadia_url},
                owner => "DIRECT",
                # т. к. результат выполнения проверяем в этом же скрипте, дополнительные нотификации на почту излишни
                notifications => [],
                requirements => { platform => 'linux_ubuntu_14.04_trusty', disk_space => 10 * 1024 * 1024 * 1024 },    # 10 Гб
            }
        );
        _sandbox_request(PUT => '/batch/tasks/start', [$task_id]);
    }
    say "task_id: $task_id";
    
    my $start_time = time();
    my $status;
    while (1) {
        my $delay = 30;    # секунд
        my $info = _sandbox_request(GET => sprintf('/task/%d', $task_id));
        $status = $info->{status};
        say "status: $status";
        last if $status eq 'SUCCESS' || $status eq 'FAILURE';
        say "Timed out" && last if time() - $start_time > $opt->{timeout};

        say "sleeping for $delay seconds";
        sleep $delay;
    }
    my $resources = _sandbox_request(GET => sprintf('task/%d/resources', $task_id))->{items};
    if ($status eq 'SUCCESS') {
        my $context = _sandbox_request(GET => sprintf('/task/%d/context', $task_id));
        say "branch: $context->{ap_arcadia_branch}" if $context->{ap_arcadia_branch};
        say "revision: $context->{ap_arcadia_revision}";

        my ($build_output_info) = grep {$_->{type} eq 'BUILD_OUTPUT'} @$resources;
        my $proxy_url = $build_output_info->{http}->{proxy} . '/direct?stream=tgz';
        my $output_file = "/tmp/sandbox-resource-$build_output_info->{id}.tar.gz";
        system('wget', $proxy_url, '-O', $output_file);
        if ($? >> 8 != 0) {
            die "wget exited with non-zero status, exiting\nCheck resulting archive at $proxy_url\n";
        }
        mkdir $opt->{unpack_dir} unless -d $opt->{unpack_dir};
        chdir $opt->{unpack_dir};
        system('tar', 'xvzf', $output_file);
        exit 0;
    } elsif ($status eq 'FAILURE') {
        my ($output_html_info) = grep {$_->{type} eq 'BUILD_OUTPUT_HTML'} @$resources;
        say "Check task output at " . $output_html_info->{http}->{proxy};
        exit 1;
    } else {
        say "It seems that task is still executing.\nTo continue waiting for it, run\n$0 --task-id $task_id";
        exit 2;
    }
}

sub parse_options {
    my %O = (
        arcadia_url => 'arcadia:/arc/trunk/arcadia',
    );
    GetOptions(
        'b|branch=s' => sub { my $branch = $_[1]; $O{arcadia_url} = "arcadia:/arc/branches/$branch/arcadia" },
        'arcadia-url=s' => \$O{arcadia_url},
        'r|revision=i' => \$O{revision},
        'v|verbose' => \$DEBUG,
        'targets=s@' => \$O{targets},
        'timeout=i' => \$O{timeout},
        'unpack-dir=s' => \$O{unpack_dir},
        'task-id=i' => \$O{task_id},
        'h|help' => sub { system "podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"; exit 0 },
    ) || die "can't parse options, stop";
    die "no targets specified\n" unless ($O{targets} && @{ $O{targets} }) || $O{task_id};
    $O{arcadia_url} .= '@' . $O{revision} if $O{revision};
    $O{timeout} //= 600;
    $O{unpack_dir} //= '.';
    return \%O;
}

sub _sandbox_request {
    my ($method, $path, $data) = @_;
    $path = "/$path" if index($path, '/') != 0;
    print STDERR "sending request " . join(" ", $method, $path, $data ? to_json($data) : ()) . "\n" if $DEBUG;

    state $ua;
    $ua //= LWP::UserAgent->new();

    state $token;
    if (! $token) {
        open my $fh, '<:encoding(UTF-8)', $SANDBOX_TOKEN_FILE or die "Can't open $SANDBOX_TOKEN_FILE: $!\n";
        $token = <$fh>;
        chomp $token;
    }

    my $req = HTTP::Request->new($method => $SANDBOX_API_URL . $path);
    $req->header('Authorization' => "OAuth $token");
    if ($data) {
        $req->header('Content-Type' => 'application/json; charset=utf-8');
        $req->content(encode_json($data));
    }
    my $res = $ua->request($req);
    die 'sandbox responded with status ' . $res->status . ': ' . $res->status_line . "\n" unless $res->is_success;
    print "sandbox response " . $res->decoded_content . "\n" if $DEBUG;
    return from_json($res->decoded_content) if $res->decoded_content;
}

1;
