#!/usr/bin/perl

=head1 TODO

    Этот скрипт скопирован с sandbox-ya-make.pl с необходимыми изменениями. Нужно выделить общую часть и избавиться от дублирования кода.

=head1 DESCRIPTION

    Поставить задачу на сборку пакета (ya package) в sandbox и, если нужно, дождаться завершения.

=head1 SYNOPSIS

    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --version 1.12345678~foo-1
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --package direct/api5/package-api5.json

    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --revision 12345678
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --arcadia-url arcadia:/arc/trunk/arcadia@12345678

    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --arcadia-url arcadia:/arc/branches/my-branch/arcadia@12345678
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --branch my-branch --revision 12345678

    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --verbose
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --verbose --no-wait

    # запустить сборку с патчем
    ./sandbox-ya-package.pl --patch https://paste.yandex-team.ru/532241/text --package direct/jobs/package/package-jobs.json --revision 3909034
    ./sandbox-ya-package.pl --patch arc:123456 --package direct/jobs/package/package-jobs.json --revision 12345678

    ./sandbox-ya-package.pl --task-id 68128177

    # Использовать Arcadia API FUSE (https://wiki.yandex-team.ru/akastornov/aapianons/)
    # включит и aapi_fallback -- откат на сборку "по старинке", если Arcadia API не работает
    ./sandbox-ya-package.pl --package direct/logviewer/package-logviewer.json --use-aapi-fuse

=cut

use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use feature 'state';
use feature 'say';

use Yandex::Retry;

use Getopt::Long;
use HTTP::Request;
use JSON;
use LWP::UserAgent;

$| = 1;

$ENV{'SVN_SSH'} = "ssh -S $ENV{'SSH_MASTER_CONN_ARCADIA'}" if $ENV{'SSH_MASTER_CONN_ARCADIA'};

our $SANDBOX_API_URL = 'https://sandbox.yandex-team.ru/api/v1.0';
our $SANDBOX_TOKEN_FILE = $ENV{SANDBOX_TOKEN_FILE} || "$ENV{HOME}/.sandbox/token";
our $DEBUG = 0;
our @GOOD_STATUSES = ( 'SUCCESS', 'RELEASED', );
our @BAD_STATUSES  = ( 'FAILURE', 'EXCEPTION', 'STOPPED', 'NOT_RELEASED' );

run() unless caller();

sub run {
    my $opt = parse_options();
    my $task_id;

    if ($opt->{task_id}) {
        $task_id = $opt->{task_id};
    } else {
        my $custom_fields = [
            ($opt->{patch} ? ({ name => "arcadia_patch", "value" => $opt->{patch} }) : ()),
            { name => 'checkout_arcadia_from_url', value => $opt->{arcadia_url} }, 
            { name => 'packages', value => join(';', @{ $opt->{packages} }) },
            { name => 'custom_version', value => $opt->{version} },
            { name => 'checkout', value => \0 },
            { name => 'force_vcs_info_update', value => \1 },
            { name => 'use_aapi_fuse', value => \1 },
            { name => 'use_arc_instead_of_aapi', value => \1 },
            { name => 'ya_yt_store', value => \1 },
            { name => 'ya_yt_store_threads', value => 32 },
            { name => 'ya_yt_token_yav_secret', value => 'sec-01crzhdg2vbrx7exarvapq1nat#yt_robot-direct-yt-ro' },
        ];

        # можем собирать deb-пакеты, можем tarball-ы для yadeploy
        if ( $opt->{yadeploy} ){
            push $custom_fields, { name => 'package_type', value => 'tarball' };
            push $custom_fields, { name => 'resource_type', value => $opt->{yadeploy_resource_type} }; #"DIRECT_LOGVIEWER_DEPLOY_PACKAGE";
            push $custom_fields, { name => 'release_to_ya_deploy', value => \1 };
        } else {
            push $custom_fields, { name => 'debian_compression_level', value => 1 };
            push $custom_fields, { name => 'sloppy_debian', value => \1 };
            push $custom_fields, { name => 'use_new_format', value => \1 };
            push $custom_fields, { name => 'publish_to', value => 'direct-trusty' };
            push $custom_fields, { name => 'debian_compression_type', value => 'gzip' };
        }

        my $type = "DIRECT_YA_PACKAGE";

        my $task = _sandbox_request(POST => '/task', {
            type => $type,
            custom_fields => $custom_fields,
        });

        $task_id = $task->{id};
        my $description = ( $opt->{yadeploy} ? "(Ya.Deploy) " : "" ) . "Version '$opt->{version}' / Build " . join(', ', @{ $opt->{packages} }) . ' from ' . $opt->{arcadia_url};
        _sandbox_request(PUT => sprintf('/task/%d', $task_id),
            {
                priority => ["SERVICE", "NORMAL"],
                description => $description,
                owner => "DIRECT",
                # т. к. результат выполнения проверяем в этом же скрипте, дополнительные нотификации на почту излишни
                notifications => [],
                requirements => { platform => 'linux_ubuntu_14.04_trusty', disk_space => 20 * 1024 * 1024 * 1024 },    # в байтах, т.е. 20 Гб
            }
        );
        _sandbox_request(PUT => '/batch/tasks/start', [$task_id]);
    }
    say "task_id: $task_id";
    say "task_url: https://sandbox.yandex-team.ru/task/$task_id";
    say "package version: $opt->{version}";
    
    # если попросили не ждать -- не ждем
    if ( !$opt->{wait} ){
        exit 0;
    }
    
    my $start_time = time();
    my $status;
    while (1) {
        my $time_ela = time() - $start_time ;
        my $delay = 30;    # секунд
        say localtime()."";
        say "$time_ela seconds since start";
        my $info = retry tries => 5, pauses => [6], sub {
            return _sandbox_request(GET => sprintf('/task/%d', $task_id))
        };
        $status = $info->{status};
        say "status: $status";
        last if grep {$status eq $_} (@BAD_STATUSES, @GOOD_STATUSES);
        say "Timed out" && last if $time_ela > $opt->{timeout};

        say "sleeping for $delay seconds...\n";
        sleep $delay;
    }
    my $resources = _sandbox_request(GET => sprintf('task/%d/resources', $task_id))->{items};
    if ( $status eq $opt->{expected_status} && $status eq 'SUCCESS' ) {
        my $ya_package_info;
        for my $rs ( @$resources ){
            if ( grep { $_ eq $rs->{type} } @{$opt->{resource_type_with_version}} ){
                $ya_package_info = $rs;
                last;
            }
        }
        say "ready version: $ya_package_info->{attributes}{resource_version}";
        exit 0;
    } elsif ( $status eq $opt->{expected_status} && $status eq 'RELEASED' ){
        # для зарелижинной таски пока непонятно, что надо вывести
    } elsif ( $status eq $opt->{expected_status} ){
        # таска закончилась с ожидаемым статусом, для которого нет никакой специфической обработки
    } elsif ( ( grep {$status eq $_} @GOOD_STATUSES ) && ($status ne $opt->{expected_status}) ){
        die "task finished with status '$status' but '$opt->{expected_status}' expected";
    } elsif ($status eq 'FAILURE') {
        my ($task_logs_info) = grep {$_->{type} eq 'TASK_LOGS'} @$resources;
        say "Check task output at " . $task_logs_info->{http}->{proxy} . '/output.html';
        exit 1;
    } elsif ( grep {$status eq $_} @BAD_STATUSES ){
        # прочие неуспешные статусы без специфической обработки
        die "task finished with (unsuccessful) status '$status'";
    } else {
        say "It seems that task is still executing.\nTo continue waiting for it, run\n$0 --task-id $task_id";
        exit 2;
    }
}

sub parse_options {
    my %O = (
        arcadia_url => 'arcadia:/arc/trunk/arcadia',
        upload => 1,
        wait   => 1,
        use_aapi_fuse => 1,
        expected_status => 'SUCCESS',
        resource_type_with_version => [ 'YA_PACKAGE', 'RTMR_USERTASK_RELEASE_DEB', ],
    );
    GetOptions(
        'b|branch=s' => sub { my $branch = $_[1]; $O{branch} = $branch; $O{arcadia_url} = "arcadia:/arc/branches/$branch/arcadia" },
        'arcadia-url=s' => \$O{arcadia_url},
        'r|revision=i' => \$O{revision},
        'v|verbose' => \$DEBUG,
        # FORGIVEME для релизов steps очень нужна сборка с патчем и обычными версиями. Когда steps научится собираться без патча -- very_special_app_steps можно удалить
        'very-special-app-steps' => \$O{very_special_app_steps},
        'package=s@' => \$O{packages},
        'yadeploy' => \$O{yadeploy},
        'yadeploy-resource-type=s' => \$O{yadeploy_resource_type},
        'timeout=i' => \$O{timeout},
        'task-id=i' => \$O{task_id},
        'version=s' => \$O{version},
        'patch=s' => \$O{patch},
        'wait!' => \$O{wait},
        'use-aapi-fuse!' => \$O{use_aapi_fuse},
        'h|help' => sub { system "podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2"; exit 0 },
        'upload!' => \$O{upload},
        's|expected-status=s' => \$O{expected_status},
        # версию какого ресурса смотрим, чтобы вывести после успешного ожидания
        'resource-type-with-version=s@' => \$O{resource_type_with_version}
    ) || die "can't parse options, stop";
    die "нужно передать путь к файлу с описанием пакета, например: --package direct/logviewer/package-logviewer.json\nили id уже выполняющейся задачи: --task-id 1234567" unless ($O{packages} && @{ $O{packages} }) || $O{task_id};
    
    if ( ! grep { $O{expected_status} eq $_ } @GOOD_STATUSES ){
        die "bad expected-status '$O{expected_status}', stop";
    }

    $O{timeout} //= 1200;
    if (!$O{task_id}) {
        for my $pkg (@{$O{packages}}) {
            die "неправильный путь $pkg, поддерживаются только пакеты Директа\n" if $pkg !~ m!^(direct|adv)/!;
            (my $package_url = $O{arcadia_url} . "/$pkg") =~ s!^arcadia:!svn+ssh://arcadia.yandex.ru!;
            print "checking existence of $package_url\n";
            system 'svn', 'ls', $package_url;
            my $svn_ls_exit_code = $? >> 8;
            exit($svn_ls_exit_code) if $svn_ls_exit_code != 0;
        }

        die "--yadeploy requires --yadeploy-resource-type" if $O{yadeploy} && !$O{yadeploy_resource_type};
        die "--yadeploy-resource-type requires --yadeploy" if !$O{yadeploy} && $O{yadeploy_resource_type};

        die "--patch невозможен вместе с --version" if $O{version} && $O{patch} && !$O{very_special_app_steps};

        $O{arcadia_url} .= '@' . $O{revision} if $O{revision};
        die "--version должна использоваться только с --branch\n" if $O{version} && !$O{branch} && !$O{very_special_app_steps};
        if ($O{branch}) {
            my ($base_rev) = $O{branch} =~ m!^direct/release/[a-z0-9\-]+/([0-9]+)!;
            if ($base_rev) {
                die "для релизной ветки нужно указывать ревизию (--revision)\n" unless $O{revision};
                my $correct_version = "1.$base_rev.$O{revision}-1";
                warn "для релизной ветки опция --version игнорируется, пакет собирается с версией $correct_version\n" if $O{version};
                $O{version} = $correct_version;
            } else {
                die "неправильная ветка $O{branch}, поддерживаются только пакеты Директа\n" if $O{branch} !~ m!^direct/!;
                # для feature-ветки пока неважно, какая версия, лишь бы не было похоже на сборку от транка или релизной ветки
                die "для не-релизной ветки нужно указать версию (--version)\n" if !$O{version};
                die "неправильный формат версии для ветки -- похож на сборку от транка (1.NNNN-1) или релизной ветки (1.NNNN.NNNN-1)\n" if $O{version} =~ m/1\.[0-9]+(\.[0-9]+)-1/;
            }
        } else {
            die "укажите ревизию (--revision)\n" unless $O{revision};
            $O{version} = "1.$O{revision}-1";
        }

        if ($O{patch} && !$O{very_special_app_steps}) {
            if ($O{patch} =~ m!paste\.yandex-team\.ru/([0-9]+)/text$!) {
                $O{version} = substr($O{version}, 0, -2)."~paste-$1-1";
            } elsif ($O{patch} =~ m!^(?:arc|rb):([0-9]+)$!) {
                $O{version} = substr($O{version}, 0, -2)."~arc-$1-1";
            } else {
                die "для патча поддерживаются ссылки с paste (шаблон: https://paste.yandex-team.ru/[0-9]+/text)\nили номера ревью реквеста (arc:[0-9]+ или rb:[0-9]+)";
            }
        }
    }
    return \%O;
}

sub _sandbox_request {
    my ($method, $path, $data) = @_;
    $path = "/$path" if index($path, '/') != 0;
    print STDERR "sending request " . join(" ", $method, $path, $data ? to_json($data) : ()) . "\n" if $DEBUG;

    state $ua;
    $ua //= LWP::UserAgent->new();

    state $token;
    if (! $token) {
        open my $fh, '<:encoding(UTF-8)', $SANDBOX_TOKEN_FILE or die "Can't open $SANDBOX_TOKEN_FILE: $!\n";
        $token = <$fh>;
        chomp $token;
    }

    my $req = HTTP::Request->new($method => $SANDBOX_API_URL . $path);
    $req->header('Authorization' => "OAuth $token");
    if ($data) {
        $req->header('Content-Type' => 'application/json; charset=utf-8');
        $req->content(encode_json($data));
    }
    my $res = $ua->request($req);
    die 'sandbox responded with error: ' . $res->status_line . "\n" unless $res->is_success;
    print "sandbox response " . $res->decoded_content . "\n" if $DEBUG;
    return from_json($res->decoded_content) if $res->decoded_content;
}

1;
