#!/usr/bin/perl -w

# $Id$

=head1 NAME

    pod2crontab.pl -- сборка кронтабов из специальных секций pod

=head1 DESCRIPTION

    Ищет скрипты в указанных каталогах, выбирает из них pod-секции (м.б. несколько) METADATA
    и собирает кронтабы в соответствии с ней (ними).

    Формат -- Apache config.

    Одной записи в кронтабе соответствует одна секция crontab:
    <crontab>
        time: */10 * * * *
        package: frontend
    </crontab>

    Если скрипт запускается из нескольких кронтабов или в одном кронтабе с разными параметрами -- должно быть несколько секций crontab:
    <crontab>
        params: 15
        time: 9,24,39,54 * * * *
        package: scripts-warns
    </crontab>
    <crontab>
        params: 30
        time: 6,36 * * * *
        package: scripts-warns
    </crontab>
    <crontab>
        params: 30
        time: */2 10-23 * * *
        package: conf-test
    </crontab>

    Если для скрипта требуется установка переменных окружения -- можно использовать поле env:
    <crontab>
        env: SETTINGS_LOCAL_SUFFIX=DevTest
        ulimit: -v 4000000 -s 65536
        params: 30
        time: */15 10-23 * * *
        package: conf-dev
    </crontab>

    Пример для использования switchman:
    <crontab>
        ...
        <switchman />
    </crontab>

    С передачей опций:
    <crontab>
        <switchman>
            group: conf-test
            lockname: abc
            <leases>
                FQDN_mem: 4096:MEMMB
                FQDN_cpu: 1:CPU
            </leases>
        </switchman>
    </crontab>

    Примеры для запуска скриптов по шардам:
    <crontab>
        sharded: 1 # см. --shard-num
        only_shards: 1,2, 3 # запуск скрипта только на указанных шардах (работает если заданно sharded: 1)
    </crontab>

	Перенаправление вывода желательно указывать в params_postfix
    <crontab>
        params: -some_param=123
        params_postfix: 2>&1 | tail -1000
        time: 0-59/1 * * * *
        package: scripts
    </crontab>


    Для поля time делается синтаксическая проверка: пять полей вида ^([\d,\-]+|\*)(/\d+)?$

    Для миграции на такую систему хранения крон-записей см. скрипт crontab2pod.pl

    Опции:
        --help, -h
            показать справку

        --default-user
            умолчальный пользователь, от которого должны запускаться скрипты из кронтабов.
            Будет использоваться для тех записей, в которых не указано явно поле user

        --project-root, --pr <path>
            полный путь к корню текущей рабочей копии, в которой собираются кронтабы
            (будет отброшен из полных имен скриптов)

        --scripts-path, --sp <path>
            полные пути к каталогам (в текущей рабочей копии), в которых следует искать скрипты
            можно указать несколько

        --crontab-path <path>
            путь к каталогу, в который будут помещены сгенерированные кронтабы
            может содержать шаблон [%package%] (заменится на имя пакета, для которого кронтаб)
            "debian/[%package%]/etc/cron.d"

        --crontab-name <str>
            имя, которое надо дать сгенерированному крон-файлу
            может (и почти наверняка должен) содержать шаблон [%package%] (заменится на имя пакета, для которого кронтаб)
            "[%package%]-auto"

        --shard-num <number>
            количество шардов, на которые надо запускать скрипты
            Если указан этот параметр i
            И
            в crontab-записи в скрипте указано sharded: 1
            , то из одной crontab-метазаписи будет сгенерировано
            несколько строчек в кронтабе (по числу шардов), с id 1 .. $shard_num

        --shard-num-for-package <package>=<number>
            переопределить число шардов, задаваемое параметром --shard-num для пакета
            например:
            --shard-num-for-package yandex-direct-conf-test-scripts=26

        --mkdir
            если указан, то каталоги для сгенерированных кронтабов будут создаваться по необходимости

        --scripts-path-prefix <path>
            полный путь для каталога, в который устанавливается проект
            в кронтабах скрипты будут записаны относительно этого каталога

        --package-prefix <str>
            общий префикс для имен пакетов
            пример: в скрипте написано package: scripts,
                    package-prefix = 'yandex-direct',
                    ==> кронтаб будет генерироваться для пакета yandex-direct-scripts

        --mailto <package>=<email>
            можно указывать несколько раз:
            --mailto yandex-direct-conf-test='direct-test-cron@yandex-team.ru' --mailto yandex-direct-conf-dev='direct-dev-letters@yandex-team.ru'
            адреса для отсылки уведомлений о кронтабах указанных пакетов. Имя пакета -- полностью, вместе с "общим префиксом"

        --default-mailto <email>
            умолчальный адрес для уведомлений (mailto)
            будет использован для пакетов, которым не назначен отдельный адрес через --mailto

        --verbose, -v
            выводить дополнительную информацию о прогрессе выполнения

        --print-packages, --pp
            выводить список пакетов, для которых генерируются кронтабы
            (полезно для внутрипроектного контроля "пишутся кронтабы только для существующих пакетов")

        --print-scripts, --ps
            выводить список скриптов, которые попали в кронтабы
            (полезно для внутрипроектного контроля "все скрипты запускаются под кроном")

        --dry-run, -n
            "try operation but make no changes" (не писать файлы)

        --debug-preview, -d
            режим для быстрого просмотра "что сгенерируется из данного perl-файла".
            Файлы должны быть заданы в командной строке, результат выводится на stdout
            pod2crontab.pl -d protected/bsClientData.pl protected/ppcStrategyToAutobudget.pl

        --default-fqdn-mem-total
            значение, которое будет подставлено в качестве "total" для FQDN_mem, если оно не указано явно
            на примере записи "FQDN_mem: 1024:int(3/4*MEMMB)" - это было бы "int(3/4*MEMMB)"
            предполаемое использование: в метаданных писать "FQDN_mem: NNNN" и указывать этот параметр
            если параметр не задан и "total" не указан - будет подставлено MEMMB

        --run-before-switchman
            команда, которую нужно выполнить перед запуском swithcman

        --fqdn-mem-meta
            имя арендуемого ресурса в метаданных switchman, который трактуется как "оперативная память".
            умолчание - FQDN_mem

        --fqdn-mem-switchman
            имя арендуемого ресурса в параметрах к switchman, который трактуется как "оперативная память".
            умолчание - FQDN_mem

        --fqdn-cpu-meta
            имя арендуемого ресурса в метаданных switchman, который трактуется как "ядра процессора".
            умолчание - FQDN_cpu

        --fqdn-cpu-switchman
            имя арендуемого ресурса в параметрах к switchman, который трактуется как "ядра процессора".
            умолчание - FQDN_cpu

        --min-memory-lease=N
            не добавлять в запуск свичмана захват семафора памяти, если требуемое значение меньше N

        --default-run <path>
            абсолютный путь к бинарнику, который запускать при пустом run в секции кронтаба
            если не указан ни --default-run, ни run, то запускаться будет скрипт, из которого строился кронтаб

        --only-package=<long package name>
            записать кронтаб только для определённого пакета скриптов (длинное имя с префиксом проекта)

        --skip-packages=<long package name>,<long package name 2>
            записать кронтаб только для пакетов скриптов, отличных от перечисленных

        --split-package meta_package=3
            крон мета-пакета meta_package разделить на 3 части, по хэшу
            реальные пакеты называться должны "${meta_package}-{1,2,3}"


    pod2crontab.pl --pr /var/www/beta.lena-san.8801 --scripts-path /var/www/beta.lena-san.8801/protected/ --crontab-path '/var/www/beta.lena-san.8801/crontabs_generated/debian/[%package%]/etc/cron.d' --crontab-name '[%package%]-auto' --scripts-path-prefix /var/www/ppc.yandex.ru --package-prefix yandex-direct --ps --pp --default-user ppc --default-mailto 'ppc-admin@yandex-team.ru' --mailto yandex-direct-conf-test='direct-test-cron@yandex-team.ru' --mkdir

    К скрипту прилагаются тесты.
    Можно зачекаутить рабочую копию из $HeadURL$, и прогнать тесты командой prove

    Записи вида '*/NN ...' рандомизируются (заменяются на xx-59/NN ...)

=head1 TODO


=cut

use strict;
use utf8;
use open ':std' => ':utf8';

use Encode;
use File::Basename;
use File::Find;
use File::Slurp;
use Getopt::Long;
use List::MoreUtils qw/uniq/;

use ScriptsMetadata;
use Yandex::ScriptDistributor ();
use Yandex::Shell q/yash_quote/;

run() unless caller();

sub run
{
my @SCRIPTS_PATH;
my $DEBUG_PREVIEW;
my ($PROJECT_ROOT, $RUN_PATH, $PACKAGE_PREFIX);
my ($CRONTAB_PATH, $CRONTAB_NAME, $CREATE_DIRECTORIES);
my ($VERBOSE, $PRINT_PACKAGES, $PRINT_SCRIPTS, $DRY_RUN);
my ($DEFAULT_MAILTO, %MAILTO);
my $DEFAULT_USER;
my $DEFAULT_RUN;
my $SHARD_NUM;
my $DEFAULT_SHARD_NUM;
my %SHARD_NUM_FOR_PACKAGE;
my $SWITCHMAN_CONFIG = '';
my $SWITCHMAN_DEFAULT_FQDN_MEM_TOTAL = 'MEMMB';
my $RUN_BEFORE_SWITCHMAN;
my $MIN_MEMORY_LEASE;
my $ONLY_PACKAGE;
my ($SKIP_PACKAGES, %SKIP_PACKAGES_HASH);
my %SPLIT_PACKAGE;
my $STDERROUT_TO_DIR;

GetOptions (
    "d|debug-preview"    => \$DEBUG_PREVIEW,
    "pr|project-root=s"  => \$PROJECT_ROOT,
    "sp|scripts-path=s"  => \@SCRIPTS_PATH,
    "cp|crontab-path=s"  => \$CRONTAB_PATH,
    "cn|crontab-name=s"  => \$CRONTAB_NAME,
    "default-user=s"     => \$DEFAULT_USER,
    "default-run=s"      => \$DEFAULT_RUN,
    "shard-num=s"        => \$SHARD_NUM,
    "shard-num-for-package=s" => \%SHARD_NUM_FOR_PACKAGE,
    "mkdir"              => \$CREATE_DIRECTORIES,
    "scripts-path-prefix=s" => \$RUN_PATH,
    "package-prefix=s"   => \$PACKAGE_PREFIX,
    "only-package=s"     => \$ONLY_PACKAGE,
    "skip-packages=s"    => \$SKIP_PACKAGES,
    "split-package=i"    => \%SPLIT_PACKAGE,
    "v|verbose"          => \$VERBOSE,
    "pp|print-packages"  => \$PRINT_PACKAGES,
    "ps|print-scripts"   => \$PRINT_SCRIPTS,
    "default-mailto=s"   => \$DEFAULT_MAILTO,
    "mailto=s"           => \%MAILTO,
    "n|dry-run"          => \$DRY_RUN,
    "h|help"             => \&ScriptsMetadata::usage,
    "switchman-config=s" => \$SWITCHMAN_CONFIG,
    "run-before-switchman=s" => \$RUN_BEFORE_SWITCHMAN,
    "default-fqdn-mem-total=s" => \$SWITCHMAN_DEFAULT_FQDN_MEM_TOTAL,
    "min-memory-lease=i" => \$MIN_MEMORY_LEASE,
    "fqdn-mem-meta=s" => \$ScriptsMetadata::MEM_NAME_IN_META,
    "fqdn-mem-switchman=s" => \$ScriptsMetadata::MEM_SWITCHMAN_LEASE,
    "fqdn-cpu-meta=s" => \$ScriptsMetadata::CPU_NAME_IN_META,
    "fqdn-cpu-switchman=s" => \$ScriptsMetadata::CPU_SWITCHMAN_LEASE,
    "script-distributor-conf-path=s" => \$Yandex::ScriptDistributor::CONF_FILE_PATH,
    "stderrout-to-dir=s" => \$STDERROUT_TO_DIR,
) or die $@;

$DEFAULT_SHARD_NUM = $SHARD_NUM;
if ($SHARD_NUM) {
    $Yandex::ScriptDistributor::SHARDS_NUM_REF = \$SHARD_NUM;
}
if ($SKIP_PACKAGES) {
    %SKIP_PACKAGES_HASH = map {$_ => undef} split /,/, $SKIP_PACKAGES;
}

if ( $DEBUG_PREVIEW ){
     $PROJECT_ROOT ||= '.';
     $RUN_PATH ||= '<path to scripts>';
     $CRONTAB_PATH ||= '<crontab_dir>';
     $CRONTAB_NAME ||= '[%package%]-auto';
     $PACKAGE_PREFIX ||= '<package_prefix>';
     $DEFAULT_USER ||= '<user>';
     $DEFAULT_RUN ||= '<run_command>';
}

$RUN_PATH =~ s!/$!!;

die "can't find path $PROJECT_ROOT" unless $PROJECT_ROOT && -d $PROJECT_ROOT;
for (@SCRIPTS_PATH) {
    die "can't find path $_" unless $_ && -d $_;
}

# составляем список файлов. О "под контролем версий" не заботимся, т.к. сборка должна работать и на чистом export'нутом коде
my @files;
if ($DEBUG_PREVIEW){
    @files = @ARGV;
} else {
    find(
        sub {
            if (/\.p[lmy]$/ && -f $File::Find::name) {
                push @files, $File::Find::name;
            }
        },
        @SCRIPTS_PATH
    );
}

# читаем метаданные из pod и заполняем хеш %POD:
# (
#   package_name_1 => [ {crontab record1}, {crontab record 2} ],
#   package_name_2 => [ ... ],
# )
my %POD;
for my $path_to_file (@files) {
    print "processing $path_to_file\n" if $VERBOSE;
    # для каждого файла
    # выбрать нужную секцию
    my $conf = ScriptsMetadata::get_conf($path_to_file);
    next unless exists $conf->{crontab};

    my $crontabs = ref $conf->{crontab} eq 'ARRAY' ? $conf->{crontab} : [$conf->{crontab}];

    (my $file = $path_to_file) =~ s!^\Q$PROJECT_ROOT\E/!!;
    if (my @errors = map {ScriptsMetadata::validate_crontab($_, package_prefix => $PACKAGE_PREFIX)} @$crontabs) {
        die join "\n", $file, @errors;
    }

    print "script: $file\n" if $PRINT_SCRIPTS;

    for my $c (@$crontabs){
        for my $short_package_name ($c->{package} ? split(/\s*,\s*/, $c->{package}) : '') {
            my $package_name = $PACKAGE_PREFIX . ( $short_package_name ? "-$short_package_name" : '' );
            push @{$POD{$package_name}->{$file}}, $c;
        }
    }
}


# записываем кронтабы в файлы
for my $package (uniq sort (keys %POD)){
    print "compiling crontab for $package\n" if $VERBOSE;
    $SHARD_NUM = $SHARD_NUM_FOR_PACKAGE{$package} || $DEFAULT_SHARD_NUM;

    if ( $ONLY_PACKAGE && $package ne $ONLY_PACKAGE ){
        print "skipping package '$package' because of --only-package=$ONLY_PACKAGE\n" if $VERBOSE;
        next;
    }

    if (exists $SKIP_PACKAGES_HASH{$package}) {
        print "skipping package '$package' because of --skip-packages=$SKIP_PACKAGES\n" if $VERBOSE;
        next;
    }

    die "empty package name" unless $package;
    print "package: $package\n" if $PRINT_PACKAGES;


    # составляем текст для записи:
    # mailto...
    my @crontab_head;
    my $mailto = $MAILTO{$package} || $DEFAULT_MAILTO;
    push @crontab_head, "MAILTO = $mailto" if $mailto;
    # неплохо бы прописывать вообще всегда
    push @crontab_head, "SHELL=/bin/bash" if $STDERROUT_TO_DIR;

    (my $crontab_name = $CRONTAB_NAME) =~ s/\[%package%\]/$package/g;

    # ... и крон-записи
    my @crontab_parts;
    for my $script (sort keys %{$POD{$package} || {} }){
        my $lines = generate_crontab_lines(
            $POD{$package}->{$script} || [],
            default_user => $DEFAULT_USER,
            default_run => $DEFAULT_RUN,
            run_path => $RUN_PATH,
            script => $script,
            shard_num => $SHARD_NUM,
            switchman_config => $SWITCHMAN_CONFIG,
            default_fqdn_mem_total => $SWITCHMAN_DEFAULT_FQDN_MEM_TOTAL,
            run_before_switchman => $RUN_BEFORE_SWITCHMAN,
            min_memory_lease => $MIN_MEMORY_LEASE,
            stderrout_to_dir => $STDERROUT_TO_DIR,
            crontab_name => $crontab_name,
        );
        push @crontab_parts, @$lines;
    }

    my @suffixes = $SPLIT_PACKAGE{$package} ? (map {"-$_"} 1..$SPLIT_PACKAGE{$package}) : ('');

    for my $i (0..$#suffixes) {
        # Важно! в конце кронтаба должна быть пустая строка (перевод строки, и только затем конец файла)
        my @lines = (
            @crontab_head,
            (map {$crontab_parts[$_]} grep {$_ % scalar(@suffixes) == $i} 0..$#crontab_parts),
        );
        my $text_to_write = join("\n\n", @lines, "");

        # составляем имена каталога и файла
        (my $dir = $CRONTAB_PATH) =~ s/\[%package%\]/$package$suffixes[$i]/g;
        system("mkdir -p $dir") == 0 or die $! if !-d $dir && $CREATE_DIRECTORIES && !$DRY_RUN;

        # собственно пишем
        next if $DRY_RUN;
        if ($DEBUG_PREVIEW) {
            print "### To write: $dir/$crontab_name$suffixes[$i]\n\n$text_to_write\n\n"
        }
        else {
            write_file("$dir/$crontab_name$suffixes[$i]", { binmode => ':utf8', atomic => 1 }, $text_to_write);
        }
    }
}


exit;

}

=head2 generate_crontab_lines

=cut

sub generate_crontab_lines
{
    my ($cron_records, %O) = @_;

    die "incorrect shard_num" if defined $O{shard_num} && $O{shard_num} !~ /^\d+$/;

    my @lines;
    for my $h ( @$cron_records ){
        my @opt_list = ScriptsMetadata::process_multiplicators($h, shard_num => $O{shard_num}, script_name => $O{script});
        for my $opt (@opt_list){
            my $shard = $opt->{shard};
            my $run_id = $opt->{run_id};

            my $rnd_key = "$O{script}/";
            if ($h->{params}) {
                $rnd_key .= $h->{params};
            }
            if ($h->{env}) {
                $rnd_key = "$h->{env}/$rnd_key";
            }
            my $rnd_time = ScriptsMetadata::randomized_time($h->{time}, $rnd_key, $shard, $O{shard_num});
            my $logname = _generate_stderrout_log_name($O{stderrout_to_dir}, $O{script}, $h->{params}, $shard, $run_id);

            push @lines, join " ", (
                $rnd_time,
                $h->{user} || $O{default_user} || die("unspecified user in $O{script}"),
                ( $h->{ulimit} ? "ulimit $h->{ulimit} ;" : () ),
                ( $O{run_before_switchman} && $h->{switchman} ? "$O{run_before_switchman} ;" : () ),
                $h->{env} || (),
                ($h->{flock} ? (_generate_flock_prefix($O{crontab_name},
                                                               $O{script},
                                                               $h->{params},
                                                               $shard,
                                                               $run_id,
                                                               $h->{env},
                                                               ))
                                 : ()),
                ($h->{switchman} ? (_generate_switchman_prefix($h->{switchman},
                                                               $O{switchman_config},
                                                               $O{script},
                                                               $h->{params},
                                                               $shard,
                                                               $run_id,
                                                               $h->{env},
                                                               fqdn_mem_total => $O{default_fqdn_mem_total},
                                                               min_memory_lease => $O{min_memory_lease},
                                                               ))
                                 : ()),
                $h->{run} || $O{default_run} || "$O{run_path}/$O{script}",
                (defined $shard ? "--shard-id $shard" : ()),
                (defined $run_id ? "--run-id $run_id" : ()),
                $h->{params} || (),
                $h->{params_postfix} || (),
                ($O{stderrout_to_dir} ? "> $logname 2>&1" : ()),
            );
        }
    }

    return \@lines;
}


sub _generate_stderrout_log_name
{
    my ($prefix, $script, $params, $shard, $run_id) = @_;
    return $prefix if ! $prefix;

    my $logname = basename($script);
    if (defined $run_id) {
        $logname .= ".run_$run_id";
    } elsif ($params) {
        $logname .= '.' . $params =~ s/[\W_]+/_/gr;
    }
    if (defined $shard && ! defined $run_id) {
        $logname .= ".shard_$shard";
    }
    $logname .= '.stderrout.unmerged.$(date +\%s).$(printf "\%05d" $((RANDOM\%10000)))';
    return "$prefix/$logname";
}


sub _generate_switchman_prefix
{
    my ($options, $switchman_config, $script, $params, $shard, $run_id, $env, %defaults) = @_;

    my @parts = ("/usr/bin/switchman");
    if ($switchman_config) {
        push @parts, "-c $switchman_config";
    }
    if (ref $options eq 'HASH') {
        if ($options->{group}) {
            push @parts, "-g $options->{group}";
        }

        if ($options->{delay}) {
            push @parts, "--delay $options->{delay}";
        }

        my $lockname = _lockname($options, $script, $params, $shard, $run_id, $env);
        push @parts, "--lockname $lockname";

        for my $lease (keys %{$options->{leases}}) {
            my $lease_name;
            if ($lease eq $ScriptsMetadata::MEM_NAME_IN_META) {
                if ($options->{leases}->{$lease} !~ m/:/) {
                    next if defined $defaults{min_memory_lease} && $options->{leases}->{$lease} < $defaults{min_memory_lease};
                    $options->{leases}->{$lease} .= ":$defaults{fqdn_mem_total}";
                }
                $lease_name = $ScriptsMetadata::MEM_SWITCHMAN_LEASE;
            } elsif ($lease eq $ScriptsMetadata::CPU_NAME_IN_META) {
                $lease_name = $ScriptsMetadata::CPU_SWITCHMAN_LEASE;
            } else {
                $lease_name = $lease;
            }
            push @parts, "--lease $lease_name=" . yash_quote($options->{leases}->{$lease});
        }
    }
    push @parts, '--';

    return @parts;
}

sub _generate_flock_prefix
{
    my ($crontab_name, $script, $params, $shard, $run_id, $env) = @_;

    my @parts = ("/usr/bin/flock");
    my $lockname = _lockname({}, $script, $params, $shard, $run_id, $env);
    return ("/usr/bin/flock", "-xn", "/run/lock/$crontab_name.$lockname");
}

sub _lockname
{
    my ($options, $script, $params, $shard, $run_id, $env) = @_;
    my $lockname;
    if ($options->{lockname}) {
        $lockname = $options->{lockname};
    } else {
        $lockname = basename($script);
        if (defined $run_id) {
            $lockname .= ".run_$run_id";
        } elsif ($params) {
            $lockname .= '.' . $params =~ s/[\W_]+/_/gr;
        }
    }
    if ($options->{lockname_with_env} && $env) {
        $lockname = ($env =~ s/[\W_]+/_/gr ) . '.' . $lockname;
    }
    if (defined $shard && ! defined $run_id) {
        $lockname .= ".shard_$shard";
    }
    return $lockname;
}
