#!/usr/bin/perl

=pod

=encoding utf-8

=head1 NAME

pod2flat_juggler_playbook.pl - преобразование метаданных в плейбук с проверками

=head1 DESCRIPTION

    Ищет скрипты в каталогах --scripts-path (необходимо указывать абсолютный путь),
    выбирает из них pod-секции (м.б. несколько) METADATA и собирает из них файл playbook.yml
    в поддиректорию --conf-path.

    Проверки собираются трех видов:
        - описанные явно в в секции <juggler> метаданных
        - соответствующие проверки для каждой <monrun> секции
        - проверки, описанные во внешних файлах ( <juggler>from_file: ...</juggler> )

    Полученные файлы пригодны к самостоятельному использованию

=head1 TODO

    - выкинуть разные захардкоженные jcheck_mark, принимать одну параметром, записывать в vars плейбука

=head1 SYNOPSIS

    pod2juggler.pl \
        --scripts-ext pl \
        --scripts-path=/var/www/beta.ppalex.9601/protected \
        --scripts-ext py \
        --scripts-path=/var/www/beta.ppalex.9601/python/scripts \
        --files-path=/var/www/beta.ppalex.9601/etc/juggler_configs \
        --conf-path=/home/ppalex/tmp/ --shards-num=4 \
        --jcheck-mark no_production_dev \
        --intermediate-checks-host checks_auto_aggregation.direct.yandex.ru \
        --monrun-raw-host "CGROUP%direct_ng_backs_prod" \
        --monrun-unreach-service "direct_ng_backs_prod:UNREACHABLE" \
        --default-juggler-name-template "scripts.%s.working" \
        --default-juggler-raw-events-template "scripts.%s.working%s" \
        --juggler-check-raw-host "checks_auto.direct.yandex.ru" \
        --default-juggler-raw-host "CGROUP%direct_ng_scripts_prod"

=head2 Опции

    --scripts-path  - путь по которому требуется искать метаданные в скриптах
    --scripts-ext   - расширения файлов, в которых следует искать метаданные
    --conf-path     - путь, в котором собирается готовый playbook
    --files-path    - путь, по которому в директории tasks лежат уже готовые
                        yaml конфиги, которые могут подключаться через from_file
                        при этом если в директории есть файл initial.yml,
                        то он будет импортирован первым
    --shards-num    - количество шардов
    --jcheck-mark   - внутренняя "метка" проверок, синхронизации и очистки
    --force-checks  - добавить в проверки __force__: true для принудительной синхронизации
    --intermediate-checks-host  - хост для заведения промежуточных проверок
                                    (без голем-нотификаций)
    --monrun-raw-host           - хост, с которого собирать результаты monrun-проверок
    --monrun-unreach-service    - имя unreach-проверки для monrun-проверок (опционально)

    Также поведение скрипта зависит от переменной окружения POD2JUGGLER_STRICT_MODE
    Если она истинна, то вместо выдачи предупреждений - скрипт умирает. Используется
    при сборке пакетов под buildbot.

=head2 Шардирование

    Поддерживаются параметры sharded, а также указание shard внутри vars
    (эквивалент only_shards в monrun-секции)
    При истинном значении sharded поле raw_events должно содержать в себе \$shard\b

=head2 Множители процессов (distributed)

    Не поддерживаются

=head2 Разные raw_events

    raw_events можно указывать несколько раз, но пользоваться этой возможностью
    НЕ РЕКОМЕНДУЕТСЯ без внимательной проверки получающегося результата.
    Отсутствие дубликатов после подстановки переменных - не проверяется.

    Может быть полезно для учета под общую проверку шардированного скрипта
    еще каких-то событий, не включающих в себя шард

=head2 Примеры

    Больше примеров правильной и ошибочной форм записи метаданных можно посмотреть
        в юнит-тесте parse_and_validate_juggler.t
    Для многих полей есть значения по умолчанию, подробнее в ScriptsMetadata

    <juggler>
        name:           scripts.ppcPdfReports.working
        host:           prod_checks.direct.yandex.ru
        raw_host:       CGROUP%direct_ng_scripts_prod
        raw_host:       ppcscripts_new.yandex.ru
        raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
        sharded:        1
        vars:           queue=std,heavy<ttl=2400>
        ttl:            20m
    </juggler>
    <monrun>
        name: bs.object_count_heavy
        crit: 1000000
        expression: 'movingAverage(one_min.direct.production.bs.object_count_heavy, "30min")'
        juggler_host: test_host.direct.yandex.ru
    </monrun>
    <juggler>
        from_file: complex_checks.yml
    </juggler>

    Из первой секции будет сгенерировано (при 4 шардах) 10 аггрегированных проверок:
        * 8 проверок (на каждый шард/очередь), каждая из которых выбирает для
                соответствующего сырого события raw_events (после раскрытия vars:
                    scripts.ppcPdfReports.working.std.shard_1
                    scripts.ppcPdfReports.working.heavy.shard_1
                    scripts.ppcPdfReports.working.std.shard_2
                    ...
                    scripts.ppcPdfReports.working.heavy.shard_4
                ) НАИЛУЧШИЙ (logic_and) статус среди всех машин raw_host (можно указывать
                кондукторные группы).
            Статусы сырых событий являются актуальными в течение ttl секунд
                (600 для *std*, 1200 - для heavy), после чего событие превращается
                в 'NO_DATA'.
            Аггрегат заводится на промежуточный хост 'aggr_host' (его значение
                определено в корневом playbook'е).
                Пример с переопределением vars:
                    queue=std,heavy<ttl=2400>
                Пример с массовым переопределением (сразу ко всем значениям,
                перечисленным после знака равенства:
                    queue<ttl=2400>=heavy1,heavy2
                ).
            Имена проверок совпадают со значениями raw_events после раскрытия
                переменных (vars).
            При наличии переопределения (одного или нескольких) выбирается
                наибольшее из возможных значений.
        * 1 аггрегированная проверка с именем name (scripts.ppcPdfReports.working)
                поверх 8 предыдущих, выбирающая НАИХУДШИЙ статус (logic_or).
            Время пересчета и жизни получившегося результата заданы в playbook'е.
            Аггрегат заводится на хост, указанный как host в <juggler>.
        * проверка, полностью совпадающая с предыдущей, но отправляемая на указанный
            в метаданных host (prod_checks.direct.yandex.ru)

    Из секции monrun будет сгенерировано 2 (одинаковых) аггрегированных проверки:
        Название проверки - такое же, как у события - из метаданных name
            (bs.object_count_heavy)
        Проверка выбирает НАИЛУЧШИЙ (logic_and) статус всех хостов, на которых
            обсчитываются проверки из monrun-секций (заданы в playbook'е)
        Время пересчета и жизни получившегося результата заданы там же.
        Получившаяся проверка будет заведена на хост "по-умолчанию", и на тот, что
            указан явно в метаданных - juggler_host (test_host.direct.yandex.ru)

    Из третьей секции:
        В playbook будет подключен конфиг по пути "--files-path/tasks/complex_checks.yml"
        Если файл отсутствует - будет выдана ошибка.

=cut

use Direct::Modern;

use File::Find;
use File::Slurp;
use Getopt::Long;
use List::MoreUtils qw/uniq/;
use List::Util qw/max/;
use Path::Tiny;
use YAML ();

use ScriptsMetadata;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::ListUtils qw/xsort xflatten/;

use open qw(:std :utf8);

use constant TIMES_FOR_AGGREGATOR => 360;


my ($CONF_PATH, $FILES_PATH, $SHARDS_NUM, $AUTO_HOST);
my (@SCRIPTS_PATH, @FILE_EXTENSIONS);
my ($INTERMEDIATE_HOST, $MONRUN_RAW_HOST, $MONRUN_UNREACH_SERVICE);
my $JCHECK_MARK;
my $FORCE_CHECKS;
my $USE_NOTIFICATIONS;
Getopt::Long::Configure('pass_through');
GetOptions(
    'scripts-path=s' => \@SCRIPTS_PATH,
    'scripts-ext=s' => \@FILE_EXTENSIONS,
    'conf-path=s' => \$CONF_PATH,
    'files-path=s' => \$FILES_PATH,
    'auto-host=s' => \$AUTO_HOST,
    'shards-num=i' => \$SHARDS_NUM,
    'default-juggler-name-template=s' => \$ScriptsMetadata::DEFAULT_JUGGLER_NAME_TEMPLATE,
    'default-juggler-raw-events-template=s' => \$ScriptsMetadata::DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE,
    'default-juggler-raw-host=s' => \@ScriptsMetadata::DEFAULT_JUGGLER_RAW_HOST,
    'juggler-check-raw-host=s' => \$ScriptsMetadata::JUGGLER_CHECK_RAW_HOST,
    'monrun-raw-host=s' => \$MONRUN_RAW_HOST,
    'monrun-unreach-service=s' => \$MONRUN_UNREACH_SERVICE,
    'intermediate-checks-host=s' => \$INTERMEDIATE_HOST,
    'jcheck-mark=s' => \$JCHECK_MARK,
    'force-checks' => \$FORCE_CHECKS,
    'use-notifications' => \$USE_NOTIFICATIONS,
) or die $@;


my %_nice_keys_order = (
    juggler_check   => 1,
    args            => 2,
    with_items      => 3,
    service         => 1,
    monrun_check    => 2,
    ttl             => 3,
    include         => 1,
);

if ($CONF_PATH) {
    die "conf-path $CONF_PATH doesn't exists!" unless -d $CONF_PATH;
} else {
    die 'conf-path is required';
}

if ($JCHECK_MARK && $AUTO_HOST) {
    die "only one of --jcheck-mark or --auto-host should be specified";
}

sub error_msg {
    if ($ENV{POD2JUGGLER_STRICT_MODE}) {
        die @_;
    } else {
        warn @_;
    }
}

@FILE_EXTENSIONS = ('.pl') unless scalar @FILE_EXTENSIONS;

my $exts = join '|', map { s/\./\\./r } @FILE_EXTENSIONS; 
my $regex_file_ext = qr/(?:$exts)$/;

my @files;
find(
    sub {
        my $f = path($_);

        if ($f->basename =~ m/($regex_file_ext)$/ && $f->is_file) {
            push @files, $f->absolute;
        }
    },
    @SCRIPTS_PATH
);

my %PLAYBOOK = (
    hosts => "localhost",
    gather_facts => 'false',
    pre_tasks => [
        {
            juggler_facts => 'jserver_api=http://juggler-api.search.yandex.net/apiv2'
        },
    ],
    tasks => [],
    post_tasks => [
        # DEPRECATED
        {
            juggler_cleanup => "jcheck_mark=direct_prod_admin",
        }, {
            juggler_cleanup => "jcheck_mark=direct_prod_auto",
        }, {
            juggler_cleanup => "jcheck_mark=direct_prod_common",
        },
    ],
    vars => {
        aggr_host => $INTERMEDIATE_HOST,
    },
);
if ($JCHECK_MARK) {
    push @{ $PLAYBOOK{post_tasks} }, { juggler_cleanup => "jcheck_mark=$JCHECK_MARK" };
    $PLAYBOOK{vars}->{check_mark} = $JCHECK_MARK;
}

if (-f "$FILES_PATH/tasks/initial.yml") {
    push @{ $PLAYBOOK{tasks} }, {
        _order => 0,
        include => "tasks/initial.yml",
    };
}

local $ScriptsMetadata::JUGGLER_CHECK_HOST_IS_MANDATORY = 1 unless $AUTO_HOST;

for my $file (@files) {
    my $conf = ScriptsMetadata::get_conf($file);
    if ($conf->{juggler}) {
        for my $j (ref $conf->{juggler} eq 'ARRAY' ? @{ $conf->{juggler} } : $conf->{juggler}) {
            my @errors = ScriptsMetadata::parse_and_validate_juggler($j,
                                                                     files_path => $FILES_PATH,
                                                                     shards_num => $SHARDS_NUM,
                                                                     this_file_path => $file,
                                                                    );
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_juggler($j);
        }
    }
    if ($conf->{juggler_check}) {
        for my $j (ref $conf->{juggler_check} eq 'ARRAY' ? @{ $conf->{juggler_check} } : $conf->{juggler_check}) {
            my @errors = ScriptsMetadata::parse_and_validate_juggler_check($j, shards_num => $SHARDS_NUM);
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_juggler_check($j);
        }
    }
    if ($conf->{monrun}) {
        for my $m (ref $conf->{monrun} eq 'ARRAY' ? @{ $conf->{monrun} } : $conf->{monrun}) {
            my @errors = ScriptsMetadata::validate_monrun($m);
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_monrun($m);
        }
    }
}

@{ $PLAYBOOK{tasks} } = _nice_tasks(@{ $PLAYBOOK{tasks} });

{
    no warnings 'once';
    $YAML::UseAliases = 0;
    $YAML::SortKeys = 1;
}

write_file("$CONF_PATH/playbook.yml", {binmode => ':utf8', atomic => 1}, YAML::Dump([ \%PLAYBOOK ]));

exit 0;

sub append_playbook_with_juggler_check {
    my ($j) = @_;

    my @hosts;
    push @hosts, $AUTO_HOST if $AUTO_HOST;
    if ($j->{host}) {
        push @hosts, ref $j->{host} ? @{ $j->{host} } : $j->{host};
    }
    @hosts = uniq @hosts;
    unless (@hosts) {
        error_msg("host for juggler_check '$j->{name}' is not specified");
    }

    # Добавляем шарды
    if ($j->{sharded} && !defined $j->{vars}->{shard}) {
        die 'shards-num is required for sharded raw_events' unless $SHARDS_NUM && $SHARDS_NUM > 0;
        $j->{vars}->{shard} = { map { $_ => {} } 1..$SHARDS_NUM };
    }

    my $raw_events = ScriptsMetadata::substitute_juggler_vars_and_times($j);
    # в $j->{ttl} может быть умолчание, поэтому его не берем.
    # substitute_juggler_vars_and_times заполнило ttl в каждом $raw_event, берем из них.
    # отсутствие переопределений ttl было проверено в тесте, но лишняя подстраховка не помешает
    my $check_ttl;
    my @uniq_ttl = uniq map { $_->{ttl} } @$raw_events;
    if (@uniq_ttl == 1) {
        $check_ttl = $uniq_ttl[0];
    } else {
        error_msg("invalid ttl for juggler_check '$j->{name}' TTL is not uniue");
        $check_ttl = max(@uniq_ttl);
    }
    # "верхний" - аггрегаты поверх всех проверок сразу (=все должны быть "ок")
    for my $host (@hosts) {
        my $args = hash_merge base_check_template($host, $j->{name}, $check_ttl), all_children_should_be_ok();
        add_notifications($args, $j->{notification});
        add_tags($args, $j->{tag});
        add_urls($args, $j->{meta_url});
        
        for my $raw_event (sort map { $_->{service} } @$raw_events) {
            push @{ $args->{children} }, "$ScriptsMetadata::JUGGLER_CHECK_RAW_HOST:$raw_event";
        }

        push @{ $PLAYBOOK{tasks} }, {
            _order => 3,
            _name => $j->{name},
            _level => 1,
            juggler_check => "$j->{name} (aggregated check on $host)",
            args => $args,
        };
    }
}

sub append_playbook_with_juggler {
    my ($j) = @_;

    if ($j->{from_file}) {
        push @{ $PLAYBOOK{tasks} }, {
            _order => 1,
            include => "tasks/$j->{from_file}",
        };
    } else {
        my @hosts;
        push @hosts, $AUTO_HOST if $AUTO_HOST;
        if ($j->{host}) {
            push @hosts, ref $j->{host} ? @{ $j->{host} } : $j->{host};
        }
        @hosts = uniq @hosts;
        unless (@hosts) {
            error_msg("host for juggler '$j->{name}' is not specified");
        }

        # Добавляем шарды
        if ($j->{sharded} && !defined $j->{vars}->{shard}) {
            die 'shards-num is required for sharded raw_events' unless $SHARDS_NUM && $SHARDS_NUM > 0;
            $j->{vars}->{shard} = { map { $_ => {} } 1..$SHARDS_NUM };
        }
        # Хостов с сырыми событиями может быть несколько
        $j->{raw_host} = [$j->{raw_host}] if ref $j->{raw_host} ne 'ARRAY';
        # Раскрываем название сырых событий (получаем список childs)
        my $raw_events = ScriptsMetadata::substitute_juggler_vars_and_times($j);
        if (@$raw_events > 1) {
            # Создаем два слоая проверок:
            for my $raw_event (@$raw_events) {
                # Перечисление всех сырых событий (скрипт/очередь/шард/...)

                my $args = hash_merge base_check_template($INTERMEDIATE_HOST, $raw_event->{service}, $raw_event->{ttl}), any_child_should_be_ok();
                for my $raw_host (@{ $j->{raw_host} }) {
                    push @{ $args->{children} }, "$raw_host:$raw_event->{service}";
                }

                # 'нижний' - поверх хостов для каждого сырого события
                push @{ $PLAYBOOK{tasks} }, {
                    _order => 2,
                    _name => $j->{name},
                    _level => 0,
                    juggler_check => "$raw_event->{service} (subcheck)",
                    args => $args,
                };
            }

            # "верхний" - аггрегаты поверх всех проверок нижнего слоя
            for my $host (@hosts) {
                my $args = hash_merge base_check_template($host, $j->{name}, TIMES_FOR_AGGREGATOR), all_children_should_be_ok();
                add_tags($args, $j->{tag});
                add_notifications($args, $j->{notification});
                add_urls($args, $j->{meta_url});

                for my $raw_event (sort map { $_->{service} } @$raw_events) {
                    push @{ $args->{children} }, "$INTERMEDIATE_HOST:$raw_event";
                }

                push @{ $PLAYBOOK{tasks} }, {
                    _order => 2,
                    _name => $j->{name},
                    _level => 1,
                    juggler_check => "$j->{name} (aggregated check - on $host)",
                    args => $args,
                };
            }
        } else {
            # так как выражение только одно - промежуточный уровень аггрегации не требуется
            # создаем проверку сразу на хост "верхнего" (с ответственными) уровня
            for my $host (@hosts) {
                my $args = hash_merge base_check_template($host, $j->{name}, $raw_events->[0]->{ttl}), any_child_should_be_ok();
                add_tags($args, $j->{tag});
                add_notifications($args, $j->{notification});
                add_urls($args, $j->{meta_url});
                
                for my $raw_host (@{ $j->{raw_host} }) {
                    push @{ $args->{children} }, "$raw_host:$raw_events->[0]->{service}";
                }

                push @{ $PLAYBOOK{tasks} }, {
                    _order => 2,
                    _name => $j->{name},
                    _level => 1,
                    juggler_check => "$j->{name} (aggregated check on $host)",
                    args => $args,
                };
            }
        }
    }
}

sub append_playbook_with_monrun {
    my ($m) = @_;

    if (!$MONRUN_RAW_HOST) {
        error_msg("monrun-raw-host is mandatory for monrun check");
    }

    my @hosts;
    push @hosts, $AUTO_HOST if $AUTO_HOST;
    if ($m->{juggler_host}) {
        push @hosts, ref $m->{juggler_host} ? @{ $m->{juggler_host} } : $m->{juggler_host};
    }
    @hosts = uniq @hosts;
    unless (@hosts) {
        error_msg("juggler_host for monrun '$m->{name}' is not specified");
    }

    for my $host (@hosts) {
        my $ttl = $m->{interval} ? int($m->{interval} * 3.5) : TIMES_FOR_AGGREGATOR;
        my $args = hash_merge base_check_template($host, $m->{name}, $ttl), any_child_should_be_ok();
        add_tags($args, $m->{tag});
        add_notifications($args, $m->{notification});
        add_urls($args, $m->{meta_url});

        push @{ $args->{children} }, "$MONRUN_RAW_HOST:$m->{name}";

        if ($MONRUN_UNREACH_SERVICE) {
            hash_merge $args->{aggregator_kwargs}, { unreach_mode => 'skip', unreach_service => [ { check => $MONRUN_UNREACH_SERVICE } ] };
        }

        push @{ $PLAYBOOK{tasks} }, {
            juggler_check => "$m->{name} (check over monrun on $host)",
            args => $args,
        };
    }
}

sub base_check_template {
    my ($host, $service, $ttl) = @_;

    _check_service_duplicates($service, $host);

    my %ARGS = (
        host => $host,
        service => $service,
        ttl => $ttl,
        flap => 'false',
        children => [],
	namespace => "direct.prod",
    );

    my $mark;
    if ($JCHECK_MARK) {
        $mark = $JCHECK_MARK;
    } elsif ($host eq $INTERMEDIATE_HOST) {
        $mark = 'direct_prod_common';
    } else {
        $mark = $host eq $AUTO_HOST ? 'direct_prod_auto' : 'direct_prod_admin';
    }
    $ARGS{jcheck_mark} = $mark;

    if ($FORCE_CHECKS) {
        $ARGS{__force__} = 'true';
    }

    return \%ARGS;
}

sub any_child_should_be_ok {
    return {
        aggregator => 'logic_and',
        aggregator_kwargs => {
            downtimes_mode => 'skip',
        },
    };
}

sub all_children_should_be_ok {
    return {
        aggregator => 'logic_or',
    };
}

sub add_tags {
    my ($args, $tags) = @_;

    if (!$tags) {
        return;
    }

    $args->{tags} //= [];

    push @{ $args->{tags} }, grep { $_ } xflatten $tags;
}

sub add_notifications {
    my ($args, $notifications) = @_;

    if (!$notifications) {
        return;
    }

    if (!$USE_NOTIFICATIONS) {
        return;
    }

    $args->{notifications} //= [];
    push @{ $args->{notifications} }, xflatten $notifications;
}

sub add_urls {
    my ($args, $meta_urls) = @_;

    if (!$meta_urls) {
        return;
    }

    $args->{meta} //= {};
    $args->{meta}->{urls} //= [];
    push @{ $args->{meta}->{urls} }, xflatten $meta_urls;
}

sub _nice_yaml_keys_order {
    my $hash = shift;
    YAML::Bless($hash)->keys([ sort { ($_nice_keys_order{$a}||99) <=> ($_nice_keys_order{$b}||99) } keys %$hash ]);
    return $hash;
}

sub _nice_tasks {
    return map {
        _nice_yaml_keys_order($_)
    } map {
        delete @{ $_ }{qw/ _level _name _order /};
        $_;
    } xsort {
        int($_->{_order} // 5), ($_->{_name} // $_->{include} // ''), (int($_->{_level} // 1)), ( $_->{juggler_check} // '')
    } @_;
}

=head3 _check_service_duplicates($service, $host)

    Проверяет, что $service является уникальным в пределах $host.
    При дублировании - выдает warning.
    Хорошо бы падать, потому что собранный playbook - невалиден.

=cut

sub _check_service_duplicates {
    my ($service, $host) = @_;
    state %duplicates;  # host -> service -> count

    if (++($duplicates{$host}->{$service}) > 1) {
        error_msg("\n###### DUPLICATE SERVICE '$service' FOR HOST: '$host' ######\n");
    }

}
