#!/usr/bin/perl

=pod

=encoding utf-8

=head1 NAME

pod2juggler.pl - преобразование метаданных в проверки

=head1 DESCRIPTION

    Ищет скрипты в каталогах --scripts-path (необходимо указывать абсолютный путь),
    выбирает из них pod-секции (м.б. несколько) METADATA и собирает из них файлы
    "vars" и "tasks" (пригодные к подключению в ansible-playbook с использованием
    ansible-juggler плагина), сохраняет их в соответствующие поддиректории --conf-path:
        vars/perl_checks_auto.yml       - данные проверок из <juggler> секций
        vars/python_checks_auto.yml     - данные проверок из <juggler> секций (из .py файлов)
        tasks/scripts_perl_auto.yml     - сами проверки для <juggler>
        tasks/scripts_python_auto.ym    - сами проверки для <juggler> (из .py файлов)
        tasks/monrun_auto.yml           - проверки для <monrun> секций (с данными)

    Проверки собираются трех видов:
        - описанные явно в в секции <juggler> метаданных
        - соответствующие проверки для каждой <monrun> секции
        - проверки, описанные во внешних файлах ( <juggler>from_file: ...</juggler> )

    Полученные файлы непригодны к самостоятельному использованию и
        требуют некоторого workaround'а:
    - описания шаблонов проверок в основном playbook'е
    - наличия фильтр-плагина для ansible, умеющего:
        преобразовывать список проверок в дочернюю проверку (aggregate_check_data)
        расширять список проверок, дописывая имя хоста к дочерним проверкам
            (expand_services_with_hosts)

=head1 SYNOPSIS

    pod2juggler.pl \
        --scripts-ext pl \
        --scripts-path=/var/www/beta.ppalex.9601/protected \
        --scripts-ext py \
        --scripts-path=/var/www/beta.ppalex.9601/python/scripts \
        --files-path=/var/www/beta.ppalex.9601/etc/juggler_configs \
        --conf-path=/home/ppalex/tmp/ --shards-num=4 \
        --auto-tag=direct_prod_auto --manual-tag=direct_prod_admin \
        --auto-host=checks_auto.direct.yandex.ru \
        --default-juggler-name-template "scripts.%s.working" \
        --default-juggler-raw-events-template "scripts.%s.working%s" \
        --juggler-check-raw-host "checks_auto.direct.yandex.ru" \
        --default-juggler-raw-host "CGROUP%direct_ng_scripts_prod"

=head2 Опции

    --scripts-path  - путь по которому требуется искать метаданные в скриптах
    --scripts-ext   - расширения файлов, в которых следует искать метаданные
    --conf-path     - путь, в котором собирается готовый playbook (содержащий
                        папки tasks, vars)
    --files-path    - путь, по которому лежат уже готовые yaml конфиги, которые
                        также следует использовать (скопировать в conf-path)
    --auto-tag      - тег, добавляемый для проверок, создаваемых
                        на автоматический хост (auto-host)
    --manual-tag    - тег, добавляемый для проверок, создаваемых
                        на вручную указанные хосты
    --shards-num    - количество шардов
    --auto-host     - хост, на который (помимо указанных явно) будут заводиться
                        все аггрегированные проверки "верхнего" уровня (поверх
                        предыдущих - например все очереди, шарды одного скрипта,
                        а также monrun проверки)

    Также поведение скрипта зависит от переменной окружения POD2JUGGLER_STRICT_MODE
    Если она истинна, то вместо выдачи предупреждений - скрипт умирает. Используется
    при сборке пакетов под buildbot.

=head2 Шардирование

    Поддерживаются параметры sharded, а также указание shard внутри vars
    (эквивалент only_shards в monrun-секции)
    При истинном значении sharded поле raw_events должно содержать в себе \$shard\b

=head2 Множители процессов

    Поддерживается параметр distributed, означающий, что скрипт получает параметры
    из Yandex::ScriptDistributor. Требуется передавать имя скрипта в переменную (vars)
    script_name как в конфигурации, а также задать количество шардов.
    В переменные и raw_events будут подставлены все возможные run_id по конфигурации.
    Можно задавать разные ttl для разных run_type.
    Путь до конфигурации нужно передавать в параметр --scripts-distributor-conf-path

    в <juggler_check> - не поддерживается.

=head2 Разные raw_events

    raw_events можно указывать несколько раз, но пользоваться этой возможностью
    НЕ РЕКОМЕНДУЕТСЯ без внимательной проверки получающегося результата.
    Отсутствие дубликатов после подстановки переменных - не проверяется.

    Может быть полезно для учета под общую проверку шардированного скрипта
    еще каких-то событий, не включающих в себя шард

=head2 Примеры

    Больше примеров правильной и ошибочной форм записи метаданных можно посмотреть
        в юнит-тесте parse_and_validate_juggler.t
    Для многих полей есть значения по умолчанию, подробнее в ScriptsMetadata

    <juggler>
        name:           scripts.ppcPdfReports.working
        host:           prod_checks.direct.yandex.ru
        raw_host:       CGROUP%direct_ng_scripts_prod
        raw_host:       ppcscripts_new.yandex.ru
        raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
        sharded:        1
        vars:           queue=std,heavy<ttl=2400>
        ttl:            20m
    </juggler>
    <monrun>
        name: bs.object_count_heavy
        crit: 1000000
        expression: 'movingAverage(one_min.direct.production.bs.object_count_heavy, "30min")'
        juggler_host: test_host.direct.yandex.ru
    </monrun>
    <juggler>
        from_file: complex_checks.yml
    </juggler>

    Из первой секции будет сгенерировано (при 4 шардах) 10 аггрегированных проверок:
        * 8 проверок (на каждый шард/очередь), каждая из которых выбирает для
                соответствующего сырого события raw_events (после раскрытия vars:
                    scripts.ppcPdfReports.working.std.shard_1
                    scripts.ppcPdfReports.working.heavy.shard_1
                    scripts.ppcPdfReports.working.std.shard_2
                    ...
                    scripts.ppcPdfReports.working.heavy.shard_4
                ) НАИЛУЧШИЙ (logic_and) статус среди всех машин raw_host (можно указывать
                кондукторные группы).
            Статусы сырых событий являются актуальными в течение ttl секунд
                (600 для *std*, 1200 - для heavy), после чего событие превращается
                в 'NO_DATA'.
            Аггрегат заводится на промежуточный хост 'aggr_host' (его значение
                определено в корневом playbook'е).
                Пример с переопределением vars:
                    queue=std,heavy<ttl=2400>
                Пример с массовым переопределением (сразу ко всем значениям,
                перечисленным после знака равенства:
                    queue<ttl=2400>=heavy1,heavy2
                ).
            Имена проверок совпадают со значениями raw_events после раскрытия
                переменных (vars).
            При наличии переопределения (одного или нескольких) выбирается
                наибольшее из возможных значений.
        * 1 аггрегированная проверка с именем name (scripts.ppcPdfReports.working)
                поверх 8 предыдущих, выбирающая НАИХУДШИЙ статус (logic_or).
            Время пересчета и жизни получившегося результата заданы в playbook'е.
            Аггрегат заводится на хост, указанный в --auto-host.
        * проверка, полностью совпадающая с предыдущей, но отправляемая на указанный
            в метаданных host (prod_checks.direct.yandex.ru)

    Из секции monrun будет сгенерировано 2 (одинаковых) аггрегированных проверки:
        Название проверки - такое же, как у события - из метаданных name
            (bs.object_count_heavy)
        Проверка выбирает НАИЛУЧШИЙ (logic_and) статус всех хостов, на которых
            обсчитываются проверки из monrun-секций (заданы в playbook'е)
        Время пересчета и жизни получившегося результата заданы там же.
        Получившаяся проверка будет заведена на хост "по-умолчанию", и на тот, что
            указан явно в метаданных - juggler_host (test_host.direct.yandex.ru)

    Из третьей секции:
        В playbook будет подключен конфиг по пути "--files-path/tasks/complex_checks.yml"
        Если файл отсутствует - будет выдана ошибка.

=cut

use Direct::Modern;

use File::Find;
use File::Slurp;
use Getopt::Long;
use List::MoreUtils qw/uniq/;
use List::Util qw/max/;
use YAML ();

use ScriptsMetadata;
use Yandex::ListUtils qw/xsort/;
use Yandex::ScriptDistributor qw/juggler_vars/;

use open qw(:std :utf8);


my ($CONF_PATH, $FILES_PATH, $SHARDS_NUM, $AUTO_HOST, $AUTO_TAG, $MANUAL_TAG);
my (@SCRIPTS_PATH, @FILE_EXTENSIONS);
GetOptions(
    'scripts-path=s' => \@SCRIPTS_PATH,
    'scripts-ext=s' => \@FILE_EXTENSIONS,
    'conf-path=s' => \$CONF_PATH,
    'files-path=s' => \$FILES_PATH,
    'auto-tag=s' => \$AUTO_TAG,
    'manual-tag=s' => \$MANUAL_TAG,
    'auto-host=s' => \$AUTO_HOST,
    'shards-num=i' => \$SHARDS_NUM,
    'default-juggler-name-template=s' => \$ScriptsMetadata::DEFAULT_JUGGLER_NAME_TEMPLATE,
    'default-juggler-raw-events-template=s' => \$ScriptsMetadata::DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE,
    'default-juggler-raw-host=s' => \@ScriptsMetadata::DEFAULT_JUGGLER_RAW_HOST,
    'juggler-check-raw-host=s' => \$ScriptsMetadata::JUGGLER_CHECK_RAW_HOST,
    'script-distributor-conf-path=s' => \$Yandex::ScriptDistributor::CONF_FILE_PATH,
) or die $@;

if ($SHARDS_NUM) {
    $Yandex::ScriptDistributor::SHARDS_NUM_REF = \$SHARDS_NUM;
}

my %_nice_keys_order = (
    juggler_check   => 1,
    args            => 2,
    with_items      => 3,
    tags            => 4,
    service         => 1,
    monrun_check    => 2,
    ttl             => 3,
    include         => 1,
    include_vars    => 1,
);

if ($CONF_PATH) {
    die "conf-path $CONF_PATH doesn't exists!" unless -d $CONF_PATH;
} else {
    die 'conf-path is required';
}

sub error_msg {
    if ($ENV{POD2JUGGLER_STRICT_MODE}) {
        die @_;
    } else {
        warn @_;
    }
}

$MANUAL_TAG //= 'admin';
$AUTO_TAG //= 'auto';
@FILE_EXTENSIONS = ('.pl') unless scalar @FILE_EXTENSIONS;

my $exts = join '|', map { s/\./\\./r } @FILE_EXTENSIONS; 
my $regex_file_ext = qr/(?:$exts)$/;

my @files;
find(
    sub {
        if ($regex_file_ext && /($regex_file_ext)$/ && -f $File::Find::name) {
            push @files, $File::Find::name;
        }
    },
    @SCRIPTS_PATH
);

my (%perl_vars, @perl_tasks, %python_vars, @python_tasks);
push @perl_tasks, {
    _order => 0,
    include_vars => 'vars/perl_checks_auto.yml',
    tags => [$AUTO_TAG, $MANUAL_TAG]
};
push @python_tasks, {
    _order => 0,
    include_vars => 'vars/python_checks_auto.yml',
    tags => [$AUTO_TAG, $MANUAL_TAG]
};
my @monrun_tasks;

for my $file (@files) {
    my $conf = ScriptsMetadata::get_conf($file);
    if ($conf->{juggler}) {
        for my $j (ref $conf->{juggler} eq 'ARRAY' ? @{ $conf->{juggler} } : $conf->{juggler}) {
            my @errors = ScriptsMetadata::parse_and_validate_juggler($j,
                                                                     files_path => $FILES_PATH,
                                                                     shards_num => $SHARDS_NUM,
                                                                     this_file_path => $file,
                                                                    );
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_juggler($j, ($file =~ m/\.py$/
                                                ? (\@python_tasks, \%python_vars)
                                                : (\@perl_tasks, \%perl_vars),
                                             ),
                                         );
        }
    }
    if ($conf->{juggler_check}) {
        for my $j (ref $conf->{juggler_check} eq 'ARRAY' ? @{ $conf->{juggler_check} } : $conf->{juggler_check}) {
            my @errors = ScriptsMetadata::parse_and_validate_juggler_check($j, shards_num => $SHARDS_NUM);
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_juggler_check($j, \@perl_tasks);
        }
    }
    if ($conf->{monrun}) {
        for my $m (ref $conf->{monrun} eq 'ARRAY' ? @{ $conf->{monrun} } : $conf->{monrun}) {
            my @errors = ScriptsMetadata::validate_monrun($m);
            die sprintf("%s: %s", $file, join('; ', @errors)) if @errors;
            append_playbook_with_monrun($m, \@monrun_tasks);
        }
    }
}

# для воспроизводимости результата: сортируем задачи по имени
for my $vars (values %perl_vars, values %python_vars) {
    @$vars = sort { $a->{service} cmp $b->{service} } @$vars;
}
@monrun_tasks = sort { $a->{juggler_check} cmp $b->{juggler_check} } @monrun_tasks;
@perl_tasks = _nice_tasks(@perl_tasks);
@python_tasks = _nice_tasks(@python_tasks);

{
    no warnings 'once';
    $YAML::UseAliases = 0;
    $YAML::SortKeys = 1;
}
my $write_opts = {binmode => ':utf8', atomic => 1};
write_file("$CONF_PATH/vars/perl_checks_auto.yml", $write_opts, YAML::Dump({checks => \%perl_vars}));
write_file("$CONF_PATH/vars/python_checks_auto.yml", $write_opts, YAML::Dump({checks => \%python_vars}));
write_file("$CONF_PATH/tasks/scripts_perl_auto.yml", $write_opts, YAML::Dump(\@perl_tasks));
write_file("$CONF_PATH/tasks/scripts_python_auto.yml", $write_opts, YAML::Dump(\@python_tasks));
write_file("$CONF_PATH/tasks/monrun_auto.yml", $write_opts, YAML::Dump(\@monrun_tasks));

exit 0;

sub append_playbook_with_juggler_check {
    my ($j, $playbook_tasks) = @_;

    my @hosts = $AUTO_HOST;
    if ($j->{host}) {
        push @hosts, ref $j->{host} ? @{ $j->{host} } : $j->{host};
    }

    # Добавляем шарды
    if ($j->{sharded} && !defined $j->{vars}->{shard}) {
        die 'shards-num is required for sharded raw_events' unless $SHARDS_NUM && $SHARDS_NUM > 0;
        $j->{vars}->{shard} = { map { $_ => {} } 1..$SHARDS_NUM };
    }

    my $raw_events = ScriptsMetadata::substitute_juggler_vars_and_times($j);
    # в $j->{ttl} может быть умолчание, поэтому его не берем.
    # substitute_juggler_vars_and_times заполнило ttl в каждом $raw_event, берем из них.
    # отсутствие переопределений ttl было проверено в тесте, но лишняя подстраховка не помешает
    my $check_ttl;
    my @uniq_ttl = uniq map { $_->{ttl} } @$raw_events;
    if (@uniq_ttl == 1) {
        $check_ttl = $uniq_ttl[0];
    } else {
        error_msg("invalid ttl for juggler_check '$j->{name}' TTL is not uniue");
        $check_ttl = max(@uniq_ttl);
    }
    # "верхний" - аггрегаты поверх всех проверок сразу (=все должны быть "ок")
    for my $host (@hosts) {
        _check_service_duplicates($j->{name}, $host);
        push @$playbook_tasks, {
            _order => 3,
            _name => $j->{name},
            _level => 1,
            juggler_check => sprintf('%s (aggregated check - %s)',
                                     $j->{name},
                                     ($host eq $AUTO_HOST ? 'auto' : 'admin'),
                                    ),
            args => sprintf('{{ check_template|hash_merge(all_is_ok, %s) }}', $host eq $AUTO_HOST ? 'alert_dev' : 'alert_admin' ),
            with_items => [{
                host => $host,
                service => $j->{name},
                children => [
                    map { "$ScriptsMetadata::JUGGLER_CHECK_RAW_HOST:$_" } sort map { $_->{service} } @$raw_events
                ],
                ttl => $check_ttl,
            }],
            tags => [$host eq $AUTO_HOST ? $AUTO_TAG : $MANUAL_TAG],
        };
    }
}


sub append_playbook_with_juggler {
    my ($j, $playbook_tasks, $playbook_vars) = @_;

    my @hosts = $AUTO_HOST;
    if ($j->{host}) {
        push @hosts, ref $j->{host} ? @{ $j->{host} } : $j->{host};
    }

    if ($j->{for_monrun}) {
        error_msg("for_monrun format is deprecated, skip");
    } elsif ($j->{from_file}) {
        push @$playbook_tasks, {
            _order => 1,
            include => $j->{from_file},
        };
    } else {
        my $vars_name = _get_unique_name($j);
        # Добавляем шарды
        if ($j->{sharded} && !defined $j->{vars}->{shard}) {
            die 'shards-num is required for sharded raw_events' unless $SHARDS_NUM && $SHARDS_NUM > 0;
            $j->{vars}->{shard} = { map { $_ => {} } 1..$SHARDS_NUM };
        }
        # Размножаем экземпляры
        if ($j->{distributed}) {
            my @names = keys %{$j->{vars}->{script_name}};
            die 'only one script name should exist for distributed scripts' unless scalar keys %{$j->{vars}->{script_name}} == 1;
            $j->{vars}->{run_id} = juggler_vars($names[0]);
        }
        # Хостов с сырыми событиями может быть несколько
        $j->{raw_host} = [$j->{raw_host}] if ref $j->{raw_host} ne 'ARRAY';
        # Раскрываем название сырых событий (получаем список childs)
        my $raw_events = ScriptsMetadata::substitute_juggler_vars_and_times($j);
        if (@$raw_events > 1) {
            # Перечисление всех сырых событий (скрипт/очередь/шард/...)
            for my $one_raw_event (@$raw_events) {
                _check_service_duplicates($one_raw_event->{service}, 'aggr_host');
                push @{ $playbook_vars->{$vars_name} }, _nice_yaml_keys_order($one_raw_event);
            }
            # Создаем два слоая проверок:
            # 'нижний' - поверх хостов для каждого сырого события
            push @$playbook_tasks, {
                _order => 2,
                _name => $j->{name},
                _level => 0,
                juggler_check => "$j->{name} (subchecks)",
                args => '{{ subcheck }}',
                with_items => sprintf('checks.%s|expand_services_with_hosts(%s)',
                                      $vars_name,
                                      join(', ', map { "'$_'" } @{ $j->{raw_host} })
                                     ),
                tags => [$AUTO_TAG, $MANUAL_TAG],
            };
            # "верхний" - аггрегаты поверх всех проверок нижнего слоя
            for my $host (@hosts) {
                _check_service_duplicates($j->{name}, $host);
                push @$playbook_tasks, {
                    _order => 2,
                    _name => $j->{name},
                    _level => 1,
                    juggler_check => sprintf('%s (aggregated check - %s)',
                                             $j->{name},
                                             ($host eq $AUTO_HOST ? 'auto' : 'admin'),
                                            ),
                    args => sprintf("{{ %s|hash_merge({'host': '%s'}) }}",
                                    ($host eq $AUTO_HOST ? 'dev_check_over_subchecks' : 'adm_check_over_subchecks'),
                                    $host,
                                   ),
                    with_items => sprintf("checks.%s|aggregate_check_data(host=aggr_host, name='%s')",
                                          $vars_name,
                                          $j->{name},
                                         ),
                    tags => [$host eq $AUTO_HOST ? $AUTO_TAG : $MANUAL_TAG],
                };
            }
        } else {
            # так как выражение только одно - промежуточный уровень аггрегации не требуется
            push @{ $playbook_vars->{$vars_name} }, _nice_yaml_keys_order($raw_events->[0]);
            # создаем проверку сразу на хост "верхнего" (с ответственными) уровня
            for my $host (@hosts) {
                _check_service_duplicates($j->{name}, $host);
                push @$playbook_tasks, {
                    _order => 2,
                    _name => $j->{name},
                    _level => 1,
                    juggler_check => sprintf('%s (aggregated check - %s)',
                                             $j->{name},
                                             ($host eq $AUTO_HOST ? 'auto' : 'admin'),
                                            ),
                    args => sprintf("{{ %s|hash_merge({'host': '%s'}) }}",
                                    ($host eq $AUTO_HOST ? 'dev_check_single' : 'adm_check_single'),
                                    $host,
                                   ),
                    with_items => sprintf('checks.%s|expand_services_with_hosts(%s)',
                                          $vars_name,
                                          join(', ', map { "'$_'" } @{ $j->{raw_host} })
                                         ),
                    tags => [$host eq $AUTO_HOST ? $AUTO_TAG : $MANUAL_TAG],
                };
            }
        }
    }
}

sub append_playbook_with_monrun {
    my ($m, $playbook_tasks) = @_;

    my @hosts = $AUTO_HOST;
    if ($m->{juggler_host}) {
        push @hosts, ref $m->{juggler_host} ? @{ $m->{juggler_host} } : $m->{juggler_host};
    }

    for my $host (@hosts) {
        _check_service_duplicates($m->{name}, $host);
        push @$playbook_tasks, _nice_yaml_keys_order({
            juggler_check => sprintf('%s (check over monrun - %s)',
                                     $m->{name},
                                     ($host eq $AUTO_HOST ? 'auto' : 'admin'),
                                    ),
            args => sprintf("{{ %s|hash_merge({'host': '%s'}) }}",
                            ($host eq $AUTO_HOST ? 'dev_check_monrun' : 'adm_check_monrun'),
                            $host,
                           ),
            with_items => $m->{name},
            tags => [$host eq $AUTO_HOST ? $AUTO_TAG : $MANUAL_TAG],
        });
    }
}

sub _nice_yaml_keys_order {
    my $hash = shift;
    YAML::Bless($hash)->keys([ sort { ($_nice_keys_order{$a}||99) <=> ($_nice_keys_order{$b}||99) } keys %$hash ]);
    return $hash;
}

sub _nice_tasks {
    return map {
        _nice_yaml_keys_order($_)
    } map {
        delete @{ $_ }{qw/ _level _name _order /};
        $_;
    } xsort {
        int($_->{_order} // 5), ($_->{_name} // $_->{juggler_check} // $_->{include} // ''), (int($_->{_level} // 1))
    } @_;
}

sub _get_unique_name {
    my ($conf) = @_;
    state %duplicates;  # name -> count

    my $new_name = $conf->{name} =~ s/\./_/gr;
    if (++$duplicates{$new_name} > 1) {
        error_msg("\n###### DUPLICATE NAME FOR CHECK: '$conf->{name}' ######\n");
        $new_name = sprintf('%s__%d', $new_name, $duplicates{$new_name});
    }
    return $new_name;
}

=head3 _check_service_duplicates($service, $host)

    Проверяет, что $service является уникальным в пределах $host.
    При дублировании - выдает warning.
    Хорошо бы падать, потому что собранный playbook - невалиден.

=cut

sub _check_service_duplicates {
    my ($service, $host) = @_;
    state %duplicates;  # host -> service -> count

    if (++($duplicates{$host}->{$service}) > 1) {
        error_msg("\n###### DUPLICATE SERVICE '$service' FOR HOST: '$host' ######\n");
    }

}
