#!/usr/bin/perl

=head1 NAME

pod2monrun.pl

=head1 DESCRIPTION

Ищет скрипты в каталогах --scripts-path (необходимо указывать абсолютный путь),
выбирает из них pod-секции (м.б. несколько) METADATA и собирает файл --conf-path

=head2 Опции

    --send-for-host -- хост по умолчанию, для которого отправлять проверки
    --shard-num -- количество шардов

=head2 Шардирование

Поддерживаются параметры sharded, only_shards.
При истинном значении sharded поле expression должно содержать подстроку \$shard\b

Пример:

    <monrun>
        name: check
        expression: 'movingAverage(system.shard_$shard.metric, "30min")'
        crit: 1000
        sharded: 1
        only_shards: 1,3
        vars: par=easy1,easy2,heavy1
    </monrun>

Секция будет превращена в аггрегатную проверку

    graphite_threshold -crit 1000 -v par=easy1,easy2,heavy1 -v shard=1,3 'movingAverage(system.shard_$shard.metric_$par, "30min")'

соответственно.

=cut

use strict;
use warnings;
use utf8;
use open qw(:std :utf8);

use File::Find;
use File::Slurp;
use Getopt::Long;
use ScriptsMetadata;

my ($CONF_PATH, @SCRIPTS_PATH, $SEND_FOR_HOST, $SHARD_NUM, @FILE_EXTENSIONS);
GetOptions (
    "conf-path=s" => \$CONF_PATH,
    "h|help" => \&ScriptsMetadata::usage,
    "scripts-path=s" => \@SCRIPTS_PATH,
    "send-for-host=s" => \$SEND_FOR_HOST,
    "shard-num=i" => \$SHARD_NUM,
    "file-ext=s" => \@FILE_EXTENSIONS,
) or die $@;

my @pl_files;
@FILE_EXTENSIONS = ("\\.pl") unless scalar @FILE_EXTENSIONS;

my $regex_file_ext = join "|", map { qr/$_/ } @FILE_EXTENSIONS;

find(
    sub {
        if ($regex_file_ext && /($regex_file_ext)$/ && -f $File::Find::name) {
            push @pl_files, $File::Find::name;
        }
    },
    @SCRIPTS_PATH
);

my @result;
for my $file (@pl_files) {
    my $conf = ScriptsMetadata::get_conf($file);
    next unless $conf->{monrun};

    my $monrun = ref $conf->{monrun} eq 'ARRAY' ? $conf->{monrun} : [$conf->{monrun}];
    if (my @errors = map {ScriptsMetadata::validate_monrun($_)} @$monrun) {
        die join "\n", $file, @errors;
    }
    push @result, @$monrun;
}

my $output = join "\n", map {generate_config($_)} sort {$a->{name} cmp $b->{name}} @result;
write_file($CONF_PATH, {binmode => ':utf8', atomic => 1}, $output);


sub generate_config
{
    my $conf = shift;

    my $send_for_host = $conf->{send_for_host} || $SEND_FOR_HOST || '';

    my $opt = '';
    for my $param (qw/crit span warn/) {
        if (defined $conf->{$param}) {
            $opt .= " -$param $conf->{$param}";
        }
    }
    if ($conf->{vars}) {
        $opt .= " -v $conf->{vars}";
    }
    if (my @shards = ScriptsMetadata::get_shards($conf, shard_num => $SHARD_NUM)) {
        $opt .= " -v shard=".join(',', @shards);
    }

    my $execution_timeout = 20;
    if ($conf->{timeout}) {
        $execution_timeout = $conf->{timeout};
    }
    my $execution_interval = 60;
    if ($conf->{interval}) {
        $execution_interval = $conf->{interval};
    }

    return _template(
        expression => $conf->{expression},
        name => $conf->{name},
        opt => $opt,
        send_for_host => $send_for_host,
        execution_timeout => $execution_timeout,
        execution_interval => $execution_interval,
        );
}


sub _template
{
    my %O = @_;
    my $send_for_host_str = $O{send_for_host} ne "" ? "send_for_host=$O{send_for_host}\n" : "";
    return <<EOS;
[$O{name}]
${send_for_host_str}execution_interval=$O{execution_interval}
execution_timeout=$O{execution_timeout}
command=/usr/local/bin/graphite_threshold$O{opt} $O{expression}
EOS
}
