#!/usr/bin/perl -w

=head1 NAME

    pod2ubic_service.pl -- сборка ubic-описание сервисов для запуска

=head1 DESCRIPTION

    pod2ubic_service.pl [OPTION]
        --module-paths директории с модулями (*.pm файлы)
        --files непосредственный список модулей (опционально)
        --ubic путь к директории с описание ubic сервисов
        --package путь + имя пакета для создания .prerm и .postinst файлов
        --log-base папка для хранения STDOUT STDERR запущенного скрипта(демона)
        --user от какого пользователя ubic будет запусть скрипты(демоны)
        --worker-base путь до gearman-воркера
        --validate провалидировать описание сервиса и вывести результат в STDOUT
        --help - эта справка        

=cut

use strict;
use warnings;
use utf8;

use File::Find;
use File::Slurp qw/write_file/;
use List::MoreUtils qw/uniq/;
use File::Path;
use Getopt::Long;
use ScriptsMetadata;


my ($user, @paths, @files, $worker_base, $log_base, $ubic, $only_validate, $package);

GetOptions(
    'module-paths=s{,}' => \@paths,
    'files:s{,}' => \@files,
    'ubic=s' => \$ubic,
    'package=s' => \$package,
    'user=s' => \$user,
    'worker-base=s' => \$worker_base,
    'log-base=s' => \$log_base,
    'validate' => \$only_validate,
    'help' => \&ScriptsMetadata::usage
);

ScriptsMetadata::usage() unless (@paths || @files) && $ubic && $user && $worker_base && $log_base;
s/(\/+)$// foreach (@paths, $ubic, $worker_base, $log_base);

use constant SERVICE_KINDS => {
    daemon => \&daemon_template,
    server_starman => \&server_starman_template,
};

find (sub {
  push @files, $File::Find::name if /\.pm$/ && -f $_
}, @paths) if @paths;


my @services = services(@files);
warn 'services empty' unless @services;
write_ubic_service(\@services, 
    user => $user,
    base => $worker_base,
    log => $log_base,
    ubic => $ubic,
    only_validate => $only_validate,
);
write_service_package($package, \@services) if $package;


sub services {

    my @files = @_;

    my %services;
    foreach my $f (@files) {
        my $conf = ScriptsMetadata::get_conf($f);

        foreach my $service_type (qw/ubic gearman/) {
            next unless exists $conf->{$service_type};

            if (my @errors = ScriptsMetadata::validate_ubic($conf->{$service_type}, kinds => SERVICE_KINDS)) {
                die join "\n", $f, $service_type, @errors;
            }

            my $ubic_params = default_params($conf->{$service_type}, $f);
            $services{$ubic_params->{module}} = $ubic_params;
        }
    }
    
    return values %services;
}

sub write_ubic_service {
    
    my ($services, %options) = @_;

    foreach my $s (@$services) {
        if ($options{only_validate}) {
            print "$options{ubic}/$s->{ubic_path}", ":\n", SERVICE_KINDS->{$s->{kind}}->($s, %options);
        } else {

            my $dirpath = $s->{ubic_path};
            $dirpath =~ s/([^\/]*?)$//;
            mkpath("$options{ubic}/$dirpath") if $dirpath;

            write_file("$options{ubic}/$s->{ubic_path}", {
                binmode => ':utf8',
                atomic => 1
            }, SERVICE_KINDS->{$s->{kind}}->($s, %options));
        }
    }
}

sub write_service_package {
    
    my ($package, $services) = @_;

    my $names = join ' ', uniq map { [split '/', $_->{ubic_path}]->[0]  } @$services;
    my %sorts = (prerm => 'stop', postinst => 'restart');
    foreach my $type (keys %sorts) {
        my $pkg = sprintf "#!/bin/bash -e\n\nubic -f %s %s\n", $sorts{$type}, $names;
        mkpath "$package/DEBIAN";
        write_file("$package/DEBIAN/${type}" , {
            binmode => ':utf8',
            atomic => 1,
            perms => 0755,
        }, $pkg);
        
        # на hardy 12-я версия File::Slurp не умеет perms. Избавиться после перехода на precise.
        chmod(0755, "$package/DEBIAN/${type}");

    }
}

sub default_params {
    
    my ($gearman, $file) = @_;
    
    my $module = $gearman->{module} || [$file =~ /([^\/]+)\.pm$/]->[0];
    my %config = %$gearman;
    if ($module) {
        $config{module} = $module unless $config{module};
        (my $path = $module) =~ s/\:\:/\//g;
        $config{ubic_path} = $path;
        $module =~ s/\:\:/_/g;
        $config{stdout} = "$module.out" unless $config{stdout};
        $config{stderr} = "$module.err" unless $config{stderr};
        $config{server_name} = $module unless $config{server_name};
    }
    $config{child} = 1 unless $config{child} && $config{child} =~ /^\d+$/;
    
    return \%config;
}

sub daemon_template {

    my ($service, %options) = @_;

    my @params = (
        $service->{user} || $options{user},
        $options{base},
        @{$service}{qw/module child/},
        @options{qw/base log/},
        $service->{stdout},
        $options{log},
        $service->{stderr},
    );

    sprintf <<EOS, @params;
use Ubic::Service::SimpleDaemon;

Ubic::Service::SimpleDaemon->new({
    user => '%s',
    bin => '%s/gearman_worker.pl --module=%s --child=%i',
    cwd => '%s',
    stdout => '%s/%s',
    stderr => '%s/%s'
});
EOS

}

sub server_starman_template {

    my ($service, %options) = @_;

    my @params = (
        $service->{user} || $options{user},
        $options{base}, $service->{psgi},
        $service->{child},
        $service->{port},
        $options{log}, $service->{server_name},
        $options{log}, $service->{stdout},
        $options{log}, $service->{stderr},
        $service->{server_name}
    );

    sprintf <<EOS, @params;
use Ubic::Service::Starman;

return Ubic::Service::Starman->new({
    user => "%s",
    app => "%s/%s",
    server_args => {
        workers => %s
    },
    port => %s,
    ubic_log => '%s/%s',
    stdout => '%s/%s',
    stderr => '%s/%s',
    pidfile => '/tmp/%s.pid'
});
EOS

}
    
# пример    
#=head1 METADATA
#
#<gearman>
#    kind: daemon
#    child: 3
#    module: Listener
#</gearman>
#
#=cut
