package ScriptsMetadata;

use strict;
use warnings;
use utf8;

use Digest::MD5;
use File::Slurp;
use List::MoreUtils qw(none);
use List::Util qw/min max/;
use Storable qw/dclone/;

use Yandex::Metadata;
use Yandex::ScriptDistributor qw/get_run_count/;

=head2 $MEM_NAME_IN_META

    Имя арендуемого ресурса в метаданных switchman, который трактуется как "оперативная память".
    Умолчание - FQDN_mem.

=cut

our $MEM_NAME_IN_META //= 'FQDN_mem';

=head2 $MEM_SWITCHMAN_LEASE

    Имя арендуемого ресурса в параметрах к switchman, который трактуется как "оперативная память".
    Умолчание - FQDN_mem.

=cut

our $MEM_SWITCHMAN_LEASE //= 'FQDN_mem';

=head2 $CPU_NAME_IN_META

    Имя арендуемого ресурса в метаданных switchman, который трактуется как "ядра процессора".
    Умолчание - FQDN_cpu.

=cut

our $CPU_NAME_IN_META //= 'FQDN_cpu';

=head2 $CPU_SWITCHMAN_LEASE

    Имя арендуемого ресурса в параметрах к switchman, который трактуется как "ядра процессора".
    Умолчание - FQDN_cpu.

=cut

our $CPU_SWITCHMAN_LEASE //= 'FQDN_cpu';

=head1 Juggler-параметры

    Описания констант, значения которых используются в качестве умолчаний при разборе метаданных

=head2 $DEFAULT_JUGGLER_NAME_TEMPLATE

    Шаблон имени события (name) по умолчанию для juggler-проверки.

=cut

our $DEFAULT_JUGGLER_NAME_TEMPLATE;

=head2 $DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE

    Шаблон шаблона выражения для имен сырых (raw_events) juggler-событий.

=cut

our $DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE;

=head2 @DEFAULT_JUGGLER_RAW_HOST

    Список хостов или кондукторных групп над которыми нужно агрегировать сырые события.

=cut

our @DEFAULT_JUGGLER_RAW_HOST;

=head2 $JUGGLER_CHECK_RAW_HOST

    Хост для сырых событий из juggler_check

=cut

our $JUGGLER_CHECK_RAW_HOST;

=head2 $JUGGLER_CHECK_HOST_IS_MANDATORY

    Требовать обяательного указания host в <juggler> и <juggler_host> в monrun.
    По умолчанию выключено

=cut

our $JUGGLER_CHECK_HOST_IS_MANDATORY //= 0;

=head2 DEFAULT_JUGGLER_TTL

    Интервал прошедшего времени в секундах, за который берутся сырые данные (raw events)
        для вычисления состояния проверки внутри Juggler (если данные поступили раньше,
        то Juggler считает, что данных нет).
    Соответствует указанию "ttl: 600" в метаданных

=cut

use constant DEFAULT_JUGGLER_TTL => 600;

=head2 MAX_JUGGLER_TTL

    Максимально возможное значение ttl (10 дней). При попытке указания большего значения
        API juggler'а отвечает ошибкой.

=cut

use constant MAX_JUGGLER_TTL => 3600 * 24 * 10;

sub get_conf
{
    my $file = shift;
    my $text = read_file($file, binmode => ':utf8');
    my $crontab_pod = extract_metadata($file => $text);

    my $conf = eval {deserialize_metadata($crontab_pod)};
    die "$file $@" if $@;
    return $conf;
}

=head2 process_multiplicators($conf, %O)

Обработка полей $conf:

    sharded
    only_shards
    distributed

Опции:

    shard_num -- число шардов

    возвращает список хешей параметров, которые надо передать скрипту 

=cut

sub process_multiplicators
{
    my ($conf, %O) = @_;

    my @shard_params; 
    my @shards = get_shards($conf, %O);
    if (@shards) {
        @shard_params = ( map { {shard => $_} } @shards );
    } else {
        @shard_params = ( {} );
    }

    my @run_id_params;
    if (exists $conf->{distributed}) {
        die "did not get a script name" unless $O{script_name};
        my $name = $O{script_name};
        $name =~ s/.*\///g; # обрезать слева до последнего слеша
        $name =~ s/\.[^\.]+$//; # и справа до первой точки
        my $run_count = get_run_count($name);
        @run_id_params = map {{run_id => "$_"}} (1 .. $run_count);
    } else {
        @run_id_params = ( {} );
    }

    my @params;
    for my $sh ( @shard_params ){
        for my $r ( @run_id_params ){
            push @params, { %$sh, %$r };
        }
    }

    return @params;
}


=head2 get_shards($conf, %O)

Получить список шардов, релевантных $conf

Обработка полей $conf:

    sharded
    only_shards

Опции:

    shard_num -- число шардов

=cut

sub get_shards
{
    my ($conf, %O) = @_;
    my @shards;
    if ($conf->{sharded} && $O{shard_num}) {
        if ($conf->{only_shards}) {
            @shards = grep {$_} split /\D+/, $conf->{only_shards};
            for my $shard (@shards) {
                die "shard $shard is out of bounds 1 .. $O{shard_num}" if $shard < 1 || $shard > $O{shard_num};
            }
        } else {
            @shards = (1 .. $O{shard_num});
        }
    }
    return @shards;
}


sub usage
{
    system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}


sub validate_crontab
{
    my ($conf, %O) = @_;

    my @errors;
    my %allowed = map {$_ => 1} qw/
        time
        run
        package
        user
        params
        env
        ulimit
        sharded
        switchman
        only_shards
        params_postfix
        distributed
        flock
    /;
    for my $f (grep {!$allowed{$_}} keys %$conf){
        push @errors, "field '$f' isn't allowed in crontab metadata";
    }

    if ($conf->{time}) {
        my @time_fields = split /\s+/, $conf->{time};
        push @errors, "expected 5 fields in 'time' value ($conf->{time})" unless @time_fields == 5;
        for my $f (@time_fields){
            push @errors, "incorrect time field $f" unless $f =~ m!^([\d,\-]+|\*)(/\d+)?$!;
        }
    } else {
        push @errors, "'time' value missed";
    }
    push @errors, "empty package name" if !$O{package_prefix} && !$conf->{package};

    if ($conf->{switchman}){
        # типовое значение для switchman_leases => [$MEM_NAME_IN_META, $CPU_NAME_IN_META]
        push @errors, validate_switchman($conf->{switchman}, valid_groups => $O{switchman_groups}, valid_leases => $O{switchman_leases});
    }

    if ($O{switchman_packages} && exists $conf->{switchman} && none {($conf->{package} // '') eq $_} @{$O{switchman_packages}}) {
        push @errors, 'package for scripts running under switchman should be one of ' . join(', ', @{$O{switchman_packages}});
    }

    if (exists $conf->{sharded} && exists $conf->{distributed}) {
        push @errors, '"sharded" and "distributed" are mutually exclusive';
    }

    return @errors;
}

sub validate_switchman {
    my ($sw, %params) = @_;
    my @errors;

    my %allowed = map {$_ => 1} qw/
        group
        lockname
        lockname_with_env
        leases
        delay
    /;
    for my $f (grep {!$allowed{$_}} keys %$sw){
        push @errors, "field '$f' isn't  allowed in switchman metadata";
    }

    for my $k (keys %$sw) {
        if ($sw->{$k} =~ /\s/) {
            push @errors, "value in field '$k' contains space bars";
        }
    }

    if ($params{valid_groups}
        && exists $sw->{group}
        && none {($sw->{group} // '') eq $_} @{$params{valid_groups}}
    ) {
        push @errors, 'switchman group should be one of ' . join(', ', @{$params{valid_groups}});
    }

    if ($params{valid_leases}) {
        my %allowed_leases;
        for my $lease (@{ $params{valid_leases} }) {
            $allowed_leases{$lease} = 1;
        }

        for my $l (grep {!$allowed_leases{$_}} keys %{ $sw->{leases} // {} }){
            push @errors, "lease '$l' isn't  allowed in switchman metadata";
        }
    }

    return @errors;
}


sub validate_monrun
{
    my $conf = shift;

    my @errors;
    my %allowed = map {$_ => 1} qw/
        name
        send_for_host
        
        juggler_host
        tag
        notification
        meta_url

        span
        expression
        warn
        crit

        sharded
        only_shards

        vars

        timeout
        interval
    /;
    for my $f (grep {!$allowed{$_}} keys %$conf){
        push @errors, "field '$f' isn't allowed in monrun metadata";
    }
    my @mandatory = qw/name expression/;
    push @mandatory, 'juggler_host' if $JUGGLER_CHECK_HOST_IS_MANDATORY;
    for my $required (@mandatory) {
        push @errors, "$required is required" unless $conf->{$required};
    }
    push @errors, 'name contains invalid characters (allowed [0-9a-zA-Z\-\._])' if _validate_name_or_host($conf->{name});
    push @errors, 'crit or warn is required' if !defined $conf->{crit} && !defined $conf->{warn};
    push @errors, 'crit must be number' if _validate_crit_or_warn($conf->{crit});
    push @errors, 'warn must be number' if _validate_crit_or_warn($conf->{warn});
    push @errors, 'only_shards requires sharded' if $conf->{only_shards} && !$conf->{sharded};
    push @errors, 'expression must contain $shard if sharded' if $conf->{sharded} && $conf->{expression} !~ /\$shard\b/;
    push @errors, "incorrect format of vars: '$conf->{vars}'" if $conf->{vars} && $conf->{vars} !~ /^ \w+ = \w+ (?: , \w+ )* $/x;
    push @errors, 'expression with $shard requires sharded or only_shards' if $conf->{expression} =~ /\$shard\b/ && !$conf->{sharded} && !$conf->{only_shards};
    push @errors, 'incorrect span value' if defined $conf->{span} && $conf->{span} !~ /^\d+(?:s|min|h|d|w|mon|y)$/;
    push @errors, 'incorrect timeout value' if defined $conf->{timeout} && ($conf->{timeout} !~ m/^[0-9]+$/ || $conf->{timeout} > 120);
    push @errors, 'incorrect interval value' if defined $conf->{interval} && ($conf->{interval} !~ m/^[0-9]+$/ || $conf->{interval} < 60 || $conf->{interval} > 1200);

    if ($conf->{juggler_host}) {
        for my $h ( ref $conf->{juggler_host} ? @{ $conf->{juggler_host} } : $conf->{juggler_host} ) {
            push @errors, "juggler_host '$h' is invalid" if _validate_name_or_host($h);
        }
    }

    return @errors;
}

sub validate_ubic
{
    my ($conf, %O) = @_;

    my @errors;
    push @errors, "undefined worker kind" unless $conf->{kind} && exists $O{kinds}->{$conf->{kind}};
    return @errors;
        
}

sub _validate_name_or_host {
    my $string = shift;
    return defined $string && $string !~ /^[a-zA-Z0-9_\-\.]+$/;
}
sub _validate_crit_or_warn {
    my $num = shift;
    return defined $num && $num !~ /^[-+]?[0-9]+(\.[0-9]+)?$/;
}


=head2 randomized_time($time, $key, $idx, $cnt)

    Рандомизация запуска скриптов для более равномерного распределения нагрузки
    $time - в формате crontab, обрабатывается только вид */NNN ...
    $key - ключ для консистентности
    $idx - порядковый номер записи с таким ключём
    $cnd - общее количество записей с таким ключём

=cut
sub randomized_time {
    my ($time, $key, $idx, $cnt) = @_;
    
    # парсим и проверяем строку
    my ($period, $rest) = $time =~ /^ \* \/ (\d+) (\s.*) /x
        or return $time;

    # 32-x битный неслучайный "random"
    # crc32 подошёл бы лучше, но его нет в стандартной поставке перла
    my $seed = unpack "L", Digest::MD5::md5($key // '0');

    my $start_min = $seed % $period;
    if (defined $idx && defined $cnt && $cnt > 1) {
        $start_min = ($start_min + int($period * $idx / $cnt + 0.5)) % $period;
    }

    return "$start_min-59/$period$rest";
}

=head2 substitute_juggler_vars_and_times($j)

    Сгенерировать массив данных для дочерних проверок.

    Параметры:
        $j - хеш с метаданными <juggler> (после обработки parse_and_validate)
    Результат:
        [
            {
                service => '',          # выражение для подпроверки. получилось из $j->{raw_events}
                                        # подстановкой переменных из $j->{vars}
                ttl => NN,              # ttl подпроверки, смотри DEFAULT_JUGGLER_TTL
            }, ...
        ]


=cut

sub substitute_juggler_vars_and_times {
    my $j = shift;
    my @expressions;

    for my $one_raw_events (@{ $j->{raw_events} }) {
        push @expressions, { service => $one_raw_events, ttl => _human2second($j->{ttl}) };
    }

    for my $varname (keys %{ $j->{vars} }) {
        my @new;
        for my $expr (@expressions) {
            if ($expr->{service} =~ /\$\Q$varname\E/) {
                for my $val (keys %{ $j->{vars}->{$varname} }) {
                    # берем из переопределений (если есть) или общую настройку
                    my $val_ttl = _human2second($j->{vars}->{$varname}->{$val}->{ttl}//$j->{ttl});
                    push @new, {
                                    service => $expr->{service} =~ s/\$\Q$varname\E/$val/gr,
                                    ttl => min(MAX_JUGGLER_TTL, max($expr->{ttl}, $val_ttl)),
                                };
                }
            } else {
                push @new, $expr;
            }
        }
        @expressions = @new;
    }
    return \@expressions;
}

=head2 parse_and_validate_juggler_check

    Почти как parse_and_validate_juggler, но для частного случая - когда raw_events заданы и все на одном сыром хосте

=cut

sub parse_and_validate_juggler_check {
    my $conf = shift;
    my %O = @_;

    if (!$JUGGLER_CHECK_RAW_HOST) {
        return '$ScriptsMetadata::JUGGLER_CHECK_RAW_HOST is not defined';
    }

    my @mandatory = qw/
        name
        raw_events
    /;
    push @mandatory, 'host' if $JUGGLER_CHECK_HOST_IS_MANDATORY;
    my %allowed = map {$_ => 1} (
        @mandatory,
        qw/
            host
            sharded
            vars
            ttl
            tag
            notification
            meta_url
        /,
    );

    my @errors;

    for my $f (grep { !$allowed{$_} } keys %$conf) {
        push @errors, "field '$f' isn't allowed in juggler_check metadata";
    }
    for my $f (grep { !$conf->{$_} } @mandatory) {
        push @errors, "field '$f' is mandatory in juggler_check metadata";
    }

    $conf->{raw_host} = [$JUGGLER_CHECK_RAW_HOST];

    push @errors, parse_and_validate_juggler($conf, shards_num => $O{shards_num}, disallow_overides => 1);

    return @errors;
}

=head2 parse_and_validate_juggler($conf, %options)

    Распарсить, преобразовать и проверить конфигурацию проверки

    Параметры позиционные:
        $conf - хеш с метаданными <juggler>. МОДИФИЦИРУЕТСЯ in-place
    Параметры именованные:
        files_path =>   '', # строка с абсолютным путем до директории с собираемым
                            # playbook (в котором поддиректория tasks), используется
                            # для проверки существования подключаемых через from_file файлов
        shards_num => N,    # число шардов. используется для раскрытия переменной $shard в raw_events
        this_file_path => , # строка с именем текущего файла. Используется при попытки автоматического
                            # определения значений name, raw_events
        disallow_overides =># булев, запрещает задание переопределений (ttl)
    Результат:
        @errors     - массив строк с описаниями ошибок в конфигурации

=cut

sub parse_and_validate_juggler {
    my $conf = shift;
    my %O = @_;

    # пробуем определить имя скрипта
    my $script_name;
    if ($O{this_file_path} && $O{this_file_path} =~ m/\.p[ly]$/) {
        $script_name = $O{this_file_path};
        $script_name =~ s/\.p[ly]$//; # remove .pl
        $script_name =~ s|^.+/||;  # remove path
    }

    my @errors;
    my %allowed = map {$_ => 1} qw/
        name
        host
        raw_host
        raw_events
        sharded
        distributed
        vars
        ttl
        tag
        notification
        meta_url
    /;

    if ($conf->{for_monrun}) {
        # DEPRECATED
    } elsif ($conf->{from_file}) {
        for my $f (grep {$_ ne 'from_file'} keys %$conf) {
            push @errors, "field '$f' isn't allowed in juggler metadata (with from_file option)";
        }
        if ($O{files_path}) {
            unless (-f "$O{files_path}/tasks/$conf->{from_file}") {
                push @errors, "file $conf->{from_file} doesn't exist in $O{files_path}/tasks/";
            }
        } else {
            push @errors, "files_path option is not specified, can't check file existence";
        }
    } else {
        # умолчания
        $conf->{name} //= _default_juggler_name_from_path($conf, $script_name);
        $conf->{raw_host} //= dclone(\@DEFAULT_JUGGLER_RAW_HOST);
        $conf->{ttl} //= DEFAULT_JUGGLER_TTL;

        for my $f (grep { !$allowed{$_} } keys %$conf) {
            push @errors, "field '$f' isn't allowed in juggler metadata";
        }

        push @errors, 'name contains invalid characters (allowed [0-9a-zA-Z\-\._])' if _validate_name_or_host($conf->{name});
        push @errors, 'raw_host is required' unless $conf->{raw_host} || ref $conf->{raw_host} eq 'ARRAY' && @{ $conf->{raw_host} };
        if ($JUGGLER_CHECK_HOST_IS_MANDATORY) {
            push @errors, 'host is required' unless $conf->{host} || ref $conf->{host} eq 'ARRAY' && @{ $conf->{host} };
        }

        for my $f (qw/host raw_host raw_events vars/) {
            if ($conf->{$f}) {
                $conf->{$f} = [$conf->{$f}] if ref $conf->{$f} ne 'ARRAY';
            } else {
                $conf->{$f} = [];
            }
        }

        for my $h (@{ $conf->{raw_host} }) {
            push @errors, "raw_host '$h' is invalid" unless $h =~ m/^(?:CGROUP%)?[a-zA-Z0-9_\-\.]+$/;
        }
        for my $h (@{ $conf->{host} }) {
            push @errors, "host '$h' is invalid" if _validate_name_or_host($h);
        }

        my (%parsed_vars, %used_vars);
        for my $v (@{ $conf->{vars} }) {
            if ($v =~ qr/^[a-zA-Z0-9_]+=    # имя переменной и знак равенства
                        (?:
                            (?:(?<!=),)?    # начало списка или запятая после предыдущего значения
                            [a-zA-Z0-9_]+   # возможное значение переменной
                            (?:<            # опциональный блок переопределения ttl
                                (?:
                                    (?:(?<!<);)?    # начало блока или ';' после предыдущего переопределения
                                    (?:ttl)=[0-9dhms]+
                                    (?=[>;])        # конец блока или ';' перед следующим переопределением
                                ){1}        # одно переопределение
                            >)?
                            (?=(?:,|$))     # конец строки или запятая перед еще одним значением
                        ){1,}$/x
            ) {
                my ($key, $values_string) = split(qr/=/, $v, 2);
                for my $one_value_string (split(qr/,/, $values_string)) {
                    my ($value, $overrides_string) = ($one_value_string =~ m/^([a-zA-Z0-9_]+)(?:<(.*)>)?$/);
                    if (exists $parsed_vars{$key}->{$value}) {
                        push @errors, "duplicate value '$value' for '$key' in vars: '$v'";
                    } else {
                        $parsed_vars{$key}->{$value} = {};
                        if ($overrides_string) {
                            if ($O{disallow_overides}) {
                                push @errors, "overrides is not allowed";
                            } elsif ($key ne 'shard') {
                                for my $override_string (split(qr/;/, $overrides_string)) {
                                    my ($override_key, $override_value) = split(qr/=/, $override_string, 2);
                                    if (exists $parsed_vars{$key}->{$value}->{$override_key}) {
                                        push @errors, "duplicate override '$override_key' for '$key' '$value' in vars: '$v'";
                                    } else {
                                        $parsed_vars{$key}->{$value}->{$override_key} = $override_value;
                                    }
                                }
                            } else {
                                push @errors, "overrides is not allowed for 'shard' variable";
                            }
                        }
                    }
                }
            } elsif ($v =~ qr/^([a-zA-Z0-9_]+)  # имя переменной
                              <(                # начало блока переопределения ttl
                                (?:
                                    (?:(?<!<);)?    # начало блока или ';' после предыдущего переопределения
                                    (?:ttl)=[0-9dhms]+
                                    (?=[>;])        # конец блока или ';' перед следующим переопределением
                                ){1}            # одно переопределение
                              )>                # конец переопределения
                              =                 # знак равенства перед перечислением значений
                              ((?:
                                (?:(?<!=),)?    # начало списка или запятая после предыдущего значения
                                [a-zA-Z0-9_]+   # возможное значение переменной
                                (?=(?:,|$))     # конец строки или запятая перед еще одним значением
                              ){1,})$/x
            ) {
                my ($key, $overrides_string, $values_string) = ($1, $2, $3);
                for my $value (split(qr/,/, $values_string)) {
                    if (exists $parsed_vars{$key}->{$value}) {
                        push @errors, "duplicate value '$value' for '$key' in vars: '$v'";
                    } else {
                        $parsed_vars{$key}->{$value} = {};
                        if ($overrides_string) {
                            if ($O{disallow_overides}) {
                                push @errors, "overrides is not allowed";
                            } elsif ($key ne 'shard') {
                                for my $override_string (split(qr/;/, $overrides_string)) {
                                    my ($override_key, $override_value) = split(qr/=/, $override_string, 2);
                                    if (exists $parsed_vars{$key}->{$value}->{$override_key}) {
                                        push @errors, "duplicate override '$override_key' for '$key' '$value' in vars: '$v'";
                                    } else {
                                        $parsed_vars{$key}->{$value}->{$override_key} = $override_value;
                                    }
                                }
                            } else {
                                push @errors, "overrides is not allowed for 'shard' variable";
                            }
                        }
                    }
                }
            } else {
                push @errors, "incorrect format of vars: '$v'";
            }
        }

        $conf->{vars} = \%parsed_vars;
        if (my @vars_to_validate_times = grep { $_ ne 'shard' } keys %{ $conf->{vars} }) {
            for my $key (@vars_to_validate_times) {
                for my $value (keys %{ $conf->{vars}->{$key} }) {
                    push @errors, _validate_juggler_ttl($conf,
                                                                 $conf->{vars}->{$key}->{$value},
                                                                 " for '$key' '$value'"
                                                                );
                }
            }
        } else {
            push @errors, _validate_juggler_ttl($conf);
        }

        unless (@errors || @{ $conf->{raw_events} }) {
            push @{ $conf->{raw_events} }, _default_juggler_raw_events_from_path($conf, $script_name);
        }

        push @errors, "name is required" unless $conf->{name};
        push @errors, "raw_events is required" unless @{ $conf->{raw_events} };

        if (@{ $conf->{raw_events} }) {
            my $has_shard_in_raw_events = 0;
            my $has_run_id_in_raw_events = 0;
            for my $one_raw_events (@{ $conf->{raw_events} }) {
                if ($one_raw_events !~ /^[a-zA-Z0-9_\-\.\$]+$/) {
                    push @errors, 'raw_events "' . $one_raw_events. '" contains invalid characters (allowed [0-9a-zA-Z\-\._\$])';
                }
                for my $var_name ($one_raw_events =~ m/\$([\w\d]+)\b/g) {
                    $used_vars{$var_name} = 1;
                }
                if ($one_raw_events =~ m/\$shard\b/) {
                    $has_shard_in_raw_events ||= 1;
                    unless ($conf->{vars} && $conf->{vars}->{shard} || $conf->{sharded}) {
                        push @errors, 'raw_events "' . $one_raw_events. '" with $shard requires sharded or shard in vars';
                    }
                }
                if ($one_raw_events =~ m/\$run_id\b/) {
                    $has_run_id_in_raw_events ||= 1;
                }
            }
            if ($conf->{sharded} && !$has_shard_in_raw_events) {
                push @errors, 'raw_events must contain $shard if sharded';
            }
            if ($has_run_id_in_raw_events && !$conf->{distributed} && !$conf->{vars}->{run_id}) {
                push @errors, 'distributed should be set if raw_events contains $run_id';
            }
            if ($conf->{distributed}) {
                die 'script_name is required for distributed scripts' unless defined $conf->{vars}->{script_name};
                die 'shards-num is required for distributed scripts'
                    unless $Yandex::ScriptDistributor::SHARDS_NUM_REF && $$Yandex::ScriptDistributor::SHARDS_NUM_REF > 0;
                die 'script-distributor-conf-path is required for distributed scripts' unless $Yandex::ScriptDistributor::CONF_FILE_PATH;
            }
        }

        if ($O{shards_num} && $used_vars{shard} && defined $conf->{vars}->{shard}) {
            my $shards = $conf->{vars}->{shard};
            for my $shard (keys %$shards) {
                push @errors, "shard $shard is out of bounds [1; $O{shards_num}]" unless 1 <= $shard && $shard <= $O{shards_num};
            }
        }

        for my $v (grep {$_ !~ /^(?:shard|run_id)$/} keys %used_vars) {
            push @errors, "var '$v' used in raw_events but not specified" unless $conf->{vars} && exists $conf->{vars}->{$v};
        }

        if (%used_vars) {
            for my $v (keys %{ $conf->{vars} }) {
                push @errors, "var '$v' is specified, but not used in raw_events" unless exists $used_vars{$v};
            }
        }
    }

    return @errors;
}

=head3 _default_juggler_name_from_path($conf, $script_name)

    Получить имя для juggler-проверки по шаблону $DEFAULT_JUGGLER_NAME_TEMPLATE и имени скрипта $script_name
    $conf - пока не используется

    Результат - строка с именем или undef

=cut

sub _default_juggler_name_from_path {
    my ($conf, $script_name) = @_;
    if ($script_name && $DEFAULT_JUGGLER_NAME_TEMPLATE) {
        return sprintf($DEFAULT_JUGGLER_NAME_TEMPLATE, $script_name);
    } else {
        return undef;
    }
}

=head3 _default_juggler_raw_events_from_path($conf, $script_name)

    Получить выражение для сырых juggler-событий по:
        шаблону $DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE
        имени скрипта $script_name
        признаку шардированности $conf->{sharded} или $conf->{vars}->{shard}

    Возвращает строку или пустой список (если недостаточно данных)

=cut

sub _default_juggler_raw_events_from_path {
    my ($conf, $script_name) = @_;

    return () unless $script_name && $DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE;

    my $shard_suf;
    if ($conf->{sharded} || $conf->{vars}->{shard}) {
        $shard_suf = '.shard_$shard';
    } else {
        $shard_suf = '';
    }
    return sprintf($DEFAULT_JUGGLER_RAW_EVENTS_TEMPLATE, $script_name, $shard_suf);
}

=head3 _human2secon($human_str)

    Переводит строку вида XdYhZmNs в секунды

=cut

sub _human2second {
    my $human = shift;
    my $seconds;
    if ($human =~ m/^(?:([0-9]+)d)?(?:([0-9]+)h)?(?:([0-9]+)m)?(?:([0-9]+)s?)?$/) {
        my ($d, $h, $m, $s) = ($1, $2, $3, $4);
        if (defined $d || defined $h || defined $m || defined $s) {
            $_ //= 0 for ($d, $h, $m, $s);
            $seconds = $d * 86400 + $h * 3600 + $m * 60 + $s;
        }
    }
    return $seconds;
}

=head3 _validate_juggler_ttl

    Проверяет значение ttl в juggler-метаданных

=cut

sub _validate_juggler_ttl{
    my ($conf, $override, $msg_suffix) = @_;
    $msg_suffix //= '';

    my @errors;
    my $ttl = $override->{ttl} // $conf->{ttl};

    if (defined $ttl) {
        my $seconds = _human2second($ttl);
        if (!defined $seconds) {
            push @errors, 'ttl has invalid format: must be XXdXXhXXmXXs or integer; for example 3600s or 60m or 1h';
        } elsif ($seconds < 60 || $seconds > MAX_JUGGLER_TTL) {
            push @errors, sprintf('ttl has invalid value "%s" (%d seconds) (must be 60 <= x <= %d)%s', $ttl, $seconds, MAX_JUGGLER_TTL, $msg_suffix);
        } else {
            # ok
        }
    } else {
        push @errors, 'ttl is required' . $msg_suffix;
    }

    return @errors;
}


1;
