#!/usr/bin/perl -w

# $Id$

=head1 NAME

    crontab2pod.pl -- одноразовый скрипт для перенесения информации из кронтабов в специальные секции pod

=head1 DESCRIPTION

    Читает указанные кронтабы и вносит в соответствующие скрипты pod-секции METADATA в формате, подходящем для pod2crontab.pl (см.)

    Опции: 
        --help, -h 
            показать справку

        --scripts-path-prefix 
            полный путь к каталогу, в который устанавливается прокет        
            (будет отброшен из полных имен скриптов)

        --package-prefix PACKAGE_PREFIX,
            общий префикс имен пакетов, будет отброшен

        --leftovers-path
            путь к каталогу, в который будут записаны файлы с "остаточными" записями из кронтабов 
            (потом этот каталог станет predefined-crontabs-path для pod2crontab.pl)

    После опций должен быть указан список крон-файлов.

    crontab2pod.pl --scripts-path-prefix '/var/www/ppc.yandex.ru' --package-prefix yandex-direct- --default-user ppc --leftovers-path predefined_crontabs packages/yandex-direct/debian/*.cron.d

=head1 TODO


=cut

use strict;

use File::Slurp;
use Encode;

use Config::General;
use Getopt::Long;

use Data::Dumper;

use utf8;
use open ':std' => ':utf8';

#..............................................................................
my @pod_header = (
    "",
    "=head1 METADATA", 
    "",
);

my @pod_footer = (
    "=cut", 
    "",
);


my ($RUN_PATH, $PACKAGE_PREFIX, $LEFTOVERS_PATH, $DEFAULT_USER);

GetOptions (
    "scripts-path-prefix=s" => \$RUN_PATH,
    "package-prefix=s" => \$PACKAGE_PREFIX,
    "leftovers-path=s" => \$LEFTOVERS_PATH, 
    "default-user=s"  => \$DEFAULT_USER,
    "help" => \&usage, 
);


my %POD;
my %LEFTOVERS;
for my $file (@ARGV){
    my ($crontab) = $file =~ m!.*/([^/]*.cron.d)!;
    my ($short_package_name) = $crontab =~ m!$PACKAGE_PREFIX(.*).cron.d! ;
    $short_package_name ||= '';

    my @lines = read_file($file, binmode => ':utf8');

    for my $l (@lines){
        chomp($l);

        next if $l =~ /^#/ || $l =~ /^\s*$/;
        if( $l =~ /^\S+\s*=/){
            # присваивание переменных -- в "остатки"
            push @{$LEFTOVERS{$crontab}}, $l;
            next;
        }

        my @time_fields;
        (@time_fields[0..4], my $user, my $command) = split /\s+/, $l, 7; 

        my ($script_name, $params) = ($command =~ m!^$RUN_PATH/(\S+)(?:\s+(.*))?$!);

        if ( !$script_name || $script_name !~ /\.pl$/ ) {
            # запуск неизвестных/не перловых скриптов -- в "остатки"
            push @{$LEFTOVERS{$crontab}}, $l;
            next;
        }

        my $rec = {
            time => join(" ", @time_fields[0..4]),
            package => $short_package_name,
        };

        $rec->{params} = $params if $params;
        $rec->{user} = $user if $user ne $DEFAULT_USER;

        push @{$POD{$script_name}}, $rec; 
    }
}


# дописываем кронтаб-записи в скрипты
for my $script_name (keys %POD){
    print "writing $script_name\n";
    -f $script_name or die "file doesn't exist: $script_name";
    my $script_text = read_file($script_name, binmode => ':utf8');

    my $conf = {};
    push @{$conf->{crontab}}, @{$POD{$script_name}} ; 

    my $conf_text = serialize_metadata($conf);

    my $pod = join "\n", @pod_header, $conf_text, @pod_footer;

    $script_text =~ s/^(.*?\n\s*\n)/$1$pod\n/s;

    write_file($script_name, {binmode => ':utf8', atomic => 1}, $script_text) or die "can't write file $script_name";
}

# записываем "остатки" -- все, что не получилось распределить по скриптам
for my $crontab (keys %LEFTOVERS){
    print "writing leftovers: $crontab\n";

    my $text = join "\n\n", @{$LEFTOVERS{$crontab}}, "";

    write_file("$LEFTOVERS_PATH/$crontab", {binmode => ':utf8', atomic => 1}, $text) or die "can't write file $crontab";
}



exit; 


sub serialize_metadata
{
    my ($conf) = @_;

    my $conf_obj = new Config::General(-StoreDelimiter => ": ", -ConfigHash => {});

    return  $conf_obj->save_string($conf);
}

