#!/usr/bin/perl

use Direct::Modern;

use Test::More;

use Yandex::Metadata;
use ScriptsMetadata;

my %good_ttl = (
    (map { $_ => 60 } qw/60 60s/),
    (map { $_ => 129600 } qw/129600 129600s 36h 36h0m 36h0m0 36h0m0s 36h0s 36h0 35h60m 35h59m60s 1d12h 0d36h/),
    (map { $_ => 5400 } qw/5400 5400s 90m 90m0 90m0s 0h90m 0h90m0 0h90m0s 89m60 89m60s 88m120s 1h30m 1h1800s 1h1800 1h29m60s/),
    '1m' => 60,
    '1h' => 3600,
    '60s' => 60,
    '1h1m1s' => 3661,
    '1h1m1' => 3661,
    '1h1s' => 3601,
    '1m1s' => 61,
    '10h' => 36000,
    '10h20m' => 37200,
    '1d' => 86400,
    '2d' => 172800,
    '1d24h' => 172800,
);

my @bad_ttl = ('', '  ', qw/
    11a22b33c44d
    h m s hm hs ms hms msh mhs mm ss hh
    1hh 1mm 1ss
    1ms 1hm 1hs
    10m10s10h 10s10m10h 10m10h10s 10h10s10m
/);

# [ 'metadata', $test_name ]
my @tests_ok = (
    [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            raw_events:     scripts.ppcPdfReports.working.another_important_process
            raw_events:     scripts.ppcPdfReports.working.$different_var
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            vars:           queue=heavy1<ttl=600>
            vars:           queue=heavy4<ttl=600>,heavy5
            vars:           different_var=value
            ttl:            123
        </juggler>',
        'full example',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3<ttl=600>
            vars:           queue=heavy4<ttl=600>
        </juggler>',
        'ttl specified in overrides',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=std<ttl=300>
            vars:           queue<ttl=600>=heavy1,heavy2,heavy3
        </juggler>',
        'ttl specified in overrides for multiple values',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=std
            vars:           queue<ttl=600>=heavy1,heavy2,heavy3
            ttl:            120
        </juggler>',
        'ttl specified both in params and overrides for multiple values',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            vars:           queue=std,heavy
            vars:           shard=1
            ttl:            123
        </juggler>',
        'shard specified in vars',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            130000
        </juggler>',
        'valid big ttl',
    ],
    ( map {
        [
            "<juggler>
                name:           scripts.ppcPdfReports.working
                host:           developer_checks.direct.yandex.ru
                raw_host:       CGROUP\%direct_ng_scripts_prod
                raw_events:     scripts.ppcPdfReports.working.\$queue.shard_\$shard
                sharded:        1
                vars:           queue=heavy3
                vars:           queue=heavy4
                ttl:            $_
            </juggler>",
            "valid ttl in human-readable format: $_",
        ],     
    } keys %good_ttl),
    ( map {
        [
            "<juggler>
                name:           scripts.ppcPdfReports.working
                host:           developer_checks.direct.yandex.ru
                raw_host:       CGROUP\%direct_ng_scripts_prod
                raw_events:     scripts.ppcPdfReports.working.\$queue.shard_\$shard
                sharded:        1
                vars:           queue=heavy3<ttl=$_>
                vars:           queue=heavy4<ttl=$_>
            </juggler>",
            "valid ttl in overrides in human-readable format: $_",
        ],     
    } keys %good_ttl),
);
my @tests_fail = (
    [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue: test;heavy
            ttl:            120
        </juggler>',
        'incorrect vars format (bad delimeter)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test, queue=heavy
            ttl:            120
        </juggler>',
        'incorrect vars format',
    ], [
    #   'incorrect vars format (bad overrides delimeter)',
    #   был тест, где внутри переопределенией был другой разделитель (не ;)
    # ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3<ttl=600;something_else=100>
            ttl:            120
        </juggler>',
        'incorrect vars format (unknown override)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3<>
            ttl:            120
        </juggler>',
        'incorrect vars format (empty override)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=std<ttl=300>
            vars:           queue<ttl=600>=heavy1<ttl=120>,heavy2,heavy3
        </juggler>',
        'incorrect vars format (duplicate overrides)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer!checks.direct.yandex.ru
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3
            ttl:            120
        </juggler>',
        'incorrect host',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       ppcback01e!yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3
            ttl:            120
        </juggler>',
        'incorrect raw_host',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,test
            ttl:            120
        </juggler>',
        'duplicate vars value',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=heavy3<ttl=600;ttl=120>
            ttl:            120
        </juggler>',
        'duplicate override value',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue
            vars:           queue=std,heavy
            only_shards:    1
            ttl:            120
        </juggler>',
        'unknown fields in juggler metadata section'
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            sharded:        1
            ttl:            120
        </juggler>',
        'sharded, but raw_events has no $shard',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            raw_events:     scripts.ppcPdfReport.not_working
            sharded:        1
            ttl:            120
        </juggler>',
        'sharded, but no one of raw_events has $shard',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            vars:           queue=std,heavy
            vars:           shard=1,500,600
            ttl:            120
        </juggler>',
        'specified shard is out of bounds',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working.$queue
            ttl:            120
        </juggler>',
        'raw_events has vars without specification',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working_good
            raw_events:     scripts.ppcPdfReport.working.$queue
            ttl:            120
        </juggler>',
        'one of raw_events has vars without specification',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working.shard_$shard
            ttl:            120
        </juggler>',
        'raw_events has vars without specification (shard)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working.working_good
            raw_events:     scripts.ppcPdfReport.working.shard_$shard
            ttl:            120
        </juggler>',
        'one raw_events has vars without specification (shard)',
    ], [
        '<juggler>
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            120
        </juggler>',
        'name is not specified',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            ttl:            120
        </juggler>',
        'raw_events (raw event name pattern) is not specified',
    ],  [
        '<juggler>
            name:           scripts ppcPdfReports working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            123
        </juggler>',
        'invalid characters in name',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            30
        </juggler>',
        'ttl is invalid (ttl should be >= 60)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            864001
        </juggler>',
        'ttl is invalid (ttl should be <= 10 days)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReport.working
            ttl:            sss
        </juggler>',
        'ttl is invalid (not a number)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.shard_$shard
            sharded:        1
            vars:           shard=1<ttl=600>
            ttl:            120
        </juggler>',
        'overrides is not allowed for "shard" vars',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.shard_$shard
            vars:           queue=std,heavy
            vars:           shard=1
            ttl:            123
        </juggler>',
        'queue specified in vars, but not used in raw_events',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard;
            vars:           queue=std,heavy
            vars:           shard=1
            ttl:            123
        </juggler>',
        'invalid characters in raw_events',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard,scripts.ppcPdfReports.working_good
            vars:           queue=std,heavy
            vars:           shard=1
            ttl:            123
        </juggler>',
        'invalid characters in raw_events',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            refresh_time:   60
            ttl:            123
        </juggler>',
        'refresh_time key is not allowed',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            vars:           queue=heavy1<ttl=600>
            vars:           queue=heavy2<refresh_time=120>
            ttl:            123
        </juggler>',
        'refresh_time is not allowed in overrides',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            vars:           queue=heavy1<ttl=600>
            vars:           queue=heavy2<refresh_time=120>
            vars:           queue=heavy4<ttl=600;refresh_time=120>,heavy5
            refresh_time:   60
            ttl:            123
        </juggler>',
        'refresh_time is not allowed in overrides',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            vars:           queue=heavy1<ttl=600;>
            ttl:            123
        </juggler>',
        'invalid overrides (; at end)',
    ], [
        '<juggler>
            name:           scripts.ppcPdfReports.working
            host:           developer_checks.direct.yandex.ru
            host:           checks_day.direct.yandex.ru
            raw_host:       CGROUP%direct_ng_scripts_prod
            raw_host:       ppcback01e.yandex.ru
            raw_events:     scripts.ppcPdfReports.working.$queue.shard_$shard
            sharded:        1
            vars:           queue=test
            vars:           queue=std,easy
            vars:           queue=heavy1<;ttl=600>
            ttl:            123
        </juggler>',
        'invalid overrides (; at start)',
    ],
    ( map {
        [
            "<juggler>
                name:           scripts.ppcPdfReports.working
                host:           developer_checks.direct.yandex.ru
                raw_host:       CGROUP\%direct_ng_scripts_prod
                raw_events:     scripts.ppcPdfReports.working.\$queue.shard_\$shard
                sharded:        1
                vars:           queue=heavy3
                vars:           queue=heavy4
                ttl:            $_
            </juggler>",
            "invalid ttl in human-readable format: $_",
        ],     
    } keys @bad_ttl),
    ( map {
        [
            "<juggler>
                name:           scripts.ppcPdfReports.working
                host:           developer_checks.direct.yandex.ru
                raw_host:       CGROUP\%direct_ng_scripts_prod
                raw_events:     scripts.ppcPdfReports.working.\$queue.shard_\$shard
                sharded:        1
                vars:           queue=heavy3<ttl=$_>
                vars:           queue=heavy4<ttl=$_>
            </juggler>",
            "invalid ttl in overrides in human-readable format: $_",
        ],     
    } keys @bad_ttl),
);

Test::More::plan(tests => @tests_ok + @tests_fail + scalar(keys(%good_ttl)) + @bad_ttl);

for my $is_error_expected (0, 1) {
    for my $test ($is_error_expected ? @tests_fail : @tests_ok) {
        my $metadata = deserialize_metadata($test->[0]);
        my @errors = ScriptsMetadata::parse_and_validate_juggler($metadata->{juggler},
                                                                 monrun => $metadata->{monrun},
                                                                 shards_num => 2,
                                                                 );
        ok((@errors ? 1 : 0) == $is_error_expected, ($is_error_expected ? 'should fail: ' : 'should pass: ') . $test->[1]);
    }
}

for my $human_ttl (keys %good_ttl) {
    is(ScriptsMetadata::_human2second($human_ttl), $good_ttl{$human_ttl}, "_human2second($human_ttl) = $good_ttl{$human_ttl}");
}

for my $human_ttl (@bad_ttl) {
    is(ScriptsMetadata::_human2second($human_ttl), undef, "_human2second($human_ttl) = undef");
}
