#!/usr/bin/perl -w

=head1 TEST

    проверяем, правильно ли извлекаются метаданные из скриптов

    $Id$

=cut

use strict;
use warnings;

use ScriptsMetadata;
use Test::Deep;
use Test::Exception;
use Test::More;
use Yandex::Metadata;

use FindBin qw/$Bin/;
require "$Bin/../bin/pod2crontab.pl";

use utf8;

# корректные скипты. Должны корректно обрабатываться
my @correct_tests = (
    {
        name => 'no_cron_no_pod',
        cron_count => 0,
        script => q%#!/usr/bin/perl
print "hello world\n";
        %,
    },
    {
        name => 'pod_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/perl

=head1 SMTH

text

=cut
print "hello world\n";
        %,
    },
    {
        name => 'metadata_without_cron',
        cron_count => 0,
        script => q%#!/usr/bin/perl
=head1 METADATA

    <smth>
        a: 1
    </smth>

=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_simple',
        cron_count => 1,
        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'cron_with_c_style_comment',
        cron_count => 1,
        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines',
        cron_count => 1,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
        <switchman>
            group: conf-test
            lockname: abc
            <leases>
                FQDN_mem: 4096:MEMMB
                FQDN_cpu: 1:CPU
            </leases>
        </switchman>
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines_2',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
    <crontab>
        time: 3 * * * *
        package: scripts
        params: --three
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'cron_without_empty_lines_3',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=pod
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
    <crontab>
        time: 3 * * * *
        package: scripts
        params: --three
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'multi_cron_with_empty_lines',
        cron_count => 2,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'comments',
        cron_count => 0,
        script => q%#!/usr/bin/perl
=head1 METADATA
    # <crontab>
    #     time: * * * * *
    #     package: scripts
    # </crontab>
=cut
print "hello world\n";
        %,
    },
);

my @incorrect_tests = (
    {
        name => 'metadata_integrated_with_other_pod',
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

    <crontab>
        time: * * * * *
        package: scripts
    </crontab>

=head2

    text    

=cut
print "hello world\n";
        %,
    },
    {
        name => 'incorrect_config_syntax',
        script => q%#!/usr/bin/perl
=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts

=cut
print "hello world\n";
        %,
    },
    {
        name => 'misspelled_time_field',
        script => q%#!/usr/bin/perl
=head1 METADATA

    <crontab>
        tmie: * * * * *
        package: scripts
    </crontab>

=cut
print "hello world\n";
        %,
    },
    {
        name => 'misspelled_package_field',
        script => q%#!/usr/bin/perl
=head1 METADATA

    <crontab>
        time: * * * * *
        pakage: scripts
    </crontab>

=cut
print "hello world\n";
        %,
    },
    {
        name => 'switchman_params_contains_spacebars',
        cron_count => 1,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
        <switchman>
            group: conf-test
            lockname: contains spacebar
            <leases>
                FQDN_mem: 4096:MEMMB
                FQDN_cpu: 1:CPU
            </leases>
        </switchman>
    </crontab>
=cut
print "hello world\n";
        %,
    },
    {
        name => 'switchman_params_contains_tabs',
        cron_count => 1,
        script => q%#!/usr/bin/perl
=head1 METADATA
    <crontab>
        time: * * * * *
        package: scripts
        <switchman>
            group: conf-test
            lockname: contains	tab
            <leases>
                FQDN_mem: 4096:MEMMB
                FQDN_cpu: 1:CPU
            </leases>
        </switchman>
    </crontab>
=cut
print "hello world\n";
        %,
    },

);

for my $t (@correct_tests){
    my $crontab = deserialize_metadata(extract_metadata($t->{name} => $t->{script}))->{crontab} || [];
    $crontab = [$crontab] unless ref $crontab eq 'ARRAY';
    is(scalar @$crontab, $t->{cron_count}, $t->{name});
    cmp_deeply([ScriptsMetadata::validate_crontab($_)], [], "$t->{name} validates") for @$crontab;
}

for my $t (@incorrect_tests){
    dies_ok {
        my $crontab = deserialize_metadata(extract_metadata($t->{name} => $t->{script}))->{crontab}; 
        $crontab = [$crontab] unless ref $crontab eq 'ARRAY';
        for my $c (@$crontab) {
            die if ScriptsMetadata::validate_crontab($c);
        }
    } $t->{name};
}

done_testing();
