#!/usr/bin/perl -w

=head1 TEST

    проверяем обработку ключа sharded

    $Id$

=cut

use strict;
use warnings;

use lib::abs "../lib";

use ScriptsMetadata;
use Test::Exception;
use Test::Deep;
use Test::More;
use Yandex::Metadata;

use FindBin qw/$Bin/;
require "$Bin/../bin/pod2crontab.pl";

use utf8;

# корректные скипты. Должны корректно обрабатываться
my @correct_tests = (
    # нет шаргинга, нешардированные скрипты
    {
        name => 'no sharding, not sharded script',
        shard_num => 0,

        shard_params => [undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'no sharding, not sharded script x2',
        shard_num => 0,

        shard_params => [undef, undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */2 * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    # 2 шарда, нешардированный скрипт
    {
        name => 'sharding=2, not sharded script',
        shard_num => 2,

        shard_params => [undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, not sharded script x2',
        shard_num => 2,

        shard_params => [undef, undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    # нет шардинга, шардированный скрипт 
    {
        name => 'no sharding, sharded script',
        shard_num => 0,

        shard_params => [undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'no sharding, sharded script x2',
        shard_num => 0,

        shard_params => [undef, undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: * * * * *
        package: scripts
        sharded: 1
    </crontab>
    <crontab>
        time: */2 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    # 3 шарда, шардированный скрипт
    {
        name => 'sharding=3, sharded script',
        shard_num => 3,

        shard_params => [1, 2, 3],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=3, sharded script x2',
        shard_num => 3,

        shard_params => [1, 2, 3, 1, 2, 3],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 1
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    # 2 шарда + частично шардированный скрипт
    {
        name => 'sharding=2, partially sharded script-0',
        shard_num => 2,

        shard_params => [1, 2, undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 1
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, partially sharded script-1',
        shard_num => 2,

        shard_params => [undef, 1, 2],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, partially sharded script-2',
        shard_num => 2,

        shard_params => [undef, 1, 2, undef, 1, 2],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
    </crontab>
    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, partially sharded script-2',
        shard_num => 2,

        shard_params => [undef, 2, undef, 1, 2],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        only_shards: 1,2,3,4,5
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
        only_shards: 2
    </crontab>
    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>
    <crontab>
        time: */5 * * * *
        package: scripts
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },    
    {
        name => 'sharding=2, partially sharded script-2',
        shard_num => undef,

        shard_params => [undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        only_shards: 1,2,3,4,5
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, partially sharded script-2',
        shard_num => 10,

        shard_params => [undef],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        only_shards: 1,2,3,4,5
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
    {
        name => 'sharding=2, partially sharded script-2',
        shard_num => 10,

        shard_params => [1, 3, 5],

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        only_shards: 1,3,5
        sharded: 1
    </crontab>

=cut

print "hello world\n";
        %,
    },                    
);

my @incorrect_tests = (
    # 
    {
        name => 'incorrect shard_num',
        shard_num => 'aaaa',

        script => q%#!/usr/bin/perl

=head1 METADATA

    <crontab>
        time: */3 * * * *
        package: scripts
        sharded: 0
    </crontab>

=cut

print "hello world\n";
        %,
    },
);

# проверяем корректные случаи
for my $t (@correct_tests){
    my $cron_lines_count = scalar @{$t->{shard_params}};

    my $crontab = deserialize_metadata(extract_metadata($t->{name} => $t->{script}))->{crontab} || [];
    $crontab = [$crontab] unless ref $crontab eq 'ARRAY';
    cmp_deeply([ScriptsMetadata::validate_crontab($_)], [], "$t->{name} validate") for @$crontab;

    my $lines = generate_crontab_lines(
        $crontab, 
        default_user => 'ppc', 
        run_path => '', 
        script => 'script.pl',
        shard_num => $t->{shard_num},
    );

    is(scalar @$lines, $cron_lines_count, "$t->{name} lines count");

    my @shard_params;
    for (@$lines) {
        if (!/--shard-id/){
            push @shard_params, undef
        } else {
            /--shard-id[ =](\S*)/;
            push @shard_params, $1;
        }
    }

    cmp_deeply(\@shard_params, $t->{shard_params}, "$t->{name} --shard-id params");
}

# проверяем некорректные случаи
for my $t (@incorrect_tests){
    dies_ok {
        my $crontab = deserialize_metadata(extract_metadata($t->{name} => $t->{script}))->{crontab}; 
        $crontab = [$crontab] unless ref $crontab eq 'ARRAY';
        for my $c (@$crontab) {
            die if ScriptsMetadata::validate_crontab($c);
        }
        my $lines = generate_crontab_lines(
            $crontab, 
            default_user => 'ppc', 
            run_path => '', 
            script => 'script.pl',
            shard_num => $t->{shard_num},
        );
    } $t->{name};
}

done_testing();
