#!/usr/bin/perl

=head1 NAME

    beta-disk-usage.pl -- статистика использования места в /var/www по пользователям/бетам

=head1 SYNOPSIS

    ./beta-disk-usage.pl

=head1 DESCRIPTION

    Группирует вывод du /var/www -d 1 по пользователям.

=cut


use strict;
use warnings;

use utf8;

use open qw/:std :encoding(UTF-8)/;

use Getopt::Long;

our $ROOT_DIR = '/var/www';
# цифры взяты с потолка и пока никак не используются
our $TOTAL_DU_THRESHOLD = 10_000_000;
our $BETA_DU_THRESHOLD = 5_000_000;

run() unless caller();

sub run {
    parse_options();
    open my $fh, '-|', 'du', $ROOT_DIR, '-d' => 1;

    my %du_stats_by_user;
    while (my $line = <$fh>) {
        chomp $line;
        my ($du, $path) = split /\s+/, $line;   # надеемся, что в именах директорий нет пробелов
        next if $path eq $ROOT_DIR;
        my $login;
        if (index($path, '-') == -1) {
            # '-' может быть как частью логина, так и именем беты (например, /var/www/beta.auto-trunk.8994)
            # если такой неоднозначности нет, можно угадать логин владельца из имени директории
            $path =~ m!^\Q$ROOT_DIR\E/*beta\.([\-a-z0-9]+)\.!;
            $login = $1;
        } else {
            # если угадать нельзя, определяем владельца "по-честному" с помощью stat()
            my $uid = (stat $path)[4];
            $login = getpwuid($uid);
        }
        # тут надо проверять логин на существование/уволенность
        # ...

        $login //= "''";    # чтобы было больше на yaml похоже. С пустым логином будут не-беты типа /var/www/ppc.yandex.ru
        if ($login !~ /^(ppc|www-data|root)$/) {
            $du_stats_by_user{ $login }->{total} += $du;
            $du_stats_by_user{ $login }->{ $path } = $du;
        } else { 
            # для нечеловеческих логинов надо делать что-то другое
        }
    }
    print "---\n";  # пишем YAML, но сортируем ключи
    for my $login (sort {$du_stats_by_user{$b}->{total} <=> $du_stats_by_user{$a}->{total}} keys %du_stats_by_user) {
        my $du_stats = $du_stats_by_user{ $login };
        print "$login:\n";
        my $total = delete $du_stats->{total};
        print "  total: " . hu($total) . "\n";
        print join("\n", map { "  $_: " . hu($du_stats->{$_}) } sort { $du_stats->{$b} <=> $du_stats->{$a} } keys %$du_stats );
        print "\n";
    }
    exit 0;
}

sub parse_options {
    GetOptions(
        'total-du-threshold=s' => \$TOTAL_DU_THRESHOLD,
        'beta-du-threshold=s' => \$BETA_DU_THRESHOLD,
        'root-dir=s' => \$ROOT_DIR,
    ) || die "can't parse options, stop";
}

# преобразуем килобайты в человеческие единицы
sub hu {
    my ($k) = @_;
    my $g = $k / 1024 / 1024;
    my $m = $k / 1024;
    if (int($g) > 0) {
        return sprintf "%.1fG", $g;
    } elsif (int($m) > 0) {
        return sprintf "%.1fM", $m;
    } else {
        return sprintf "%dK", $k;
    }
}

1;
