#!/usr/bin/perl -w

=head1 NAME

commit-reminder - напомнить, какие коммиты отложены
 
=head1 SYNOPSIS

=head1 DESCRIPTION

посмотреть свои собственные напоминалки:
commit-reminder.pl

посмотреть напоминалки для других разработчиков:
commit-reminder.pl --login icenine --login mirage

посмотреть все напоминалки:
commit-reminder.pl --all

После автоматической (по крону) отправки писем напоминалки удаляются.

=head1 TODO

  + почту отправлять только по специальному ключу, 
  + основной режим -- просмотреть лог для себя (или другого пользователя, или всех)  

=cut

# $Id$

use strict;
use warnings;

use Data::Dumper;
use Encode;
use File::Slurp;
use Getopt::Long;
use JSON;
use YAML;

use Yandex::SendMail;

use utf8;
use open ':std' => ':utf8';

my $LOG = '/var/spool/direct-release-flag/failed-commits/log';
my $ST_URL = 'https://st.yandex-team.ru';

run() unless caller();

sub run
{
    my $opt = parse_options();

    exit 0 unless -f $LOG;

    if ( $opt->{email} ){
        my $new_log = $LOG . ".process";
        system("mv $LOG $new_log") == 0 or die "can't mv $LOG $new_log, stop"; 
        $LOG = $new_log;
    }

    my @records = grep {!/^\s*$/} split /\n+/, File::Slurp::read_file($LOG, binmode => ':utf8');

    my %FAILED_COMMITS;
    for my $rec_str ( @records ){
        my $rec = eval {from_json($rec_str)};
        if ( $@ ){
            print STDERR "can't parse record: $rec_str\n";
            next;
        }
        if ( $rec->{login} =~ /^\s*$/ ){
            print STDERR "empty login in record: $rec_str\n";
            next;
        }
        push @{$FAILED_COMMITS{$rec->{login}}}, $rec;
    }

    my %REMINDERS;
    my @logins = $opt->{only_logins} ? @{$opt->{only_logins}} : (keys %FAILED_COMMITS);
    for my $login ( @logins){
        next unless @{ $FAILED_COMMITS{$login} || [] } > 0;
        for my $commit ( sort { $a->{logtime} cmp $b->{logtime}  } @{$FAILED_COMMITS{$login}} ){
            chomp $commit->{pwd};
            push @{$REMINDERS{$login}}, {
                date => $commit->{logtime},
                issues => $commit->{commit_params}->{issue_keys},
                argv => $commit->{argv},
                pwd => $commit->{pwd},
            };
        }
    }

    my $hostname = `hostname -f`;
    chomp $hostname;
    my %MESSAGES;
    for my $login ( keys %REMINDERS ){
        my $message = '';
        $message .= "### $login @ $hostname\n\n";
        for my $rem ( @{$REMINDERS{$login}} ){
            $message .= "date: $rem->{date}
pwd: $rem->{pwd}
issues: ".join(", ", @{@{$rem->{issues}} ? [map {"$_ [ $ST_URL/$_ ]"} @{$rem->{issues}}] : []})."
argv: ".join(" ", @{$rem->{argv}||[]})."\n\n";
        }
        $MESSAGES{$login} = $message;
    }

    if ($opt->{email}){
        for my $login ( keys %MESSAGES ){
            my $message = "За прошедший релизный мораторий были отложены твои коммиты, возможно, надо их повторить:\n\n".$MESSAGES{$login};
            my $to = "$login\@yandex-team.ru";
            Yandex::SendMail::sendmail($to, "direct-maintenance\@yandex-team.ru", "$hostname: окончание релизного моратория в Директе ($login)", \$message);
            print STDERR "$to ($hostname)\n$message\n\n";
        }
    } else {
        print $_ for values %MESSAGES;
    }

    exit; 
}


sub parse_options
{
    my %O = (
        only_logins => [[getpwuid($<)]->[0]],
    );

    GetOptions (
        "h|help" => sub {
            system("podselect -section NAME -section DESCRIPTION -section OPTIONS -section EXAMPLES $0 | pod2text-utf8 | LESSCHARSET=utf-8 less"); 
            exit 0;
        },
        "email" => \$O{email},
        "login=s@" => \$O{only_logins},
        "all|all-logins" => sub{ $O{only_logins} = undef },
    );

    return \%O;
}


