#!/usr/bin/perl

=head1 NAME

    disk-usage-logger.pl

=head1 SYNOPSIS

    Можно передать опции:
        --dir
            директории, по которым хотим видеть отдельную статистику (добавляются к /var/www и /)
            
     Обычное использование:
         disk-usage-logger.pl
     
     С указанием параметров:
         disk-usage-logger.pl --dir /etc --dir /bin

=head1 DESCRIPTION

	Cтатистика использования места в корне, /var/www и других директориях.
	Пишет сортированные логи du в /var/log/disk-usage	

=cut


use strict;
use warnings;

use utf8;
use Parallel::ForkManager;
use POSIX qw(strftime);

use open qw/:std :encoding(UTF-8)/;

use Getopt::Long;

my $PARALLEL = 5;

my $LOG_DIR = '/var/log/yandex/disk-usage/';

my @dirs = (
    { dir => '/',
      name => 'root'},
    { dir => '/var/www',
      name => 'var_www'},
    { dir => '/local',
      name => 'local'},
);


my $time_str = strftime "%Y%m%d", localtime;

run() unless caller();

sub get_logname {
    (my $name, my $is_temporary) = @_;
    if ($is_temporary) {
        return $LOG_DIR . '/du_' . $name . '.' . $time_str . '.raw';
    } else {
        return $LOG_DIR . '/du_' . $name . '.' . $time_str;
     }
}

sub run {
    parse_options();
    
    my $pm = Parallel::ForkManager->new($PARALLEL);

    my $failed = 0;
    $pm->run_on_finish(sub {
        my ($pid, $exit_code, $ident, $exit_signal, $core_dumped, $data) = @_;
        if ($exit_code) {
            $failed++;
        }
    });
    
    # создаем директорию для логов, если её нет
    unless (-d $LOG_DIR) {
        mkdir $LOG_DIR;
    }

    for my $dir (@dirs) {
        next if $failed;
        my $result;
        my $pid = $pm->start and next;
        my $log_filename = get_logname($dir->{name}, 1);
        
        `du -hcx $dir->{dir} > $log_filename`;
        
        # сортируем полученные данные в новый файл
        my $sorted_log_filename = get_logname($dir->{name}, 0);
        $result = system("sort -hr $log_filename > $sorted_log_filename");
        $pm->finish($result >> 8) if $result;
        
        # удаляем неотсортированную версию
        unlink $log_filename or $pm->finish($? >> 8);
        
        $pm->finish(0);
    }
    
    $pm->wait_all_children;
    warn "Someting went wrong" if $failed;
    
    # удаляем старые логи
    my $MIN_LOGS = (scalar @dirs) * 2;
    `ls -dt $LOG_DIR/* | awk 'NR > $MIN_LOGS '| xargs -r rm`;

    exit 0;
}

sub parse_options {
    my @custom_dirs;
    GetOptions(
         "dir=s" => \@custom_dirs,
    ) || die "can't parse options, stop";
    push @dirs, map { {dir => $_, name => $_ =~ s/\//_/rg } } @custom_dirs;
}

1;
