#!/usr/bin/perl

=head1 NAME

    procs-for-ports.pl -- найти процессы (apache/nginx), которые слушают на указанных портах

=head1 SYNOPSIS

    procs-for-ports.pl <порт> [<порт2> ...]

=head1 DESCRIPTION

    Выводит результат в виде последовательности строк вида

        (master|worker) <pid>   <cmdline>

    Процессы отбираются на основе вхождения нужной подстроки в cmdline, так что в принципе могут найтись "лишние".

=cut

use strict;
use warnings;

use utf8;

use open ":std" => ":utf8";

use Getopt::Long;

GetOptions(
    "h|help" => sub {
        system("podselect -section NAME -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8");
        exit;
    },
) or die "can't parse params, stop\n";

die "usage: procs-for-ports.pl <port> [<port2> ...]\nprocs-for-ports.pl -h for detailed description\n" unless @ARGV;

my @ports = @ARGV;
my @invalid_ports = grep { /[^0-9]/ } @ports;
die "invalid port number(s): " . join(", ", @invalid_ports) . "\n" if @invalid_ports;

my @ps_cmd = ("ps", "-e", "-o", "pid,ppid,cmd");
open PS, "-|:utf8", @ps_cmd or die "piping ps failed: $!";

my %ppid;
my %cmd;
my %has_child;
my @pids;
while (my $line = <PS>) {
    chomp $line;
    next unless $line =~ /[0-9]/;   # skip header
    next unless $line =~ /apache|nginx/;

    my ($pid, $ppid, $cmd) = $line =~ /([0-9]+)\s+([0-9]+)\s+(.+)/;

    push @pids, $pid;
    $ppid{ $pid } = $ppid;
    $cmd{ $pid } = $cmd;
    $has_child{ $ppid } = 1;
}

my $port_re = join "|", @ports;
my %is_master;  # i. e. nginx/apache master
my %is_worker;  # nginx/apache worker
for my $pid (@pids) {
    next unless $cmd{ $pid } =~ m{/var/www/beta \. [^/]+ \. ($port_re)/}x;
    if ($has_child{ $pid }) {
        $is_master{ $pid } = 1;
    } else {
        $is_worker{ $pid } = 1;
    }
}

if (! %is_master) {
    exit 1;
} else {
    for my $pid (@pids) {
        $is_worker{ $pid } = 1 if ($ppid{$pid} == 1 && $cmd{ $pid } =~ m{/var/www/beta \. [^/]+ \. ($port_re)/}x) || $is_master{ $ppid{$pid} };
    }
    for my $pid (keys %is_worker, keys %is_master) {
        printf "%s\t%d\t%s\n", ($is_master{ $pid } ? "master" : "worker"), $pid, $cmd{ $pid };
    }
    exit 0;
}
