package Startrek::Client::Easy;

=pod

=encoding utf-8

=head1 NAME

Startrek::Client::Easy - Perl-клиент для st

=head1 SYNOPSIS

    my $startrek = Startrek::Client::Easy->new( startrek => 'test' );
    my $startrek = Startrek::Client::Easy->new( api_url => 'https://st-api.test.yandex-team.ru/v2' );

    my $issue = $startrek->get(key => 'DIRECT-31948');               # вернёт ссылку на хэш
    my $issue = $startrek->get(key => 'DIRECT-31948', array => 1);   # вернёт ссылку на массив ссылок на хэш
    my $issues = $startrek->get(key => ['DIRECT-31948', 'DIRECT-31949']);
    my $issues = $startrek->get(query => 'Assignee: yukaba', array => 1);

    my $new_issue_key = $startrek->do(
        create  => 1,
        queue   => 'DIRECT',
        type    => 'task',
        summary => 'Новая задача',
    );
    $startrek->do(
        key        => 'DIRECT-31948',
        followers  => ['lena-san', 'zhur'],
        resolve    => 1,
        close      => 1,
        resolution => 'fixed',
    );

    # Можно делать запросы напрямую к API Стартрека

    my $allowed_transitions = $startrek->request('GET', '/issues/DIRECT-31949/transitions');
    my $issues_from_test = $startrek->request('POST', '/issues/_search', { query => 'Queue: DIRECT' });

    # Примеры однострочников для отладки

    чтобы работать с тестовым Стартреком, объект создавать с параметрами:
    $st = Startrek::Client::Easy->new( startrek => 'test' );

    прочитать тикет:
    perl -MStartrek::Client::Easy -MYAML -le '$st=Startrek::Client::Easy->new();  print YAML::Dump $st->get(key => "DIRECT-31948")'
    
    переоткрыть тикет:
    perl -MStartrek::Client::Easy -MYAML -le '$st=Startrek::Client::Easy->new();  print YAML::Dump $st->do(key => "DIRECT-31948", actions=>["reopen"])'
    
    зарезолвить:
    perl -MStartrek::Client::Easy -MYAML -le '$st=Startrek::Client::Easy->new();  print YAML::Dump $st->do(key => "DIRECT-31948", actions=>["resolve"])'

    проставить Ready For Test:
    perl -Ilib -MStartrek::Client::Easy -MYAML -le '$st=Startrek::Client::Easy->new();  print YAML::Dump $st->do(key => "DIRECT-7047", actions=>["ready_for_test"])'

    закрыть:
    perl -MStartrek::Client::Easy -MYAML -le '$st=Startrek::Client::Easy->new();  print YAML::Dump $st->do(key => "DIRECT-31948", close => 1)'

    прочитать большой список тикетов:
    perl -MStartrek::Client::Easy -e '$st=Startrek::Client::Easy->new(); for $p (1 .. 100){$releases = $st->get(query => q!Queue: DIRECT Type: Release  Components: "Releases: Direct"!, page => $p, perPage => 100); last unless scalar @$releases; print join "", map {"$_->{key} $_->{createdBy}\n"} @$releases; } '

    создать тикет:
    perl -MStartrek::Client::Easy -Mutf8 -le '$st=Startrek::Client::Easy->new(); print $st->do(create=> 1, queue => "DIRECT", type => "task", summary => "Новая задача", description => "большое описание")'


    посмотреть смены статусов тикета:
    perl -MYAML -MStartrek::Client::Easy -e '$st=Startrek::Client::Easy->new(); @ch = grep {$_->{type} eq "IssueWorkflow"} @{$st->request("GET", "/issues/DIRECTMOD-2570/changelog")}; print YAML::Dump(\@ch);' 

    Посмотреть для тикетов в определенном статусе время перехода в этот статус (более точно -- время последней смены статуса).
    Последним аргументом указывается статус тикетов.
    Однострочник выводит id тикета, время последней смены статуса, заголовок, сортирует тикеты по дате последней смены статуса.
    perl -MStartrek::Client::Easy -MDate::Parse -Mutf8 -Mopen=:std,:utf8 -e '$st=Startrek::Client::Easy->new(); for $p (1 .. 100){$found = $st->get(query => qq!Queue: DIRECT Status: $ARGV[0]!, page => $p, perPage => 100); last unless scalar @$found; $issues{$_->{key}} = $_ for @$found;}  for $key (keys %issues) { @ch = grep {$_->{type} =~ /^IssueCreated|IssueWorkflow|IssueUpdated$/} @{$st->request("GET", "/issues/$key/changelog")}; $time = eval{str2time(@ch[-1]->{updatedAt}) }|| 0; push @res, {key => $key, updated_at => $time}; } print map { "$_->{key}\t".localtime($_->{updated_at})." \t$issues{$_->{key}}->{summary}\n" } sort {$b->{updated_at} <=> $a->{updated_at}} @res; ' Beta-tested

=head1 DESCRIPTION

    Клиент для Стартрека.

=cut

use strict;
use warnings;

use Encode;
use HTTP::Request;
use JSON;
use LWP::UserAgent;

use ProjectSpecific qw/ get_startrek_client_token /;

use utf8;

our @ISSUE_MAIN_FIELDS = qw(
    assignee
    description
    queue
    summary
    type
    components
    commited
    branch
    tags
    6061b65f6669ed0a30e9e041--sandboxBuildTask
);

our $USE_DEFAULT_ASSIGNEE = 1;

=head1 METHODS

=head2 new

    Конструктор.
    Именованные параметры:

        token

            OAuth-токен пользователя, от имени которого будут выполняться действия в Стартреке.
            Если параметр не указан, то будет вызвана соответствующая функция из ProjectSpecific.

        startrek
            
            'prod' или 'test' для доступа к "боевому" или тестовому Стартреку соответственно, по умолчанию prod

=cut

sub new {
    my $class = shift;
    my %O = @_;

    $O{startrek} //= 'prod';

    my $self = {
        token   => $O{token} || get_startrek_client_token(),
        api_url => $O{api_url} || ProjectSpecific::get_data(startrek_api_url => $O{startrek}) || die "unknown Startrek instance",
    };

    return bless $self, $class;
}

=head2 request


=cut

sub request {
    my ($self, $http_method, $path, $body) = @_;
    $http_method = uc $http_method;

    my $ua = LWP::UserAgent->new(agent => ProjectSpecific::get_project_data('user_agent_for_startrek'));
    $ua->default_header(Authorization => "OAuth $self->{token}");

    my $url = $self->{api_url} . $path;

    my $response;
    if ($http_method eq 'GET') {
        $response = $ua->get($url);
    } elsif ($http_method =~ /^(POST|PATCH)$/) {
        $body //= '';
        if (ref $body eq 'HASH' || ref $body eq 'ARRAY') {
            $body = encode_json($body);
        }
        my $r = HTTP::Request->new($http_method, $url, ['Content-Type' => 'application/json'], $body);
        $response = eval { $ua->request($r) };
        if ($@) {
            die "Unable to send request to $self->{api_url}: $@\n";
        }
    } else {
        die "$http_method not implemented";
    }

    # почему-то в $response->decoded_content хранится нераскодированная строка (то есть то же самое, что в content)
    # поэтому раскодируем вручную
    die Encode::decode_utf8($response->content) unless $response->is_success;
    return decode_json($response->content);
}

=head2 get

    Найти тикеты в Стартреке.
    Именованные параметры:
        
        key

            Ключ или ссылка на массив ключей тикетов, информацию о которых нужно получить.

        query

            Запрос для Стартрека.

        limit

            Если указана опция query, определяет ограничение количества тикетов, выдаваемых по запросу.
            превращается в параметр perPage Стартрекового api

        perPage
            если указана опция query, определяет количество тикетов "на странице". См. также page

        page
            если указана опция query, определяет, которую по счету "страницу" результатов надо вернуть. Нумерация начинается с 1

        array

            Возвращать ссылку на массив, даже если ищем один тикет.


    Пример возвращаемого значения для одного тикета:

        {
          'priority' => 'normal',
          'createdBy' => 'pavryabov',
          'status' => 'open',
          'components' => [
                            '8371'
                          ],
          'self' => 'https://st-api.test.yandex-team.ru/v2/issues/DIRECT-31943',
          'key' => 'DIRECT-31943',
          'assignee' => 'pankovpv',
          'createdAt' => '2014-05-26T16:26:18.000+0000',
          'summary' => "Добавить в интапи установку скидки клиенту",
          'id' => '53838606e4b0f29179f0bc87',
          'followers' => [
                           'omaz'
                         ],
          'stand' => "не указано",
          'updatedAt' => '2014-05-26T16:35:56.000+0000',
          'updatedBy' => 'pankovpv',
          'version' => '1401165416824',
          'autotesting' => "Не определено",
          'description' => "Сейчас скидку клиенту можно установить только в интерфейсе, послав фейковую нотификацию о скидке.

Для регрессионных автотестов API необходима ручка для установки скидки",
          'commited' => "Нет",
          'manualTesting' => "Не определено",
          'queue' => 'DIRECT',
          'type' => 'task',
          'qaEngineer' => 'pavryabov',
          'testScope' => "Да"
        };

=cut

sub get {
    my ($self, %O) = @_;

    die "issue key(s) or query needed" if !$O{key} && !$O{query};

    my @res;

    if ($O{key}) {
        my $keys = ref $O{key} eq 'ARRAY' ? $O{key} : [$O{key}];
        for my $key (@$keys) {
            my $issue = $self->request('GET', "/issues/$key");
            push @res, $issue;
        }
    }
    
    if ($O{query}) {
        my $body = encode_json({ query => $O{query} });
        my $path = '/issues/_search';

        my @paging_params;
        die "'perPage' and 'limit' can't be used together" if $O{limit} && $O{perPage};
        if ($O{limit}) {
            push @paging_params, "perPage=$O{limit}";
        }
        if ($O{perPage}) {
            push @paging_params, "perPage=$O{perPage}";
        }
        if ($O{page}) {
            push @paging_params, "page=$O{page}";
        }
        if (scalar @paging_params){
            $path .= "?".join("&", @paging_params);
        }

        push @res, @{ $self->request('POST', $path, $body) };
    }

    # В возвращаемой API структуре некоторые поля выглядят так:

    # "status" : {
    #     "self" : "https://st-api.test.yandex-team.ru/v2/statuses/3",
    #     "id" : "3",
    #     "key" : "closed",
    #     "display" : "Closed"
    #  }
 
    # Использовать такие структуры в коде неудобно и не нужно, поэтому выбираем только поле key или id, если нет key (например, для логинов и компонент)
    for my $issue (@res) {
        for my $key (keys %$issue) {
           if (ref $issue->{$key} eq 'HASH') {
               $issue->{$key} = $issue->{$key}->{key} || $issue->{$key}->{id};
           } elsif (ref $issue->{$key} eq 'ARRAY') {
               $issue->{$key} = [ map { ref $_ eq 'HASH' ? ($_->{key} || $_->{id}) : $_ } @{ $issue->{$key} } ];
           }
        }
    }
   
    return @res == 1 && !$O{array} ? $res[0] : \@res;
}


=head2 do

=cut

sub do {
    my ($self, %O) = @_;

    my $updates = $self->_calculate_updates(%O);

    $self->_apply_updates($updates);

    return $O{create} ? $updates->{key} : '';
}


sub _calculate_updates {
    my ($self, %OPT) = @_;

    my $res = {};
    my $creating = $OPT{create} ? 1 : 0;

    unless ($creating) {
        $res->{key} = $OPT{key};
        $res->{old_values} = $self->get(key => $OPT{key});
    }

    if ($creating) {
        $OPT{type}     ||= 'bug';
        if ($USE_DEFAULT_ASSIGNEE) {
            $OPT{assignee} ||= [getpwuid($<)]->[0];
        }
    }

    my $mainfields_dest = $creating ? 'create' : 'new_values';

    # основные поля -- создать или обновить
    for my $f (@ISSUE_MAIN_FIELDS) {
        $res->{$mainfields_dest}->{$f} = $OPT{$f} if $OPT{$f}
    }

    # Internal
    if ($OPT{internal}){
        $res->{new_values}->{autotesting}   = 'Тестирование не требуется';
        $res->{new_values}->{testScope}     = 'нет';
        $res->{new_values}->{manualTesting} = 'Не требуется проверка';
        push @{$res->{comment}}, 'исправления внутренние, тестирование не требуется'
    }
   
    # Cc 
    if ($OPT{followers}) {
        $res->{new_values}->{followers} = ref $OPT{followers} eq 'ARRAY' ? $OPT{followers} : [ $OPT{followers} ];
    }

    # Comment
    push @{$res->{comment}}, $OPT{comment} if $OPT{comment};

    $res->{comment} = join("\n", @{$res->{comment} || []});

    $res->{actions} = $OPT{actions}    if $OPT{actions};
    push @{$res->{actions}}, 'resolve' if $OPT{resolve};
    push @{$res->{actions}}, 'ready_for_test' if $OPT{ready_for_test};
    push @{$res->{actions}}, 'close'   if $OPT{close};

    $res->{resolution} = $OPT{resolution} || 'fixed' if $OPT{close} || $OPT{resolve};

    return $res;
}

sub _apply_updates
{
    my ($self, $upd) = @_;

    # Применяем все нужные обновления:
    # 1. Создаем
    if (exists $upd->{create}) {
        my $issue = $self->request('POST', '/issues', $upd->{create});
        $upd->{key} = $issue->{key};
    }

    # 2. Обновляем
    if (exists $upd->{new_values}) {
        $self->request('PATCH', "/issues/$upd->{key}", $upd->{new_values});
    }

    # 3. Комментируем
    if ($upd->{comment}) {
        $self->request('POST', "/issues/$upd->{key}/comments", { text => $upd->{comment} });
    }

    # 4. Выполняем workflow actions
    if (exists $upd->{actions}) {
        for my $act (@{$upd->{actions}}) {
            my $body = ($act eq 'close' || $act eq 'resolve' || $act eq 'ready_for_test') ? { resolution => $upd->{resolution} } : '';
            $self->request('POST', "/issues/$upd->{key}/transitions/$act/_execute", $body);
        }
    }

    return;
}

1;
