#!/usr/bin/perl

use strict;
use warnings;
use Getopt::Long;
use POSIX qw(strftime);
use YAML;
use Data::Dumper;

my $CONFDIR = '/etc/stderrout-merger';
my $MERGED_LOGS_DIR = '/var/log/stderrout';
GetOptions(
    "config-dir=s" => \$CONFDIR,
);

logmsg("script started");
opendir(my $conf_dh, $CONFDIR);
while (readdir $conf_dh) {
    next if ! /[\w-]+\.conf/;
    my $conf = load_config("$CONFDIR/$_");
    next if ! $conf;
    print "options: " . Dumper($conf) if $ENV{DEBUG};
    logmsg("parse conf $_ ...");
    merge_dir($conf);
}
logmsg("script finished");

sub merge_dir {
    my ($conf) = @_;

    my %logs;
    my %opened;
    my $log_re = qr($conf->{unmerged_regex});

    map { /($log_re)/ && do { $opened{$1} = 1 } } split /\n/, qx(lsof +D $conf->{unmerged_logs_dir});
    print "opened logs: " . Dumper(\%opened) if $ENV{DEBUG};

    opendir(my $dh, $conf->{unmerged_logs_dir}) || die "can't open $conf->{unmerged_logs_dir}: $!";
    while (readdir $dh) {
        if ($opened{$_} || ! /$log_re/) {
            print "already opened or not matched: $_\n" if $ENV{DEBUG};
            next;
        }
        my ($log_prefix, $start_time, $date_suffix) = ($2, $3, $4);

        my $merged = "$MERGED_LOGS_DIR/$log_prefix.stderrout.log.$date_suffix";
        my $unmerged = "$conf->{unmerged_logs_dir}/$_";
        $logs{$unmerged} = { prefix => $log_prefix, merged => $merged, start_time => $start_time, date_suffix => $date_suffix };
    }
    print "to merge: " . Dumper(\%logs) if $ENV{DEBUG};

    for my $unmerged (sort { $logs{$a}{start_time} cmp $logs{$b}{start_time} } keys %logs) {
        my $merged = $logs{$unmerged}{merged};
        my ($merged_fh, $unmerged_fh);
        if (! open($unmerged_fh, '<', $unmerged)) {
            warn "can't open $unmerged: $!";
            next;
        }
        if (! open($merged_fh, '>>', $merged)) {
            warn "can't open $merged: $!";
            next;
        }

        print $merged_fh "#### $logs{$unmerged}{start_time} start stderrout log: $logs{$unmerged}{prefix}\n";
        while (my $line = <$unmerged_fh>) {
            print $merged_fh $line;
        }
        print $merged_fh "#### $logs{$unmerged}{start_time} end stderrout log: $logs{$unmerged}{prefix}\n";
        print "merged $unmerged with $merged\n" if $ENV{DEBUG};

        close($unmerged_fh);
        unlink $unmerged || warn "can't remove $unmerged: $!";
        print "removed $unmerged\n" if $ENV{DEBUG};
    }
}

sub load_config {
    my $file = $_[0];
    my $yaml_str;
    {
        local $/= undef;
        my $fh;
        if (! open($fh, '<', $file)) {
            warn "Can't open $file: $!";
            return undef;
        }
        $yaml_str = <$fh>;
        close($fh);
    }
    return Load($yaml_str);
}

sub logmsg {
    chomp(my $date = qx(date));
    print "$date @_\n";
}
