package SvnRelease::MkFiles;

# $Id$

=head1 NAME

    SvnRelease::MkFiles - внутренний модуль для семейства утилит mk-*-packages-files

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use Data::Dumper;
use Getopt::Long::Descriptive;
use File::Slurp;
use File::Path;
use File::Basename;

use Template;

sub new {
    my (undef, %opt) = @_;
    my $self = bless {
        files => $opt{files},
        substvars => $opt{substvars},
    };
    if ($opt{templates_fh}) {
        $self->separate_template($opt{templates_fh});
    }
    $self->{conf} = grep {$_->{conf}} @{$opt{files}};
    return $self;
}

sub separate_template {
    my ($self, $fh) = @_;
    # считываем шаблоны управляющих файлов из скрипта
    $self->{templates} = {};
    my $tmpl_name;
    while(<$fh>) {
        if (/^<<\s*(\w+)/) {
            $tmpl_name = $1;
            $self->{templates}->{$tmpl_name} = '';
        } elsif ($tmpl_name) {
            $self->{templates}->{$tmpl_name} .= $_;
        }
    }
}

sub parse_options {
    my ($self, $params, %O) = @_;
    my @templates_names = sort grep {defined} map {$_->{tmpl}} @{$self->{files}};
    my ($opt, $usage) = describe_options(
        "%c %o package_build_dir $params",
        [ 'help|h' => 'помощь' ],
        [ 'template|t=s%', 'файлы с шаблоном для '.join(", ", @templates_names).". Пример: -t $templates_names[0]=../$templates_names[0].tt2", {default => {}} ],
        [ 'define|d=s%', "опции для шаблонов конфига. Пример: -d variable=value", {default => {}}],
        ($self->{conf} ? [ 'conf|c=s%', "альтернативные конфигурации, имя=директория_сборки. Пример: -c test=./debian/yandex-conf-test", {default => {}}] : () ),
        );
    $self->{usage} = $usage;
    if ($opt->help) {
        $self->usage();
    } elsif (!@ARGV || !-d $ARGV[0]) {
        $self->usage('First parameter is not directory');
    } elsif ($self->{conf} && keys %{$opt->conf}) {
        if (my @failed_dirs = grep {!-d} values %{$opt->conf}) {
            $self->usage("Directory don't exists: ".join(", ", @failed_dirs));
        }
    }
    $self->{dir} = shift @ARGV;
    $self->{vars} = $opt->define;
    $self->{opt} = $opt;
    return @ARGV;
}

=head2 $self->confs()

    Получить ссылку на хэш: название конфигурации -> директория

=cut
sub confs {
    my ($self) = @_;
    return $self->{conf} ? $self->{opt}->conf : {};
    
}

sub usage {
    my ($self, $str) = @_;
    if (defined $str) {
        print "$str\n";
    } else {
        system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2");
    }
    if ($self->{usage}) {
        print "USAGE\n$self->{usage}";
    }
    exit 1;
}

sub mk {
    my ($self, $files) = @_;
    $self->mkfiles();
    $self->mksubstvars();
}

sub mkfiles {
    my ($self, $files) = @_;

    my $t = Template->new( RELATIVE => 1, ABSOLUTE => 1);

    for my $file (@{$files || $self->{files}}) {
        for my $conf (undef, $file->{conf} ? keys %{$self->{opt}->conf} : ()) {
            $self->{vars}->{conf} = $conf;
            if ($file->{cond} && !$file->{cond}->($self->{vars})) {
                next;
            }
            my $dir = defined $conf ? $self->{opt}->conf->{$conf} : $self->{dir};
            my $filename;
            if ($file->{file}) {
                $t->process(\$file->{file}, $self->{vars}, \$filename) || die $t->error;
                $filename = "$dir/$filename";
                mkpath(dirname("$filename"));
                my $text;
                if ($self->{opt}->template->{$file->{tmpl}}) {
                    $t->process($self->{opt}->template->{$file->{tmpl}}, $self->{vars}, \$text) || die $t->error;
                } else {
                    $t->process(\($self->{templates}->{$file->{tmpl}}), $self->{vars}, \$text) || die $t->error;
                }
                write_file($filename, $text);
            } elsif ($file->{dir}) {
                $t->process(\$file->{dir}, $self->{vars}, \$filename) || die $t->error;
                $filename = "$dir/$filename";
                mkpath([$filename]);
            } else {
                die "Neither file nor dir defined: ".Dumper($file);
            }
            if ($file->{perm}) {
                chmod($file->{perm}, "$filename") || die "Can't chmod $filename: $!";
            }
            if ($file->{owner}) {
                my $owner;
                $t->process(\$file->{owner}, $self->{vars}, \$owner) || die $t->error;
                my ($login, $group) = split ':', $owner, 2;
                my (undef, undef, $uid) = getpwnam($login)
                    or die "user $login not in passwd file";
                my (undef, undef, $gid) = getgrnam($group)
                    or die "group $group not in groups file";
                if (!chown($uid, $gid, $filename)) {
                    die "Can't chown $uid:$gid for $filename";
                }
            }
        }
    }
}

sub mksubstvars {
    my ($self) = @_;

    return unless $self->{substvars} && keys %{$self->{substvars}};
    my %vars = %{$self->{substvars}};

    for my $key (keys %vars) {
        if (ref $vars{$key} eq 'CODE') {
            $vars{$key} = $vars{$key}->($self->{vars});
        }
    }

    # превращаем debian/yandex-direct/ -> debian/yandex-direct.substvars
    (my $subst_file = $self->{dir}) =~ s/\/*$/.substvars/;

    # если есть файл - считываем и парсим его
    if (-e $subst_file) {
        for my $line (grep {$_} map {s/\s*$//; $_} read_file($subst_file)) {
            my ($name, $val) = split /\s*=\s*/, $line, 2;
            $vars{$name} = $val unless exists $vars{$name};
        }
    }

    # пишем файл на диск
    my @content = map {"$_=$vars{$_}\n"} sort keys %vars;
    write_file($subst_file, {atomic => 1}, @content);
}

1;
