#!/usr/bin/perl -w

=head1 NAME
    
    direct-svn-up.pl

=head1 SYNOPSIS

    direct-svn-up.pl [<working_copy_path>] [<svn_url>|<revision>] [startrek:<release_number>]

=head1 DESCRIPTION

    Переключает указанную рабочую копию на указанный svn_url или обновляет до указанной ревизии <revision>
    svn_url может быть задан неявно, через версию Директ-пакетов: 
      1.11234-1       ~~ $DT@11234
      1.11234.11256-1 ~~ $DRT/releases/release-11234@11256

    Если svn_url не указан -- делает svn up в указанной рабочей копии

    Если указан параметр startrek:X -- делает апдейт до релиза по стартреку. Нумерация релизов относительная: 0/без номера - последний, -1 - предпоследний и т.д.

    Если при переключении/обновлении в рабочей копии поменялись хоть какие-нибудь файлы -- перезапускает apache. 

    Если не указана рабочая копия -- делает svn up + рестарт apache в текущем каталоге

    Опции:
        -h, --help
            показать этот текст и завершиться
        -v, --verbose
            выводить подробную информацию о выполнении
            флаг, т.е. можно писать отрицание: --no-verbose
        -rr, --restart
            перезапускать apache после обновления
            по умолчанию опция включена, для отключения нужно использовать отрицание: --no-restart

    TODO 
      может быть, restart не делать, а сигнализировать про результат кодом возврата? Чтобы можно было direct-svn-up.pl && restart

=cut

# $Id$


use strict;

use Getopt::Long;
use Scalar::Util qw(looks_like_number);

use ProjectSpecific qw/svn_url/;
use Startrek::Client::Easy;
use Yandex::Svn qw/svn_info/;

use utf8;
use open ':locale';

#.......................................................
{
my $VERBOSE = 0;

sub p{print @_,"\n" if $VERBOSE;}

sub set_verbose{$VERBOSE = shift;}
}

#.......................................................



my %OPT = parse_options();

usage() unless $OPT{wc_path};

chdir $OPT{wc_path} or die "can't cd to $OPT{wc_path}";

check_paths(%OPT);

my $needs_restart = update_wc(%OPT);

p "needs restart: $needs_restart";
if ( $needs_restart && $OPT{restart} ){
    my $apache_dir = -d 'apache' ? 'apache' : 'apache2';
    my $r_result = qx!rm -f $apache_dir/logs/* ;$apache_dir/init.sh restart!; 
    p $r_result;
}

exit();


#......................................................

# проверяет, что текущий и новый svn url подходящие, ревизия хорошая
# если проверка не проходит -- умирает
sub check_paths
{ 
    my %OPT = @_;

    # 1. Проверить, что находимся в корне репозитория
    my $svn_info = svn_info('.');
    my ($repository_root) = $svn_info->{repository_root};
    my ($current_svn_url) = $svn_info->{url};
    p "\$repository_root: '$repository_root'";
    p "\$current_svn_url: '$current_svn_url'";
    p "new svn url: '$OPT{svn_url}'";

    my $root_url = svn_url('root');
    my $branch_name_re = '[^/]+';
    my $release_base_rev_re = '[0-9]+';
    my $trunk_url = svn_url('trunk');
    (my $branch_re = quotemeta(svn_url(branch => 'PLACEHOLDER'))) =~ s/PLACEHOLDER/$branch_name_re/;
    (my $release_branch_re = quotemeta(svn_url(release => 'PLACEHOLDER'))) =~ s/PLACEHOLDER/$release_base_rev_re/;
    die "current svn url should be ".svn_url('trunk').", or ".svn_url(branch=>"smth").", or ".svn_url(release=>'NNNNN') unless $current_svn_url =~ m!^(\Q$trunk_url\E|$branch_re|$release_branch_re)/*$!;

    return "" unless $OPT{svn_url};

    # 2. Проверить, что переключаемся тоже на подходящую ветку
    die "new svn url should be ".svn_url('trunk').", or ".svn_url(branch=>"smth").", or ".svn_url(release=>'NNNNN') unless $OPT{svn_url} =~ m!^(\Q$trunk_url\E|$branch_re|$release_branch_re)/*$!;

    die "revision should be a number" unless $OPT{svn_revision} =~ /^\d*$/;

    return "";
}


# svn up
# результат -- 0 или 1 (флаг -- было что-то обновлено?)
sub update_wc
{
    my %OPT = @_;

    my $svn_cmd;
    if (!$OPT{svn_url}) {
        $svn_cmd = "svn up" . ($OPT{svn_revision} ? " -r$OPT{svn_revision}" : "");
    } else {
        my $new_svn_path = $OPT{svn_url} .($OPT{svn_revision} ? "\@$OPT{svn_revision}" : "");
        $svn_cmd = "svn switch $new_svn_path";
    }

    p $svn_cmd;

    my $up_result = qx/LC_ALL=C $svn_cmd | grep -v 'External at revision ' | grep -v '^\$' | grep -v '^Fetching external' | grep -v '^Updated to revision ' |grep -v 'At revision '/;
    p $up_result;
    my $revision = svn_info($OPT{wc_path})->{revision}; # надёжнее, чем грепать вывод $svn_cmd
    p "revision: $revision";

    return $up_result ? 1 : 0;
}


# разбирает параметры командной строки. Первый -- путь к рабочей копии (обязательно), второй -- бранч/версия (необязательно)
sub parse_options
{
    my %O = (
        verbose       => 0, 
        svn_url       => '',
        svn_revision  => '',
        restart       => 1,
    );

    GetOptions (
        "h|help"            => \&usage,
        "v|verbose!"        => \$O{verbose},
        "rr|restart!"       => \$O{restart},
    );
    set_verbose($O{verbose});

    $O{wc_path} = $ARGV[0] || '.';

    return %O unless $ARGV[1];

    if ($ARGV[1] =~ /^[0-9]+$/) {
        # 12345, 68954
        $O{svn_revision} = $ARGV[1];
    } elsif ($ARGV[1] =~ /^1\.([0-9]+)(?:\.([0-9]+))?-[0-9]+$/) {
        my $version = $ARGV[1];
        $O{svn_url} = ProjectSpecific::version2svnpath($version);
        $O{svn_revision} = ProjectSpecific::version2rev($version, rev => 'last');
    } elsif ($ARGV[1] =~ /^startrek:([\w_-]+)$/) {
        my $st_index = $1;
        $st_index = -1 if $st_index eq 'last';
        $st_index = -2 if $st_index eq 'before-last';

        die("Wrong startrek release index value, it should be 'last', 'before-last', or integer < 0\n") unless looks_like_number($st_index) && $st_index < 0;
        die("Startrek release index should be >= -10\n") if $st_index < -10;

        my $queue_name = ProjectSpecific::get_project_data('default_tracker_queue');
        my $components = ProjectSpecific::get_project_data('startrek_release_component') // '';
        $components = qq(Components: "$components") if $components;

        my $st = Startrek::Client::Easy->new();
        my $release_tickets = $st->get(
            query => qq(
                Queue: $queue_name
                Type: Release
                $components
                "Sort By": Created DESC
            ),
            array => 1,
        );
        my $release_ticket = @$release_tickets[ -1 - $st_index];
        my $version = ($release_ticket->{'summary'} =~ /(\d+(?:\.\d+)?\.\d+-\d+)/)[0];
        print "version: $version\n";

        $O{svn_url} = ProjectSpecific::version2svnpath($version);
        $O{svn_revision} = ProjectSpecific::version2rev($version, rev => 'last');
    } else {
        # $DT, $DB/..., $DT@nnnnn
        $ARGV[1] =~ /^([^@]*)(?:\@(.*))?$/;

        $O{svn_url} = $1;
        $O{svn_revision} = $2 || '';
    }

    return %O;
}


# Печатает usage-сообщение
sub usage {
    system("podselect -section SYNOPSIS -section DESCRIPTION $0 | pod2text-utf8 >&2");
    exit(1);
}


