#!/usr/bin/perl -w

# $Id$

=head1 NAME

    old-svn-branches.pl -- рекомендации по удалению/архивированию старых бранчей в svn-репозитории

=head1 DESCRIPTION

    Опции:
    --help 
        справка

    -c, --conf-dir <path>
        каталог с конфигами

    -p, --send-personal-emails
        отправлять индивидуальные письма пользователями, которые не дали разрешение на автообработку своих бранчей

    -e, --send-general-email
        отправлять общий отчет (адрес берется из конфига)

    -q --quiet
        не выводить отчет на stdout (полезно для запуска из cron'а)

=cut

use strict;
use Getopt::Long;
use Pid::File::Flock qw/:auto/;
use Data::Dumper;
use YAML;
use POSIX;

use Yandex::Svn;
use Yandex::SendMail;

use utf8;
use open ':std' => ':utf8';

run() unless caller();

sub run
{
    my ($SEND_EMAIL, $SEND_PERSONAL_EMAILS, $CONF_DIR, $QUIET);
    GetOptions(
        "help" => sub {system("podselect -section NAME -section DESCRIPTION $0 | pod2text-utf8 >&2"); exit 0;},
        "e|send-general-email!" => \$SEND_EMAIL,
        "p|send-personal-emails" => \$SEND_PERSONAL_EMAILS,
        "c|conf-dir=s" => \$CONF_DIR,
        "q|quiet" => \$QUIET,
    ) || die "can't parse options";

    $CONF_DIR ||= '/etc/old-svn-branches';

    my @conf;
    opendir(my $c_dh, $CONF_DIR) || die "Can't open dir $CONF_DIR: $!";
    for my $file (sort grep {!/^\./} readdir($c_dh)) {
        push @conf, YAML::LoadFile("$CONF_DIR/$file");
    }

    for my $c (@conf){
        eval{
            process_conf(
                $c, 
                send_general_email => $SEND_EMAIL, 
                send_personal_emails => $SEND_PERSONAL_EMAILS,
                quiet => $QUIET,
            );
        };
        print STDERR $@ if $@;
    }

    exit;
}


sub process_conf
{
    my ($conf, %O) = @_;

    my $stat = svn_branches_stat($conf->{svn_root});

    my %stat;

    # считаем дату устаревания. 
    my $obsolescence_date = POSIX::strftime("%Y-%m-%d", localtime(time - 60*60*24*90));
    for my $br (@$stat){
        my $owner = $br->{create_author};
        if ( $br->{merged} ){
            $owner = 'AUTO' if $conf->{permissions}->{$owner}->{delete};
            push @{$stat{$owner}->{merged}}, $br;
        } elsif ( $br->{edit_date} le $obsolescence_date ){
            $owner = 'AUTO' if $conf->{permissions}->{$owner}->{archive};
            push @{$stat{$owner}->{old}}, $br;
        } 
    }

    # списки бранчей переводим в рекомендации "что делать"
    for my $owner (keys %stat){
        $stat{$owner}->{'to delete'} = 'OK';
        if (exists $stat{$owner}->{merged}){
            $stat{$owner}->{'to delete'} = "svn rm ". join " ", map {"$conf->{svn_root}/branches/$_->{branch}"} @{$stat{$owner}->{merged}};
        }
        delete $stat{$owner}->{merged};

        $stat{$owner}->{'to archive'} = 'OK';
        if (exists $stat{$owner}->{old}){
            $stat{$owner}->{'to archive'} = [];
            for my $br ( @{$stat{$owner}->{old}} ){
                my $o = '';
                $o = " by $br->{create_author}" if $owner ne $br->{create_author};
                push @{$stat{$owner}->{'to archive'}}, 
                join ("\n",
                    "$br->{branch}$o last edit $br->{edit_date}",
                    "svn mv $conf->{svn_root}/branches/$br->{branch} $conf->{svn_root}/archive/$br->{branch}",
                    "or",
                    "svn rm $conf->{svn_root}/branches/$br->{branch}",
                );
            }
        }
        delete $stat{$owner}->{old};
    }

    my $AUTO = delete $stat{AUTO};

    my $res = {
        general => $AUTO,
        personal => \%stat,
    };

    my $todo = YAML::Dump($res);

    if (!$O{quiet}){
        print $todo;
    } 

    local $Yandex::SendMail::FROM_FIELDS_FOR_ALERTS = $conf->{email};
    local $Yandex::SendMail::ENVELOPE_SENDER = $conf->{email};
    if ($O{send_general_email} && $conf->{email}){
        send_alert($todo, "устаревшие бранчи в $conf->{svn_root}", $conf->{email});
    }
    if ($O{send_personal_emails}){
        for my $login (sort keys %stat){
            next if $stat{$login}->{'to delete'} eq 'OK' && $stat{$login}->{'to archive'} eq 'OK';
            my $personal_todo = YAML::Dump($stat{$login});
            my $text = qq[Привет! 

У тебя есть устаревшие бранчи в репозитории $conf->{svn_root} .
Пожалуйста, удали их или перенеси в архив. 

Список бранчей и рекомендуемых действий: 

$personal_todo

Или доверь удаление/архивирование своих старых бранчей автоматике, см. письмо "новое про старые бранчи" на direct-dev\@yandex-team.ru от 2011-11-02.

Вопросы и предложения -- к $conf->{responsible_person} .

-- 
Робот-Бранчеед
];
            send_alert($text, "$login: устаревшие бранчи в $conf->{svn_root}", "$login\@yandex-team.ru");
        }
    }

    return;
}

