#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import argparse
import sys
import os
import time
import signal
import shutil
import MySQLdb

from subprocess import PIPE, Popen

#команда для тестового запуска mysqld
CMDTEMPL = "/usr/sbin/mysqld --basedir=/usr --datadir=/opt/mysql.{0}{3} --plugin-dir=/usr/lib/mysql/plugin --open-files-limit=65536 --pid-file={2} --socket={1} --skip-networking --skip-external-locking --innodb_file_per_table --innodb_checksum_algorithm=crc32 --innodb_data_file_path=ibdata:128M:autoextend --innodb_file_format=Barracuda"
#для быстрого подключения по сокету
CMDMYSQL = "mysql -u root -S {0} {1}"
#для инициализации БД с нуля
CMDINITZ = "/usr/sbin/mysqld --datadir=/opt/mysql.{0} --basedir=/usr --user=mysql --initialize-insecure --skip-networking --skip-external-locking --innodb_file_per_table --innodb_checksum_algorithm=crc32 --innodb_data_file_path=ibdata:128M:autoextend --innodb_file_format=Barracuda"
#основоной datadir
MYSQLDIR = "/opt/mysql.{0}/data"
MYSQLDIROLD = "/opt/mysql.{0}"
#временный сокет для подключения к БД
MYSQLSOCK = "/tmp/{0}.sock"
MYSQLPID = "/tmp/{0}.pid"
#mysql может не стартануть с пустым конфигом при наличии некоторых продакшен файлов.
CLEANFILES = [ "xb_doublewrite", "ib_buffer_pool", "ib_logfile" ]

USAGE = '''Программа для подготовки mysql к работе в тестовой среде.
1. Чистится рабочая копия от продакшен баз.
2. Запускается временная БД с минимальными ресурсами.
3. Накатываются гранты ТС пользователей.
4. Создаются и чистятся дополнительные таблицы.

Если в ходе запуска БД, находятся старые работающие временные базы, то они останавливаются по TERM/KILL.
'''

#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch

#По указанным instance находим запущенные старые БД и возвращаем pids в виде списка.
#Если к указанному pid нет процесса, удаляем старый pid файл.
def checkRunProcess(instances):
    pids = []
    for instance in instances:
        pidfile = MYSQLPID.format(instance)
        if os.path.exists(pidfile):
            pid = open(pidfile).read().strip()
            if os.path.exists('/proc/{0}'.format(pid)):
                pids.append(pid)
            else:
                logger.info("not found running process from {0}. Remove file {1}".format(pid, pidfile))
                os.remove(pidfile)
    return pids

#Отправляет TERM/KILL сигнал указанным pid'ам процессов
def killMysqldProcess(pids, kill9=False):
    sig = signal.SIGKILL if kill9 else signal.SIGTERM
    [os.kill(int(pid), sig) for pid in pids]
    return

#Принимает команду, которая готовит mysqld дамп, и отправляет его в БД через сокет.
#В результате выводит статус выполнения(true/false) и ошибку(error).
def runMysqlDumpCommand(command, socket, db=""):
    try:
        logger.info("run command {0} with output to mysql socket {1} for db {2}".format(command, socket, db))
        fileno_logfile = logch.stream.fileno()
        p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=PIPE)
        p2 = Popen(CMDMYSQL.format(socket, db), shell=True, bufsize=0, stdin=p1.stdout, stderr=fileno_logfile, stdout=fileno_logfile)
        while True:
            rcode1 = p1.poll()
            rcode2 = p2.poll()
            if rcode1 is not None and rcode1 != 0:
                raise ValueError("error run command {0} code {1}".format(command, rcode1))
            if rcode2 is not None and rcode2 != 0:
                raise ValueError("error run command {0} code {1}".format(command, rcode2))
            if rcode1 == 0 and rcode2 == 0:
                break
        p1.wait()
        p2.wait()
    except Exception as err:
        return False, ValueError(err)
    return True, None

#Принимает команду на выполнение.
#В результате выводит статус выполнения(true/false) и ошибку(error).
def runMysqlShellCommand(command):
    try:
        logger.info("run command {0}".format(command))
        fileno_logfile = logch.stream.fileno()
        p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
        while True:
            rcode1 = p1.poll()
            if rcode1 is not None and rcode1 != 0:
                raise ValueError("error run command {0} code {1}".format(command, rcode1))
            if rcode1 == 0:
                 break
        p1.wait()
    except Exception as err:
        return False, ValueError(err)
    return True, None

#Принимает SQL команду, сокет и имя БД. Выводит результат и ошибку(error)
def runMysqlOutput(mysql_command, socket, db=""):
    try:
        db = MySQLdb.connect(host="localhost", user="root", passwd="", db="mysql", unix_socket=socket)
        cur = db.cursor()
        cur.execute(mysql_command)
        output = cur.fetchall()
    except Exception as err:
        return None, err
    finally:
        cur.close()
        db.close()
    return output, None

#Проверяем подключение к сокету mysqld. Возвращает стутус выполнения(true/false) и ошибку(error).
def checkMysqlConnect(socket):
    try:
        if not os.path.exists(socket):
            msg = "socket {0} dosnt exists".format(socket)
            return False, ValueError(msg)
        db = MySQLdb.connect(host="localhost", user="root", passwd="", db="mysql", unix_socket=socket)
        cur = db.cursor()
        cur.execute("SELECT NOW()")
        _ = cur.fetchall()
        cur.close()
        db.close()
        return True, None
    except Exception as err:
        return False, ValueError(err)

#чистим дирректории от не нужных старых файлов
def cleanMysqlDirs():
    for instance in instances:
        logger.info("start clean oldfiles for {0} dir".format(instance))
        mysqldir = MYSQLDIROLD.format(instance) if instance.startswith('ppclog') else MYSQLDIR.format(instance)
        for path, dirs, f1les in os.walk(mysqldir):
            removed_files = [ f for f in f1les for cl in CLEANFILES if f.startswith(cl) ]
            if len(removed_files) == 0: continue
            removed_path = [ os.path.join(path, f1le) for f1le in removed_files ]
            [ os.remove(i) for i in removed_path ]
            logger.info("finish clean oldfiles for {0} dir".format(instance))
    return

#прибиваем старые процессы, если таковый находятся.
def stopMysqlProc():
    retry = 0
    logger.info("start kill old mysql instances")
    while True:
        pids = checkRunProcess(instances)
        if len(pids) == 0: break
        if retry > 10:
            msg = "current retry kill process more 12"
            raise ValueError(msg)
        logger.info("kill old mysqld with pids {0} try {1}".format(pids, retry+1))
        _ = killMysqldProcess(pids)  if retry < 10 else killMysqldProcess(pids, True)
        retry += 1
        time.sleep(30)
    logger.info("finish kill old mysql instances")
    return

#запускаем новые процессы mysqld
def startMysqlProc():
    for instance in instances:
        logger.info("start run mysql instance {0}".format(instance))
        mysql_data_dir = "" if instance.startswith("ppclog") else "/data"
        command = CMDTEMPL.format(instance, MYSQLSOCK.format(instance), MYSQLPID.format(instance), mysql_data_dir)
        print command
        proc = Popen(command, shell=True, bufsize=0, stdin=PIPE, stdout=PIPE)
        logger.info("finish run mysql instance {0}".format(instance))
    return

#проверяем, что все БД запущены
def checkMysqlProc():
    retry = 0
    logger.info("start check running instances")
    while True:
        newpids = checkRunProcess(instances)
        if len(instances) == len(newpids):
            logger.info("success check running instances: {0}".format(newpids))
            break
        retry += 1
        logger.critical("not found running instances. waiting... retry {0}".format(retry))
        if retry > 10:
            msg = "failed check running instances: len instances {0} != len pids {2}".format(len(instances), len(newpids))
            logger.critical(msg)
            raise ValueError(msg)
        time.sleep(60)
    logger.info("finish check running instances")
    return

#и все сокеты готовы к работе
def checkMysqlSocket():
    for instance in instances:
        logger.info("start check socket {0}".format(instance))
        retry = 0
        while True:
            socket = MYSQLSOCK.format(instance)
            ok, err = checkMysqlConnect(socket)
            if ok:
                logger.info("success check socket {0}".format(instance))
                break
            retry += 1
            if retry > 10:
                msg = "failed connect socket {0}: {1}".format(socket, err)
                logger.critical(msg)
                raise ValueError(msg)
            time.sleep(60)
    logger.info("finish check socket {0}".format(instance))
    return

#применяем гранты для тестовой среды
def applyMysqlGrants():
    for instance in instances:
        logger.info("start apply grants for {0}".format(instance))
        cmd = "/usr/local/bin/generate-mysql-grant-statements.pl -i any --db-host any --conf /etc/sync-db/direct-test-mysql-grants.yaml"
        ok, err = runMysqlDumpCommand(cmd, MYSQLSOCK.format(instance))
        if not ok:
            raise ValueError("error apply grants for {0}: {1}".format(instance, err))
    logger.info("finish apply grants for {0}".format(instance))
    return 

def initializeMysqlDb():
    for instance in instances:
        if instance.startswith('ppclog'):
            mysql_data_dir = "/opt/mysql.{0}".format(instance)
            if os.path.exists(mysql_data_dir):
                shutil.rmtree(mysql_data_dir)
            logger.info("start initialize ppclog database {0}".format(instance))
            command = CMDINITZ.format(instance)
            proc = Popen(command, shell=True, bufsize=0, stdin=PIPE, stdout=PIPE)
            _, err = proc.communicate()
            if err is not None:
                raise ValueError("error initialize database {0}: {1}".format(instance, err))
            logger.info("finish initialize ppclog database {0}".format(instance))
    return

#накатываем дополнительные данные для тестовой среды
def applyExtraScripts():
    for instance in instances:
        socket = MYSQLSOCK.format(instance)
        if instance.startswith('ppcdata'):
            ###Step 1
            logger.info("start apply direct-fake-fio for {0}".format(instance))
            cmd = "cat /etc/sync-db/direct-fake-fio.sql"
            ok, err = runMysqlDumpCommand(cmd, socket, "ppc")
            if not ok:
                raise ValueError("error apply direct-fake-fio for {0}: {1}".format(instance, err))
            logger.info("finish apply direct-fake-fio for {0}".format(instance))
            ###Step 2
            logger.info("start apply direct-fake-freelancers for {0}".format(instance))
            cmd = "cat /etc/sync-db/direct-fake-freelancers.sql"
            ok, err = runMysqlDumpCommand(cmd, socket, "ppc")
            if not ok:
                raise ValueError("error apply direct-fake-freelancers for {0}: {1}".format(instance, err))
            logger.info("finish apply direct-fake-freelancers for {0}".format(instance))

        if instance.startswith('ppcdict'):
            ###Step 3
            logger.info("start update tables for {0}".format(instance))
            cmd = "/usr/local/bin/sync-db-direct-prepare-ppcdict.pl {0}".format(socket)
            ok, err = runMysqlShellCommand(cmd)
            if not ok:
                raise ValueError("error update tables for {0}: {1}".format(instance, err))
            logger.info("finish update tables for {0}".format(instance))

        if instance.startswith('ppclog'):
            ###Step 4
            logger.info("start update tables for {0}".format(instance))
            cmd = "cat /etc/sync-db/direct-ppclog-struct.sql"
            ok, err = runMysqlDumpCommand(cmd, socket, "mysql")
            if not ok:
                raise ValueError("error apply direct-ppclog-struct for {0}: {1}".format(instance, err))
            logger.info("finish apply direct-ppclog-struct for {0}".format(instance))
    return

def main():
    cleanMysqlDirs() #чистим дирректории от не нужных старых файлов
    stopMysqlProc() #прибиваем старые процессы, если таковый находятся.
    initializeMysqlDb() #инициализируем БД, если таковых нет в бекапах
    startMysqlProc() #запускаем новые процессы
    checkMysqlProc() #проверяем, что все БД запущены
    checkMysqlSocket() #и все сокеты готовы к работе
    applyMysqlGrants() #применяем гранты для тестовой среды
    applyExtraScripts() #накатываем дополнительные данные для тестовой среды
    stopMysqlProc() #останавливаем mysql
    cleanMysqlDirs() #удаляем все что нагенерировалось
    return

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description=USAGE,
                    epilog='Пример: %(prog)s ppcdata8 ppcdata9')
    parser.add_argument("-d", "--debug", action="store_true",
                    dest='debug', help="включить debug режим")
    parser.add_argument("instances", nargs='*', type=str, action='store',
            help="список инстансов mysqld(например: ppcdata1, ppcdata2.new)")
    opts = parser.parse_args()

    if len(opts.instances) == 0:
        parser.print_help()
        sys.exit(1)

    log_level = "DEBUG" if opts.debug else "INFO"
    logger, logch = startLogging(level=log_level)
    instances = opts.instances
    main()

