#!/usr/bin/env python
# *-* encoding: utf-8 *-*

import argparse
import sys
import os
import shutil
import json
import signal
import socket
import time
import logging

from subprocess import Popen

CONFIG = "/etc/haproxy/ppctest-mysql-{0}.cfg"
INITD = "/etc/init.d/haproxy-mysql-{0}"

TEMPLATE_HEAD = '''
#config generated by script {0}
global
    log /dev/log    local0
    pidfile /var/run/ppctest-mysql-{1}.pid
    maxconn 16384
    #chroot /usr/share/haproxy
    user haproxy
    group haproxy
    daemon
    debug

defaults
        log     global
        mode    tcp
        option  dontlognull
        retries 3
        option  redispatch
        maxconn 16384
        # Set the maximum time to wait for a connection attempt to a server to succeed
        timeout connect     30s
        # Set the maximum inactivity time on the client side.
        timeout client      24h
        # Set the maximum inactivity time on the server side.
        timeout server      24h
        option clitcpka
        option srvtcpka
        default-server on-marked-down shutdown-sessions on-marked-up shutdown-backup-sessions
'''

TEMPLATE_BODY = '''
listen {0}
    bind :::{1} v4v6
        server {0}-{3} {2}:{1}
'''

USAGE = '''Генерирует init.d сприпт, конфиг haproxy на основе json данных и добавляет в автозапуск systemv.

Например для команды:
%(prog)s --doit '[{"host":"ppclogshutter01f.ppc.yandex.ru","instance":"ppcdict","port":3307,"group":"dev1"},{"host":"ppclogshutter01f.ppc.yandex.ru","instance":"ppcmonitor","port":3308,"group":"dev1"},{"host":"ppclogshutter01f.ppc.yandex.ru","instance":"sandbox","port":3306,"group":"dev1"}]'
Будет сгенерирован конфиг /etc/haproxy/ppctest-mysql-dev1.cfg + /etc/init.d/haproxy-mysql-dev1 и выполнен /usr/sbin/update-rc.d haproxy-mysql-dev1 defaults 60 20
'''

TEMPLATE_INITD = '''#!/bin/sh
### BEGIN INIT INFO
# Provides:          haproxy
# Required-Start:    $local_fs $network $remote_fs $syslog $named
# Required-Stop:     $local_fs $remote_fs $syslog $named
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: fast and reliable load balancing reverse proxy
# Description:       This file should be used to start and stop haproxy.
### END INIT INFO

#generated by script FIXNAME

PATH=/sbin:/usr/sbin:/bin:/usr/bin
PIDFILE=/var/run/ppctest-mysql-FIXGROUP.pid
CONFIG=/etc/haproxy/ppctest-mysql-FIXGROUP.cfg
HAPROXY=/usr/sbin/haproxy
RUNDIR=/run/haproxy
EXTRAOPTS=

export LD_PRELOAD=/usr/lib/libtcmalloc_minimal.so.4

test -x $HAPROXY || exit 0

if [ -e /etc/default/haproxy ]; then
	. /etc/default/haproxy
fi

test -f "$CONFIG" || exit 0

[ -f /etc/default/rcS ] && . /etc/default/rcS
. /lib/lsb/init-functions


check_haproxy_config()
{
	$HAPROXY -c -f "$CONFIG" >/dev/null
	if [ $? -eq 1 ]; then
		log_end_msg 1
		exit 1
	fi
}

haproxy_start()
{
	[ -d "$RUNDIR" ] || mkdir "$RUNDIR"
	chown haproxy:haproxy "$RUNDIR"
	chmod 2775 "$RUNDIR"

	check_haproxy_config

    start-stop-daemon --quiet --oknodo --start --pidfile "$PIDFILE" \
		--exec $HAPROXY -- -f "$CONFIG" -D -p "$PIDFILE" \
		$EXTRAOPTS || return 2
	return 0
}

haproxy_stop()
{
	if [ ! -f $PIDFILE ] ; then
		# This is a success according to LSB
		return 0
	fi

	ret=0
	for pid in $(cat $PIDFILE); do
		if kill -0 $pid 2> /dev/null; then
			/bin/kill $pid || ret=4
		fi
	done

	[ $ret -eq 0 ] && rm -f $PIDFILE

	return $ret
}

haproxy_reload()
{
	check_haproxy_config

	$HAPROXY -f "$CONFIG" -p $PIDFILE -D $EXTRAOPTS -sf $(cat $PIDFILE) \
		|| return 2
	return 0
}

haproxy_status()
{
	if [ ! -f $PIDFILE ] ; then
		# program not running
		return 3
	fi

	for pid in $(cat $PIDFILE) ; do
		if ! ps --no-headers p "$pid" | grep haproxy > /dev/null ; then
			# program running, bogus pidfile
			return 1
		fi
	done

	return 0
}


case "$1" in
start)
	log_daemon_msg "Starting haproxy" "haproxy"
	haproxy_start
	ret=$?
	case "$ret" in
	0)
		log_end_msg 0
		;;
	1)
		log_end_msg 1
		echo "pid file '$PIDFILE' found, haproxy not started."
		;;
	2)
		log_end_msg 1
		;;
	esac
	exit $ret
	;;
stop)
	log_daemon_msg "Stopping haproxy" "haproxy"
	haproxy_stop
	ret=$?
	case "$ret" in
	0|1)
		log_end_msg 0
		;;
	2)
		log_end_msg 1
		;;
	esac
	exit $ret
	;;
reload|force-reload)
	log_daemon_msg "Reloading haproxy" "haproxy"
	haproxy_reload
	ret=$?
	case "$ret" in
	0|1)
		log_end_msg 0
		;;
	2)
		log_end_msg 1
		;;
	esac
	exit $ret
	;;
restart)
	log_daemon_msg "Restarting haproxy" "haproxy"
	haproxy_stop
	haproxy_start
	ret=$?
	case "$ret" in
	0)
		log_end_msg 0
		;;
	1)
		log_end_msg 1
		;;
	2)
		log_end_msg 1
		;;
	esac
	exit $ret
	;;
status)
	haproxy_status
	ret=$?
	case "$ret" in
	0)
		echo "haproxy is running."
		;;
	1)
		echo "haproxy dead, but $PIDFILE exists."
		;;
	*)
		echo "haproxy not running."
		;;
	esac
	exit $ret
	;;
*)
	echo "Usage: /etc/init.d/haproxy {start|stop|reload|restart|status}"
	exit 2
	;;
esac
'''


#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch

#принимает на вход команду. В случае успеха ничего не выводит. В случае ошибки вызывает exception.
def runShellCommand(command):
    logger.info("run command {0}".format(command))
    fileno_logfile = logch.stream.fileno()
    p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
    while True:
        rcode1 = p1.poll()
        if rcode1 is not None and rcode1 != 0:
            raise ValueError("error run command {0} code {1}".format(command, rcode1))
        if rcode1 == 0:
            break
    p1.wait()
    return 0

def start_haproxy(group):
    try:
        initd = INITD.format(group)
        command = "{0} start".format(initd)
        script = command.split(' ')[0]
        if not os.path.exists(script):
            raise ValueError("script {0} not found".format(script))
        return runShellCommand(command)
    except Exception as err:
        logger.critical("error start haproxy: {0}".format(err))
    return 2

def stop_haproxy(group):
    try:
        initd = INITD.format(group)
        command = "{0} stop".format(initd)
        script = command.split(' ')[0]
        if not os.path.exists(script):
            raise ValueError("script {0} not found".format(script))
        return runShellCommand(command)
    except Exception as err:
        logger.critical("error stop haproxy: {0}".format(err))
    return 2

def get_groups(data):
    groups = [ i.get('group') for i in data ]
    return groups

def all_instances_by_group(data):
    instances = list(set([ i.get('instance') for i in data ]))
    return instances

def all_ports(data):
    ports = list([ (i.get('port'), i.get('instance')) for i in data ])
    return dict(ports)

def host_by_instance(data, instance, group):
    hosts = [ i for i in data if i.get('instance') == instance and i.get('group') == group ]
    return hosts

def read_json(jfile):
    if not os.path.exists(jfile):
        raise ValueError("не найден файл с конфигурацией json: {0}".format(jfile))
    with open(jfile) as fd:
        position_mysqls = json.load(fd)
    return position_mysqls

def generate_haproxy_initd(group):
    initd = INITD.format(group)
    if os.path.exists(initd):
        return
    data = TEMPLATE_INITD.replace("FIXNAME", sys.argv[0])
    data = data.replace("FIXGROUP", group)
    if len(initd) == 0:
        raise ValueError("[generate_haproxy_initd] empty init.d script")
    with open(initd, 'w') as fd:
        fd.write(data)
    os.chmod(initd, 0755)
    return

def generate_haproxy_startup(group):
    proc = Popen("/usr/sbin/update-rc.d haproxy-mysql-{0} defaults 60 20".format(group), shell=True)
    proc.communicate()
    return proc.returncode

def generate_haproxy_config(jfile):
    listen_body = list()
    position_mysqls = read_json(jfile)
    if len(position_mysqls) == 0:
        raise ValueError("пустой json: {0}".format(position_mysqls))
    group = get_groups(position_mysqls)[0]
    group_name = group.split('@')[0]
    config_path = CONFIG.format(group_name)
    for instance in all_instances_by_group(position_mysqls):
        for num, host in enumerate(host_by_instance(position_mysqls, instance, group)):
            hostname = host.get('host')
            if hostname.find(socket.getfqdn()) > -1: continue
            port = host.get('port')
            instance = host.get('instance')
            part = TEMPLATE_BODY.format(instance, port, hostname, num+1)
            listen_body.append(part)
    head_body = TEMPLATE_HEAD.format(sys.argv[0], group)
    config_values = '{0}\n{1}'.format(head_body, '\n'.join(listen_body))
    if os.path.exists(config_path):
        os.rename(config_path, "{0}.old".format(config_path))
    if not os.path.exists(os.path.dirname(CONFIG)):
        os.mkdir(os.path.dirname(CONFIG))
    with open(config_path, 'w') as fd:
        fd.write(config_values)
    return group_name

def check_haproxy_ports(jfile):
    bad_connects = list()
    position_mysqls = read_json(jfile)
    ports = all_ports(position_mysqls)
    for port in ports:
        try:
            sock = socket.socket(socket.AF_INET6)
            sock.settimeout(5)
            addr = socket.getaddrinfo('localhost', int(port), socket.AF_INET6, 0, socket.SOL_IP)
            sock.connect(addr[0][4])
        except Exception as err:
            logger.critical('error connect {1}: {0}'.format(err, addr[0][4]))
            bad_connects.append('{0}:{1}'.format(ports[port], port)) #instance:port
            if opts.debug: raise
        finally:
            sock.close()
    if len(bad_connects) > 0:
        raise ValueError("невозможно подключиться к базам: {0}".format(','.join(bad_connects)))
    return

def run():
    if opts.doit:
        group_name = generate_haproxy_config(opts.jfile)
        if len(group_name) == 0:
            raise ValueError("empty group name for {0}".format(opts.jfile))
        if opts.generate_initd:
            generate_haproxy_initd(group_name)
            generate_haproxy_startup(group_name)
        stop_haproxy(group_name) #тут надо ловить код. но пока не важно
        start_haproxy(group_name) #тоже самое
        check_haproxy_ports(opts.jfile) #проверяем работу haproxy с новым конфигом
    return

if __name__ == '__main__':
    parser = argparse.ArgumentParser(usage=USAGE)
    parser.add_argument("--file", type=str, action='store', dest="jfile",
        help="json файл, содержащий положение баз mysql")
    parser.add_argument("--doit", action='store_true', dest="doit",
        help="выполнить установку или удаление пакетов yandex-direct-mysql")
    parser.add_argument("--group", type=str, action='store', dest="group",
        help="имя группы машин для имени конфига. Например: dev7")
    parser.add_argument("-d", "--debug", action='store_true',
        help="включить отладочный режим")
    parser.add_argument("--generate-initd", action='store_true', dest="generate_initd",
        help="сгенерировать init.d скрипт и добавить его в автозагрузку")
    opts = parser.parse_args()

    level = 'DEBUG' if opts.debug else 'INFO'
    logger, logch = startLogging(level)

    if opts.jfile is None or len(opts.jfile) == 0:
        raise ValueError("пустая переменная json data")

    run()
