#!/usr/bin/env python
# *-* encoding: utf-8 *-*

import apt
import apt_pkg
import argparse
import sys
import os
import psutil
import logging

from subprocess import Popen

PKGPPCDATA = "yandex-direct-mysql"
PKGPPCDICT = "yandex-du-ppcdict-mysql-test"

USAGE = '''Скрипт для проверки и установки пакетов mysqld на сервер. 

Пример:

%(prog)s [instance1] [instance2]
'''

MYCONFD = {
    "buffer-pool.cnf": "#generated {script_name}\n[mysqld]\ninnodb_buffer_pool_size = {buffer_size}\n",
    "lock_wait_timeout.cnf": "#generated {script_name}\n[mysqld]\nlock_wait_timeout = 15\n"
    }

REMOVE_MYSQL_CONFIGS = [ "/etc/mysql/{0}.conf.d/audit.cnf" ]
MYCONFPATH = "/etc/mysql/{0}.conf.d"

#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch

#принимает на вход ссылку на apt-cache и список инстансов, на которые нужно поставить пакеты.
#На выходе получается словать со списком пакетов, которые нужно поставить и удалить. 
# { "installing": ['yandex-mysql-ppcdata1', 'yandex-mysql-ppcdata2'], "removing": ['yandex-mysql-ppcdata3'] }
def analyze_packages(cache, instances):
    installed = [ pack.name for pack in cache.packages if pack.current_state == apt_pkg.CURSTATE_INSTALLED and ( pack.name.startswith(PKGPPCDATA) or pack.name.startswith(PKGPPCDICT) ) ]
    generate_name = lambda i: "{0}-{1}".format(PKGPPCDATA, i) if i.find('ppcdict') == -1 else PKGPPCDICT
    installing = [ generate_name(i) for i in instances ]
    need_installing = list(set(installing)-set(installed))
    need_removing = list(set(installed)-set(installing))
    return { "installing": need_installing, "removing": need_removing }


#принимает словарь с пакетами, которые нужно поставить и удалить. Не выводит ничего. 
#При ошибке вызывает exception.
def execute_packages(packages):
    if len(packages.get("installing")) == 0 and len(packages.get("removing")) == 0:
        return
    cache = apt.cache.Cache()
    cache.update()
    cache.open()

    for package in packages.get("installing", []):
        name = package.split('=', 1)
        pkg = cache[name[0]]
        if len(name) > 1:
            pkg.candidate = pkg.versions.get(name[1])
        if pkg.is_installed:
            print "{0} already installed".format(package)
        else:
            pkg.mark_install()

    for package in packages.get("removing", []):
        name = package.split('=', 1)
        pkg = cache[name[0]]
        if not pkg.is_installed:
            print "{0} already removed".format(name)
        else:
            pkg.mark_delete(True, purge=True)

    try:
        cache.commit()
    except Exception as err:
        raise ValueError("packages installation failed: {0}".format(err))
    return

#def clean_mysql_confd(instances):
#    for instance in instances:
#        mysql_dir = MYCONFPATH.format(instance)
#        shutil.rmtree(mysql_dir, ignore_errors=True)
#    return

#переводит байты в человекочитаемый вид. Нужно для генерации buffer-pool.
#На входе число, на выходе строка.
def humanize_bytes(value):
    const = 1024
    if value < const:
        return "{0}{1}".format(value, "B")
    div, exp = const, 0
    n = value/const
    while n > const:
        div *= const
        exp += 1
        n /= const
    return "{0}{1}".format( int(n), "KMGTPE"[exp])

#на основе устанавливаемых инстансов, генерирует конфиги в <instance>.conf.d
#Например на основании доступной памяти и количества инстансов высчитывается buffer_pool. 
#На вход принимает список инстансов. Важно передавать список, т.к. надо считать память для mysqld.
def generate_mysql_confd(instances):
    svmem = psutil.virtual_memory()
    reserv_mem = float(svmem.total) * 0.3
    buffer_size = (float(svmem.total) - reserv_mem) / float(len(instances))
    variables = { "script_name": ' '.join(sys.argv),
                "buffer_size": humanize_bytes(buffer_size)
                }
    for instance in instances:
        mysql_dir = MYCONFPATH.format(instance)
        if not os.path.exists(mysql_dir):
            os.makedirs(mysql_dir)
        for f1le, value in MYCONFD.items():
            with open(os.path.join(mysql_dir, f1le), 'w') as fd:
                fd.write(value.format(**variables))
            logger.info("generate {0} with values: {1}".format(f1le, value.format(**variables)))
    return

#принимает на вход команду. В случае успеха ничего не выводит. В случае ошибки вызывает exception.
def runShellCommand(command):
    logger.info("run command {0}".format(command))
    fileno_logfile = logch.stream.fileno()
    p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
    while True:
        rcode1 = p1.poll()
        if rcode1 is not None and rcode1 != 0:
            raise ValueError("error run command {0} code {1}".format(command, rcode1))
        if rcode1 == 0:
            break
    p1.wait()
    return

#готовит chroot для mysqld:ppcdict
def prepare_ppcdict_directory(cache):
    if os.path.exists('/opt/root.ppcdict.1'): return
    pkg = cache['yandex-direct-mysql-ppcdict-pxc']
    versions = pkg.version_list
    runShellCommand("/usr/local/bin/ppcdict-create-jails {0} {1} {2}".format(versions[0].ver_str, node_number, group))
    return


#удаляем лишние настройки, которые могут приехать с продакшен пакетами.
#например audit.log
def remove_mysql_configs(instances):
    for instance in instances:
        for temp in REMOVE_MYSQL_CONFIGS:
            f1le = temp.format(instance)
            if os.path.exists(f1le):
                logger.info("удаляем {0} для бд {1}".format(f1le, instance))
                os.remove(f1le)
    return

def run():
    apt_pkg.init_config()
    apt_pkg.init_system()
    cache = apt_pkg.Cache() #читаем кэш

    packages = analyze_packages(cache, opts.instances) #создаем список пакетов для установки и удаления
    logger.info("start command {0} {1}".format(sys.argv[0], ' '.join(opts.instances)))
    logger.info("Installing: {0}, Removing: {1}".format(packages["installing"], packages["removing"]))

    if not opts.doit: return #если не указан флаг, не делаем больше ничего
    execute_packages({'installing': ['yandex-du-mysql-monitor=4.3-1']}) #чтобы не писать в мониторинг в продакшеновый zookeeper
    execute_packages(packages) #устанавливаем и удаляем пакеты
    generate_mysql_confd(opts.instances) #генерируем mysqld.conf.d конфиги

    remove_mysql_configs(opts.instances)
    if 'ppcdict' in opts.instances: #создаем chroot для ppcdict, если его нет еще
        prepare_ppcdict_directory(cache)
    return

if __name__ == '__main__':
    try:
        parser = argparse.ArgumentParser(usage=USAGE)
        parser.add_argument("instances", nargs='*', type=str, action='store',
            help="список инстансов для установки пакетов")
        parser.add_argument("--doit", action='store_true', dest="doit",
            help="выполнить установку или удаление пакетов yandex-direct-mysql")
        parser.add_argument("-g", "--group", action="store", dest="group",
            help="имя группы машин(например dev7, devtest)", default="")
        parser.add_argument("-n", "--node-number", action="store", dest="nodes",
            help="количество нод под ppcdict машины(по умолчанию 1)", default=1)
        parser.add_argument("-d", "--debug", action='store_true',
            help="включить отладочный режим")
        opts = parser.parse_args()

        level = 'DEBUG' if opts.debug else 'INFO'
        group = opts.group
        node_number = opts.nodes
        logger, logch = startLogging(level)

        if len(opts.instances) == 0:
            raise ValueError("не указаны инстансы для установки пакетов")
        
        run()
    except ValueError as err:
        logger.critical("{0}".format(err))
        sys.exit(2)
