#!/usr/bin/env python
# *-* encoding: utf-8 *-*

import argparse
import os
import json
import socket
import logging

from subprocess import Popen

USAGE = '''Выставляет настройки репликации для mysqld. Данные берутся из /tmp/replica.json.

Например:
%(prog)s --doit --file /tmp/replica.json
'''

CMD = '''/usr/local/bin/lm {0} change-master '{1}:{2}:auto:auto:{3}:{4}' '''
REPLICATE_USER = "rplcat"
REPLICATE_PASSWD = "/etc/direct-tokens/mysql_rplcat_test"

#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch

#принимает на вход команду. В случае успеха ничего не выводит. В случае ошибки вызывает exception.
def runShellCommand(command):
    logger.info("run command {0}".format(command))
    fileno_logfile = logch.stream.fileno()
    p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
    while True:
        rcode1 = p1.poll()
        if rcode1 is not None and rcode1 != 0:
            raise ValueError("error run command '{0}' code {1}".format(command, rcode1))
        if rcode1 == 0:
            break
    p1.wait()
    return 0


#вычитывает пароль rplcat из файла
def get_replicate_password():
    if not os.path.exists(REPLICATE_PASSWD):
        raise ValueError("not found file {0}".format(REPLICATE_PASSWD))
    with open(REPLICATE_PASSWD, "r") as fd:
        return fd.read().strip()

'''
JSON структуру преобразуем в команды lm для запуска репликации:
[
 {"host":"ppclogshutter01f.ppc.yandex.ru","instance":"ppclog","port":3309,"group-name":"dev7","replica-name":"rep1"},
 {"host":"ppclogshutter01f.ppc.yandex.ru","instance":"ppcmonitor","port":3313,"group-name":"dev7","replica-name":"rep2"},
 {"host":"ppclogpusher-storage02f.yandex.ru","instance":"ppclog","port":3309,"group-name":"dev7","replica-name":"rep2"}
]
'''
def generate_replication_commands(jfile):
    commands = list()
    if not os.path.exists(jfile):
        raise ValueError("не найден файл с конфигурацией json: {0}".format(jfile))
    with open(jfile) as fd:
        cluster_mysqls = json.load(fd)
    if len(cluster_mysqls) == 0:
        raise ValueError("список mysql пуст: {0}".format(jfile))
    hostname = socket.getfqdn()
    local_mysqls = [ i for i in cluster_mysqls if i.get("host", "").startswith(hostname) ]
    if len(local_mysqls) == 0:
        raise ValueError("не найден сервер {0} в конфигурации {1}".format(hostname, cluster_mysqls))
    groupname = local_mysqls[0].get("group-name", "")
    replname = local_mysqls[0].get("replica-name", "")
    for lms in local_mysqls:
        instance = lms.get("instance")
        if instance.startswith('ppcdict') or instance.startswith('sandbox'):
            continue
        remoute_mysql = [ i for i in cluster_mysqls if i.get("group-name", "").startswith(groupname) and
                i.get("instance", "") == instance and not i.get("replica-name", "").startswith(replname) ]
        if len(remoute_mysql) == 0:
            raise ValueError("не найдена парная машина для инстанса {0} группы {1}".format(instance, groupname))
        cmd = CMD.format(instance, remoute_mysql[0].get("host"), remoute_mysql[0].get("port"), REPLICATE_USER, get_replicate_password())
        commands.append(cmd)
    return commands

def run():
    commands = generate_replication_commands(opts.jfile)
    logger.debug("command list {0}".format(commands))
    if opts.doit:
        for command in commands:
            runShellCommand(command)
    return

if __name__ == '__main__':
    parser = argparse.ArgumentParser(usage=USAGE)
    parser.add_argument("--file", type=str, action='store', dest="jfile",
        help="json файл, содержащий положение баз mysql")
    parser.add_argument("--doit", action='store_true', dest="doit",
        help="выполнить установку или удаление пакетов yandex-direct-mysql")
    parser.add_argument("-d", "--debug", action='store_true',
        help="включить отладочный режим")
    opts = parser.parse_args()

    level = 'DEBUG' if opts.debug else 'INFO'
    logger, logch = startLogging(level)

    if opts.jfile is None or len(opts.jfile) == 0:
        raise ValueError("пустая переменная json data")

    run()
