#!/usr/bin/env python
# *-* encoding: utf-8 *-*

import apt
import apt_pkg
import argparse
import sys
import os
import psutil
import shutil
import signal
import time
import logging

from subprocess import Popen

INSTANCEDIR = "/opt"
INSTANCEBASE = "/opt/{0}"
ROOT_PPCDICT = "/opt/root.ppcdict.1"
CHROOT_PPCDICT = "{0}/opt".format(ROOT_PPCDICT)
MYSQL_SOCKET = "/var/run/mysqld.{0}/mysqld.sock"
MYSQL_PIDFILE = "/var/run/mysqld.{0}/mysqld.pid"
MYSQL_PPCDICT_PIDFILE = "/run/mysqld.{0}/mysqld.pid"
MYSQL_PPCDICT_SOCKET = "/run/mysqld.{0}/mysqld.pid"
HAPROXYCNF = "/etc/haproxy"
HAPROXY_PIDFILE = "/var/run/ppctest-mysql-{0}.pid"

USAGE = '''Скрипт для остановки старых баз и включения новых. Проверяет

Пример:

%(prog)s [instance1] [instance2]
'''


#логирование осуществляется в STDOUT/STDERR
def startLogging(level='DEBUG'):
    logger = logging.getLogger('stream logs to console')
    logger.setLevel(level=getattr(logging, level))
    formatter = logging.Formatter('%(asctime)s %(message)s')
    logch = logging.StreamHandler()
    logch.setLevel(level=getattr(logging, level))
    logch.setFormatter(formatter)
    logger.addHandler(logch)
    return logger, logch


#принимает на вход команду. В случае успеха ничего не выводит. В случае ошибки вызывает exception.
def runShellCommand(command):
    logger.info("run command {0}".format(command))
    fileno_logfile = logch.stream.fileno()
    p1 = Popen(command, shell=True, bufsize=0, stderr=fileno_logfile, stdout=fileno_logfile)
    while True:
        rcode1 = p1.poll()
        if rcode1 is not None and rcode1 != 0:
            raise ValueError("error run command {0} code {1}".format(command, rcode1))
        if rcode1 == 0:
            break
    p1.wait()
    return 0

def get_instance(path):
    db = path.split('.')
    return db[1] if len(db) > 1 else None

def get_db_type(path):
    db = path.split('.')
    return db[2] if len(db) > 2 else None

#Принимает список инстансов, которые должны быть задействованы и готовит соответствующие дирректории под них. Ничего не выводит.
#Текущие дирректории перемещаются в old при условии отсутсвия своего инстанса в переменной instances, либо при наличии в instances и присутствия новых дирректорий .new
#Для ppcdict делается дополнительный symlink(например для работы с get_resources.py).
def move_databases(instances):
    all_databases = [ i for i in os.listdir(INSTANCEDIR) if i.startswith('mysql.') ]
    new_databases = [ i for i in all_databases if i.endswith('.new') and len(i.split('.')) > 2 ]
    old_databases = [ i for i in all_databases if i.endswith('.old') and len(i.split('.')) > 2 ]
    current_databases = [ i for i in all_databases if len(i.split('.')) == 2 ]

    cur_path_template = lambda x: os.path.join(INSTANCEDIR, "mysql.{0}".format(x)) if x.find('ppcdict') == -1 else os.path.join(CHROOT_PPCDICT, "mysql.{0}".format(x))
    old_path_template = lambda x: os.path.join(INSTANCEDIR, "mysql.{0}.old".format(x))
    new_path_template = lambda x: os.path.join(INSTANCEDIR, "mysql.{0}.new".format(x))
    parse_instances = lambda y: list(set(map(lambda x: x.replace('mysql.', '').replace('.old', '').replace('.new', ''), y)))

    #2. Перемещаем в old все базы, которых нет в instances, или они указаны в instance и имеют готовые new копии
    for dbname in parse_instances(current_databases):
        if (dbname not in instances) or ((dbname in instances) and (dbname in parse_instances(new_databases))):
            logger.info("текущая БД {0} больше не используется и будет перемещена в {1}".format(cur_path_template(dbname), old_path_template(dbname)))
            if os.path.exists(old_path_template(dbname)):
                logger.info("найден старый бекап {0}, который будет удален".format(old_path_template(dbname)))
                if opts.doit:
                    shutil.rmtree(old_path_template(dbname))
            if os.path.exists(cur_path_template(dbname)):
                if opts.doit:
                    os.rename(cur_path_template(dbname), old_path_template(dbname))
                    logger.info("данные перемещены из {0} в {1}".format(cur_path_template(dbname), old_path_template(dbname)))

    #3. Перемещаем в рабочии копии все базы из new, которые указаны в instances. Если в new БД нет, вероятнее всего она была уже перемещена.
    for dbname in parse_instances(new_databases):
        if dbname in instances:
            logger.info("новая БД {0} будет перемещена в {1}".format(new_path_template(dbname), cur_path_template(dbname)))
            if opts.doit:
                if os.path.exists(cur_path_template(dbname)): shutil.rmtree(cur_path_template(dbname))
                os.rename(new_path_template(dbname), cur_path_template(dbname))
            logger.info("новые данные перемещены из {0} в {1}".format(new_path_template(dbname), cur_path_template(dbname)))

    #4. symlink для ppcdict. Т.к. БД монтируется в сторонюю дирректорию, мутил hardlink на нее.
    dbname = 'ppcdict'
    prod_ppcdict_path = os.path.join(INSTANCEDIR, "mysql.{0}".format(dbname))
    if os.path.islink(prod_ppcdict_path) and 'ppcdict' not in instances:
        logger.info("бд ppcdict больше не используется, удаляем symlink {0}".format(prod_ppcdict_path))
        if opts.doit: os.unlink(prod_ppcdict_path)
    elif not os.path.islink(prod_ppcdict_path) and 'ppcdict' in instances:
        logger.info("бд ppcdict теперь используется, добавляем symlink {0} --> {1}".format(prod_ppcdict_path, cur_path_template(dbname)))
        if opts.doit: os.symlink(cur_path_template(dbname), prod_ppcdict_path)

def start_databases(instances):
    for instance in instances:
        trying = 0
        command = "/etc/init.d/mysql.{0} start".format(instance)
        while True:
            try:
                code = runShellCommand(command)
                if code == 0: break
            except Exception as err:
                if trying >= 3: raise
            finally:
                trying += 1
                time.sleep(10)
    return

def get_pid(pidfile):
    if not os.path.exists(pidfile):
        logger.warning("потерян pidfile y {0}".format(pidfile))
        return None
    strpid = open(pidfile).read()
    if len(strpid) == 0:
        logger.warning("пустой pidfile y {0}".format(pidfile))
        return None
    return int(strpid)


#Отправляет TERM/KILL сигнал указанным pid'ам процессов.
def killMysqldProcess(pids, kill9=False):
    try:
        sig = signal.SIGKILL if kill9 else signal.SIGTERM
        [os.kill(int(pid), sig) for pid in pids]
    except Exception as err:
        logger.warning(err)
    return

#Принимает списки pid'ов и возвращает еще работающие.
def checkRunningProc(checking_pids):
    running_pids = list()
    for pid in checking_pids:
        if os.path.exists('/proc/{0}'.format(pid)):
            logger.info("found running process from {0}".format(pid))
            running_pids.append(pid)
        else:
            logger.info("not found running process from {0}".format(pid))
    return running_pids

def stop_all_databases():
    path = '/var/run'
    if os.path.exists(path):
        instances = [ name.split('.')[1] for name in os.listdir(path) if name.startswith('mysql') and len(name.split('.')) > 1 ]
        stop_databases(instances)
    path = os.path.join(ROOT_PPCDICT, 'run')
    if os.path.exists(path):
        instances = [ name.split('.')[1] for name in os.listdir(path) if name.startswith('mysql') and len(name.split('.')) > 1 ]
        stop_databases(instances)
    return

#на основе сокета, находим pid запущенных mysql и останавливаем. 10 попыток TERM, дальше KILL
def stop_databases(instances):
    pids = list()
    for instance in instances:
        socket = MYSQL_SOCKET.format(instance) if instance.find('ppcdict') == -1 else os.path.join(ROOT_PPCDICT, MYSQL_PPCDICT_SOCKET.format(instance).lstrip('/'))
        pidfile = MYSQL_PIDFILE.format(instance) if instance.find('ppcdict') == -1 else os.path.join(ROOT_PPCDICT, MYSQL_PPCDICT_PIDFILE.format(instance).lstrip('/'))
        pids.append(get_pid(pidfile))
        #command = "mysqladmin --socket={0} shutdown".format(socket)
        logger.info('stop database {0} pidfile {1}'.format(instance, pidfile))
    default_mysql_pid = '/var/run/mysqld/mysqld.pid'
    default_mysql_socket = '/var/run/mysqld/mysqld.sock'
    if os.path.exists(default_mysql_pid):
        pids.append(get_pid(default_mysql_pid))
        logger.info('stop database default pidfile {0}'.format(default_mysql_pid))
        #command = "mysqladmin --socket={0} shutdown".format(default_mysql_socket)

    logger.info("проверка останавливаемых mysqld: {0}".format(pids))
    retry = 0
    while True:
        pids = checkRunningProc(pids)
        if len(pids) == 0: break
        if retry < 10:
            retry += 1
            killMysqldProcess(pids)
        else:
            killMysqldProcess(pids, True)
        time.sleep(5)
    return

#haproxy может использовать какие-то порты, которые в будущей выкладке баз нужно освободить. Останавливаем его.
def stop_haproxy(group):
    try:
        runShellCommand("/etc/init.d/haproxy-mysql-{0} stop".format(group))
    except Exception as err:
        logger.critical("error stop haproxy: {0}".format(err))
    pid = get_pid(HAPROXY_PIDFILE.format(group))
    pids = checkRunningProc([pid])
    if len(pids) == 0: return
    killMysqldProcess(pids, True)
    return

def run():
    if opts.doit: stop_all_databases()
    if opts.doit: stop_haproxy(opts.group)
    move_databases(opts.instances)
    if opts.doit: start_databases(opts.instances)
    return

if __name__ == '__main__':
    try:
        parser = argparse.ArgumentParser(usage=USAGE)
        parser.add_argument("instances", nargs='*', type=str, action='store',
            help="список инстансов для установки пакетов")
        parser.add_argument("--doit", action='store_true', dest="doit",
            help="выполнить установку или удаление пакетов yandex-direct-mysql")
        parser.add_argument("--group", action='store', dest="group",
            help="имя группы машин. Используется для остановки haproxy")
        parser.add_argument("-d", "--debug", action='store_true',
            help="включить отладочный режим")
        opts = parser.parse_args()

        level = 'DEBUG' if opts.debug else 'INFO'
        logger, logch = startLogging(level)

        if len(opts.instances) == 0:
            raise ValueError("не указаны инстансы для установки пакетов")

        run()
    except ValueError as err:
        logger.critical("{0}".format(err))
        raise
        sys.exit(2)
